//--------------------------------------------------------------------------------------
// File: Tutorial06.cpp
//
// Texturing
//
//--------------------------------------------------------------------------------------
#include <windows.h>
#include <d3d10.h>
#include <d3dx10.h>
#include <stdio.h>
#include <assert.h>

//--------------------------------------------------------------------------------------
// Global Variables
//--------------------------------------------------------------------------------------
HINSTANCE               g_hInst = NULL;
HWND                    g_hWnd = NULL;
// from Tutorial 02
D3D10_DRIVER_TYPE       g_driverType = D3D10_DRIVER_TYPE_NULL;
ID3D10Device*           g_pd3dDevice = NULL;
IDXGISwapChain*         g_pSwapChain = NULL;
ID3D10RenderTargetView* g_pRenderTargetView = NULL;
// from Tutorial 03
ID3D10Effect*           g_pEffect = NULL;
ID3D10EffectTechnique*  g_pTechnique = NULL;
ID3D10InputLayout*      g_pVertexLayout = NULL;
ID3D10Buffer*           g_pVertexBuffer = NULL;
// from Tutorial 04
ID3D10Texture2D*            g_pDepthStencil = NULL;
ID3D10DepthStencilView*     g_pDepthStencilView = NULL;
ID3D10Buffer*               g_pIndexBuffer = NULL;
ID3D10EffectMatrixVariable* g_pWorldVariable = NULL;
ID3D10EffectMatrixVariable* g_pViewVariable = NULL;
ID3D10EffectMatrixVariable* g_pProjectionVariable = NULL;
D3DXMATRIX                  g_World;
D3DXMATRIX                  g_View;
D3DXMATRIX                  g_Projection;
// new in Tutorial 05
ID3D10EffectVectorVariable* g_pLightDirVariable = NULL;
ID3D10EffectVectorVariable* g_pLightColorVariable = NULL;
// new in Tutorial 06
ID3D10Resource*                     g_pTexture = NULL;
ID3D10ShaderResourceView*           g_pTextureRV = NULL;
ID3D10EffectShaderResourceVariable* g_pDecalVariable = NULL;

#define SIZE_OF_ONE_VERTEX (32)
#define SIZE_OF_ONE_ORIG_VERTEX (124)
#define NUM_VERTICES (63488)
#define NUM_INDICES (91932)

//--------------------------------------------------------------------------------------
// Forward declarations
//--------------------------------------------------------------------------------------
HRESULT InitWindow( HINSTANCE hInstance, int nCmdShow );
HRESULT InitDevice();
void CleanupDevice();
LRESULT CALLBACK    WndProc( HWND, UINT, WPARAM, LPARAM );
void Render();


//--------------------------------------------------------------------------------------
// Entry point to the program. Initializes everything and goes into a message processing 
// loop. Idle time is used to render the scene.
//--------------------------------------------------------------------------------------
int WINAPI wWinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance, LPWSTR lpCmdLine, int nCmdShow )
{

	if( FAILED( InitWindow( hInstance, nCmdShow ) ) )
		return 0;

	if( FAILED( InitDevice() ) )
	{
		CleanupDevice();
		return 0;
	}

	// Main message loop

	// We change the message loop by replacing GetMessage() with PeekMessage(). In contrast to
	// GetMessage(), PeekMessage returns immediately even if the message loop is empty
	MSG msg = {0};
	while( WM_QUIT != msg.message )
	{
		if( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE ) )
		{
			TranslateMessage( &msg );
			DispatchMessage( &msg );
		}
		else
		{
			// if there are no events pending, we render the scene
			Render();
		}
	}

	CleanupDevice();

	return ( int )msg.wParam;
}


//--------------------------------------------------------------------------------------
// Register class and create window
//--------------------------------------------------------------------------------------
HRESULT InitWindow( HINSTANCE hInstance, int nCmdShow )
{
	// Register class
	WNDCLASSEX wcex;

	// clear out the window class for use
	ZeroMemory(&wcex, sizeof(WNDCLASSEX)); // never forget to zero out, if you do not set all the values!

	// fill the window class struct
	wcex.cbSize			= sizeof( WNDCLASSEX );
	wcex.style			= CS_HREDRAW | CS_VREDRAW; // Redraw when width or height changes
	wcex.lpfnWndProc	= WndProc; // function pointer 
	wcex.hInstance		= hInstance; // Instance handle: important to find window class (CreateWindow)
	wcex.hCursor		= LoadCursor( NULL, IDC_ARROW ); // shape of the cursor
	wcex.hbrBackground	= ( HBRUSH )( COLOR_WINDOW );    // background color
	wcex.lpszClassName	= L"TutorialWindowClass"; // Class name: important to find window class (CreateWindow)

	if( !RegisterClassEx( &wcex ) )
		return E_FAIL;

	// Create window
	g_hInst = hInstance;
	RECT rc = { 0, 0, 640, 480 };
	AdjustWindowRect( &rc, WS_OVERLAPPEDWINDOW, FALSE ); // adjust to match client rect
	g_hWnd = CreateWindow( L"TutorialWindowClass", L"Direct3D 10 Tutorial 0: Setting Up Window", WS_OVERLAPPEDWINDOW,
		CW_USEDEFAULT, CW_USEDEFAULT, rc.right - rc.left, rc.bottom - rc.top, NULL, NULL, hInstance,
		NULL );
	if( !g_hWnd )
		return E_FAIL;

	ShowWindow( g_hWnd, nCmdShow );

	return S_OK;
}

//--------------------------------------------------------------------------------------
// Called every time the application receives a message
//--------------------------------------------------------------------------------------
LRESULT CALLBACK WndProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
	PAINTSTRUCT ps;
	HDC hdc;

	switch( message )
	{
	case WM_PAINT:
		hdc = BeginPaint( hWnd, &ps );
		EndPaint( hWnd, &ps );
		break;

	case WM_DESTROY:
		PostQuitMessage( 0 );
		break;

	default:
		return DefWindowProc( hWnd, message, wParam, lParam );
	}

	return 0;
}


//--------------------------------------------------------------------------------------
// Create Direct3D device and swap chain
//--------------------------------------------------------------------------------------
HRESULT InitDevice()
{
	HRESULT hr = S_OK;

	// get the size of the window's client rect
	RECT rc;
	GetClientRect( g_hWnd, &rc );
	UINT width = rc.right - rc.left;
	UINT height = rc.bottom - rc.top;


	UINT createDeviceFlags = 0;
#ifdef _DEBUG
	createDeviceFlags |= D3D10_CREATE_DEVICE_DEBUG;
#endif


	// find an type of dx10 driver supported by the hardware
	D3D10_DRIVER_TYPE driverTypes[] =
	{
		D3D10_DRIVER_TYPE_HARDWARE,
		D3D10_DRIVER_TYPE_REFERENCE,
	};
	UINT numDriverTypes = sizeof( driverTypes ) / sizeof( driverTypes[0] );

	// fill the swap chain descriptor
	DXGI_SWAP_CHAIN_DESC sd; 

	ZeroMemory( &sd, sizeof( sd ) );
	// number of buffers
	sd.BufferCount       = 1;

	// buffer descriptor
	sd.BufferDesc.Width  = width; 
	sd.BufferDesc.Height = height;
	sd.BufferDesc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
	sd.BufferDesc.RefreshRate.Numerator = 60;
	sd.BufferDesc.RefreshRate.Denominator = 1;

	// buffer usage: use as a render target (might as well be used as a texture)
	sd.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;

	// handle to output window that will show the front buffer
	sd.OutputWindow = g_hWnd;

	// multisampling support (turned off for now)
	sd.SampleDesc.Count = 1;
	sd.SampleDesc.Quality = 0;

	// windowed or fullscreen
	sd.Windowed = TRUE;

	// try to create a swap chain for a specific driver type
	for( UINT driverTypeIndex = 0; driverTypeIndex < numDriverTypes; driverTypeIndex++ )
	{
		g_driverType = driverTypes[driverTypeIndex];
		hr = D3D10CreateDeviceAndSwapChain( 
			NULL,				// [in] you can use a specific graphics device here, NULL means use the first one
			g_driverType,		// [in] driver type
			NULL,				// [in] RESERVED: Used in future versions to hand over a software rasterizer
			createDeviceFlags,	// [in] create flags
			D3D10_SDK_VERSION,	// [in] SDK version
			&sd,				// [in] swap chain descriptor (see above)
			&g_pSwapChain,		// [out] pointer to swap chain interface
			&g_pd3dDevice );	// [out] pointer to D3D device interface
		if( SUCCEEDED( hr ) )
			break; // break out of the for loop if creation succeeded
	}
	if( FAILED( hr ) )
		return hr;

	// Resources is just raw memory on the device
	// A resource view allows you to specify the use of the memory (as a frame buffer, as a texture etc)
	// We need a render target view to get access to the back buffer in the swap chain
	//
	// a chunk of memory in D3D is referred to using the Texture2D interface
	//
	ID3D10Texture2D* pBackBuffer;
	hr = g_pSwapChain->GetBuffer( 0, __uuidof( ID3D10Texture2D ), ( LPVOID* )&pBackBuffer );
	if( FAILED( hr ) )
		return hr;

	// Create a render target view onto the memory
	//
	hr = g_pd3dDevice->CreateRenderTargetView( pBackBuffer, NULL, &g_pRenderTargetView );
	pBackBuffer->Release();
	if( FAILED( hr ) )
		return hr;

	// Create depth stencil texture
	D3D10_TEXTURE2D_DESC descDepth;
	descDepth.Width = width;
	descDepth.Height = height;
	descDepth.MipLevels = 1;
	descDepth.ArraySize = 1;
	descDepth.Format = DXGI_FORMAT_D32_FLOAT;
	descDepth.SampleDesc.Count = 1;
	descDepth.SampleDesc.Quality = 0;
	descDepth.Usage = D3D10_USAGE_DEFAULT;
	descDepth.BindFlags = D3D10_BIND_DEPTH_STENCIL;
	descDepth.CPUAccessFlags = 0;
	descDepth.MiscFlags = 0;
	hr = g_pd3dDevice->CreateTexture2D( &descDepth, NULL, &g_pDepthStencil );
	if( FAILED( hr ) )
		return hr;

	// Create the depth stencil view
	D3D10_DEPTH_STENCIL_VIEW_DESC descDSV;
	descDSV.Format = descDepth.Format;
	descDSV.ViewDimension = D3D10_DSV_DIMENSION_TEXTURE2D;
	descDSV.Texture2D.MipSlice = 0;
	hr = g_pd3dDevice->CreateDepthStencilView( g_pDepthStencil, &descDSV, &g_pDepthStencilView );
	if( FAILED( hr ) )
		return hr;

	g_pd3dDevice->OMSetRenderTargets(
		1, // number of render targets, (for use with multiple render targets)
		&g_pRenderTargetView, // array of render target views to be bound
		g_pDepthStencilView // pointer to Depth/StencilView
		);

	// Setup the viewport
	D3D10_VIEWPORT vp;
	vp.Width = width;
	vp.Height = height;
	vp.MinDepth = 0.0f;
	vp.MaxDepth = 1.0f;
	vp.TopLeftX = 0;
	vp.TopLeftY = 0;
	g_pd3dDevice->RSSetViewports( 1, &vp );

	//-------------------------------------------------------------------------
	// Create the effect
	//-------------------------------------------------------------------------

	DWORD dwShaderFlags = D3D10_SHADER_ENABLE_STRICTNESS;
#if defined( DEBUG ) || defined( _DEBUG )
	// Set the D3D10_SHADER_DEBUG flag to embed debug information in the shaders.
	// Setting this flag improves the shader debugging experience, but still allows 
	// the shaders to be optimized and to run exactly the way they will run in 
	// the release configuration of this program.
	dwShaderFlags |= D3D10_SHADER_DEBUG;
#endif
	hr = D3DX10CreateEffectFromFile( 
		L"..\\src\\Tutorial06.fx",  // filename
		NULL,                       // macros
		NULL,                       // customized open/close procedures
		"fx_4_0",                   // profile
		dwShaderFlags,              // HLSL flags
		0,                          // FX flags
		g_pd3dDevice,               // device interface
		NULL,                       // pointer to an effect pool
		NULL,                       // pointer to a thread pump (asynchonous loading of files)
		&g_pEffect,                 // [out] pointer to the effect interface
		NULL,                       // [out] pointer to memory that contains effect compile errors
		NULL );                     // [out] pointer to return values (if thread pump used)

	if( FAILED( hr ) )
	{
		MessageBox( NULL,
			L"The FX file cannot be located!", L"Error", MB_OK );
		return hr;
	}

	// Obtain the technique
	g_pTechnique = g_pEffect->GetTechniqueByName( "Render" );

	// Obtain the variables
	g_pWorldVariable = g_pEffect->GetVariableByName( "World" )->AsMatrix();
	g_pViewVariable = g_pEffect->GetVariableByName( "View" )->AsMatrix();
	g_pProjectionVariable = g_pEffect->GetVariableByName( "Projection" )->AsMatrix();

	g_pLightDirVariable   = g_pEffect->GetVariableByName( "vLightDir" )->AsVector();
	g_pLightColorVariable = g_pEffect->GetVariableByName( "vLightColor" )->AsVector();
	g_pDecalVariable      = g_pEffect->GetVariableByName( "txDecal" )->AsShaderResource();

	//-------------------------------------------------------------------------
	// Create the vertex layout (stream input descriptor)
	//-------------------------------------------------------------------------

	// Define the input layout
	D3D10_INPUT_ELEMENT_DESC layout[] =
	{
		{ "POSITION",   0, DXGI_FORMAT_R32G32B32_FLOAT,    0, 0,  D3D10_INPUT_PER_VERTEX_DATA, 0 },
		{ "TEXCOORD",   0, DXGI_FORMAT_R32G32_FLOAT,       0, 12, D3D10_INPUT_PER_VERTEX_DATA, 0 },
		{ "NORMAL",     0, DXGI_FORMAT_R32G32B32_FLOAT,    0, 20, D3D10_INPUT_PER_VERTEX_DATA, 0 },
	};
	UINT numElements = sizeof( layout ) / sizeof( layout[0] );

	// Create the input layout
	D3D10_PASS_DESC PassDesc;
	g_pTechnique->GetPassByIndex( 0 )->GetDesc( &PassDesc );
	hr = g_pd3dDevice->CreateInputLayout( 
		layout,							// vertex layout
		numElements,					// number of elements
		PassDesc.pIAInputSignature,     // input signature of render pass
		PassDesc.IAInputSignatureSize,  // input size of render pass
		&g_pVertexLayout );             // [out] vertex layout

	if( FAILED( hr ) )
		return hr;

	// Set the input layout
	g_pd3dDevice->IASetInputLayout( g_pVertexLayout );

	//-------------------------------------------------------------------------
	// Create the vertex data (geometry)
	//-------------------------------------------------------------------------

	unsigned char *pData = new unsigned char[NUM_VERTICES * SIZE_OF_ONE_VERTEX];
	FILE *fp = fopen("..\\media\\MarsienneSkin_VB.raw","rb");
	assert(fp != NULL);
	fread(pData,SIZE_OF_ONE_VERTEX,NUM_VERTICES,fp);
	fclose(fp);

	// buffer descriptor
	D3D10_BUFFER_DESC bd;
	ZeroMemory(&bd, sizeof(D3D10_BUFFER_DESC));
	bd.Usage	 = D3D10_USAGE_DEFAULT;        // access by GPU for read and write    
	bd.ByteWidth = SIZE_OF_ONE_VERTEX * NUM_VERTICES; // size of the buffer
	bd.BindFlags = D3D10_BIND_VERTEX_BUFFER;   // type of the buffer (vertex buffer, index buffer, stream out buffer, etc)

	// vertex data (content of the buffer)
	D3D10_SUBRESOURCE_DATA InitData;
	ZeroMemory(&InitData, sizeof(D3D10_SUBRESOURCE_DATA));
	InitData.pSysMem = pData;

	// create a buffer
	hr = g_pd3dDevice->CreateBuffer( &bd, &InitData, &g_pVertexBuffer );
	if( FAILED( hr ) )
		return hr;

	delete [] pData;
	pData = NULL;

	// Set vertex buffer
	UINT stride = SIZE_OF_ONE_VERTEX;
	UINT offset = 0;
	g_pd3dDevice->IASetVertexBuffers( 0, 1, &g_pVertexBuffer, &stride, &offset );

	//-------------------------------------------------------------------------
	// Create the index buffer (geometry)
	//-------------------------------------------------------------------------

	// Create index buffer
	pData = new unsigned char[NUM_INDICES*sizeof(unsigned int)];

	fp = fopen("..\\media\\MarsienneSkin_IB.raw","rb");
	assert(fp != NULL);
	fread(pData,sizeof(unsigned int), NUM_INDICES,fp);
	fclose(fp);

	ZeroMemory(&bd, sizeof(D3D10_BUFFER_DESC));
	bd.Usage = D3D10_USAGE_DEFAULT;
	bd.ByteWidth = sizeof(unsigned int) * NUM_INDICES;        // 36 vertices needed for 12 triangles in a triangle list
	bd.BindFlags = D3D10_BIND_INDEX_BUFFER;
	
	InitData.pSysMem = pData;
	hr = g_pd3dDevice->CreateBuffer( &bd, &InitData, &g_pIndexBuffer );
	if( FAILED( hr ) )
		return hr;

	delete [] pData;
	pData = NULL;

	// Set index buffer
	g_pd3dDevice->IASetIndexBuffer( g_pIndexBuffer, DXGI_FORMAT_R32_UINT, 0 );

	// Set primitive topology
	g_pd3dDevice->IASetPrimitiveTopology( D3D10_PRIMITIVE_TOPOLOGY_TRIANGLELIST );

	// Initialize the world matrices
	D3DXMatrixIdentity( &g_World );

	// Initialize the view matrix
	D3DXVECTOR3 Eye( 0.0f, 1.0f, -50.0f );
	D3DXVECTOR3 At( 0.0f, 1.0f, 0.0f );
	D3DXVECTOR3 Up( 0.0f, 1.0f, 0.0f );
	D3DXMatrixLookAtLH( &g_View, &Eye, &At, &Up );

	// Initialize the projection matrix
	D3DXMatrixPerspectiveFovLH( &g_Projection, ( float )D3DX_PI * 0.25f, width / ( FLOAT )height, 0.1f, 100.0f );

#ifndef EASY_WAY
	// Create the texture
	hr = D3DX10CreateTextureFromFile( g_pd3dDevice, L"..\\media\\MarsienneDecalMap.png", NULL, NULL, &g_pTexture, NULL );
	D3DX10FilterTexture(g_pTexture,0,D3DX10_FILTER_BOX);

	if( FAILED( hr ) )
		return hr;
		
	D3D10_SHADER_RESOURCE_VIEW_DESC srvDesc;
	D3D10_RESOURCE_DIMENSION type;
	g_pTexture->GetType( &type );
	assert( type == D3D10_RESOURCE_DIMENSION_TEXTURE2D);
			
	D3D10_TEXTURE2D_DESC desc;
	ID3D10Texture2D *pTexture2D = (ID3D10Texture2D*)g_pTexture;
	pTexture2D->GetDesc( &desc );

	srvDesc.Format = desc.Format;
	srvDesc.ViewDimension = D3D10_SRV_DIMENSION_TEXTURE2D;
	srvDesc.Texture2D.MostDetailedMip = 0;
	srvDesc.Texture2D.MipLevels = desc.MipLevels;

	ID3D10ShaderResourceView *pSRView = NULL;
	g_pd3dDevice->CreateShaderResourceView( g_pTexture, &srvDesc, &g_pTextureRV );

#else

	hr = D3DX10CreateShaderResourceViewFromFile( g_pd3dDevice, L"..\\media\\MarsienneDecalMap.png", NULL, NULL, 
		&g_pTextureRV, NULL );
	if( FAILED( hr ) )
		return hr;
#endif

	g_pDecalVariable->SetResource( g_pTextureRV );

	return S_OK;
}


//--------------------------------------------------------------------------------------
// Render the frame
//--------------------------------------------------------------------------------------
void Render()
{
	// Update our time
	static float t = 0.0f;
	static DWORD dwTimeStart = 0;
	DWORD dwTimeCur = GetTickCount();
	if( dwTimeStart == 0 )
		dwTimeStart = dwTimeCur;
	t = ( dwTimeCur - dwTimeStart ) / 1000.0f;

	//
	// Animate the cube
	//
	D3DXMatrixRotationY( &g_World, t );

	// Setup our lighting parameters
	D3DXVECTOR4 vLightDirs[2] =
	{
		D3DXVECTOR4( -0.577f, 0.577f, -0.577f, 1.0f ),
		D3DXVECTOR4( 0.0f, 0.0f, -1.0f, 1.0f ),
	};
	D3DXVECTOR4 vLightColors[2] =
	{
		D3DXVECTOR4( 1.0f, 1.0f, 1.0f, 1.0f ),
		D3DXVECTOR4( 0.5f, 0.0f, 0.0f, 1.0f )
	};

	//
	// Clear the back buffer
	//
	float ClearColor[4] = { 0.5f, 0.5f, 1.0f, 1.0f }; // red,green,blue,alpha
	g_pd3dDevice->ClearRenderTargetView( g_pRenderTargetView, ClearColor );
	//
	// Clear the depth buffer to 1.0 (max depth)
	//
	g_pd3dDevice->ClearDepthStencilView( g_pDepthStencilView, D3D10_CLEAR_DEPTH, 1.0f, 0 );

	//
	// Update variables
	//
	g_pWorldVariable->SetMatrix( ( float* )&g_World );
	g_pViewVariable->SetMatrix( ( float* )&g_View );
	g_pProjectionVariable->SetMatrix( ( float* )&g_Projection );

	//
	// Update lighting variables
	//
	g_pLightDirVariable->SetFloatVectorArray( ( float* )vLightDirs, 0, 2 );
	g_pLightColorVariable->SetFloatVectorArray( ( float* )vLightColors, 0, 2 );
	//
	// Renders a triangle
	//
	D3D10_TECHNIQUE_DESC techDesc;
	g_pTechnique->GetDesc( &techDesc );
	for( UINT p = 0; p < techDesc.Passes; ++p )
	{
		g_pTechnique->GetPassByIndex( p )->Apply( 0 );
		g_pd3dDevice->DrawIndexed( NUM_INDICES, 0, 0 );        // 36 vertices needed for 12 triangles in a triangle list
	}

	g_pSwapChain->Present( 0, 0 );
}


//--------------------------------------------------------------------------------------
// Clean up the objects we've created
//--------------------------------------------------------------------------------------
void CleanupDevice()
{
	if( g_pd3dDevice ) g_pd3dDevice->ClearState();

	if( g_pVertexBuffer )	g_pVertexBuffer->Release();
	if( g_pIndexBuffer )	g_pIndexBuffer->Release();
	if( g_pVertexLayout )	g_pVertexLayout->Release();
	if( g_pTextureRV )		g_pTextureRV->Release();
	if( g_pEffect )			g_pEffect->Release();
	if( g_pRenderTargetView ) g_pRenderTargetView->Release();
	if( g_pSwapChain )		g_pSwapChain->Release();
	if( g_pd3dDevice )		g_pd3dDevice->Release();
}


