/*
 * Copyright (C) 2013, 2014, 2015
 * Computer Graphics Group, University of Siegen
 * Written by Martin Lambers <martin.lambers@uni-siegen.de>
 * All rights reserved.
 */

#include <cassert>
#include <cstdio>
#include <cstdlib>
#include <cstdint>
#include <cmath>
#include <algorithm>

#ifdef _MSC_VER
# define snprintf _snprintf
#endif

#include "lodepng.h"


/* Flow data set description */

const char* filename = "flow.raw";

const int x_cells = 400;
const float x_start = -0.5f;
const float x_end = +7.5f;
const float x_step = (x_end - x_start) / x_cells;

const int y_cells = 50;
const float y_start = -0.5f;
const float y_end = +0.5f;
const float y_step = (y_end - y_start) / y_cells;

const int t_cells = 1001;
const float t_start = 15.0f;
const float t_end = 23.0f;
const float t_step = (t_end - t_start) / t_cells;

const size_t data_size = x_cells * y_cells * t_cells * 2 * sizeof(float);
float* data = NULL; // will be read in main()


/* Data set access */

class vector
{
  public:
      float x, y;
      vector() { x = 0.0f; y = 0.0f; }
      vector(float X, float Y) { x = X; y = Y; }
};

vector flow(int x, int y, int t)
{
    // Direct grid cell access.
    float vx = data[2 * (t * y_cells * x_cells + y * x_cells + x) + 0];
    float vy = data[2 * (t * y_cells * x_cells + y * x_cells + x) + 1];
    return vector(vx, vy);
}

vector flow(float x, float y, float t)
{
    // Nearest neighbor flow access using physical units for x,y,t
    // TODO: implement interpolation!
    int cx = round((x - x_start) / (x_end - x_start) * x_cells);
    int cy = round((y - y_start) / (y_end - y_start) * y_cells);
    int ct = round((t - t_start) / (t_end - t_start) * t_cells);
    cx = std::min(std::max(cx, 0), x_cells - 1);
    cy = std::min(std::max(cy, 0), y_cells - 1);
    ct = std::min(std::max(ct, 0), t_cells - 1);
    return flow(cx, cy, ct);
}


/* Create image for one time step from vector lengths */

void time_cell_image(int time_cell, float max_length, uint32_t* image)
{
    for (int y = 0; y < y_cells; y++) {
        for (int x = 0; x < x_cells; x++) {
            vector v = flow(x, y, time_cell);
            float length = std::sqrt(v.x * v.x + v.y * v.y);
            uint32_t red = length / max_length * 255.0f;
            if (red > 255)
                red = 255;
            uint32_t green = red;
            uint32_t blue = red;
            uint32_t alpha = 255;
            uint32_t color = (alpha << 24) | (red << 16) | (green << 8) | blue;
            image[y * x_cells + x] = color;
        }
    }
}

int main(void /* int argc, char *argv[] */)
{
    /* Read the flow data set */
    data = static_cast<float*>(malloc(data_size));
    assert(data);
    FILE* f = fopen("flow.raw", "rb");
    assert(f);
    fread(data, 1, data_size, f);
    fclose(f);

    /* Produce a series of images, one for each time step. */
    uint32_t* image = new uint32_t[x_cells * y_cells];
    for (int t = 0; t < t_cells; t++) {
        char filename[16];
        snprintf(filename, 16, "img-%04d.png", t);
        time_cell_image(t, 1.5f /* arbitrary guess */, image);
        unsigned r = lodepng::encode(std::string(filename),
                reinterpret_cast<unsigned char*>(image), x_cells, y_cells);
        assert(r == 0);
    }

    /* You can draw 2D geometries into your images, either in your own code, or
     * using e.g. ImageMagick <http://imagemagick.org/>:
     *
     * $ convert -stroke "rgb(255,0,0)" -draw "line 20,25 380,25" \
     *           img-0000.png out-0000.png
     *
     * $ convert -stroke "hsb(33.3%,100%,100%)" -draw "line  20,25 140,25" \
     *           -stroke "hsb(66.7%,50%,50%)"   -draw "line 141,25 260,25" \
     *           -stroke "hsb(100%,75%,75%)"    -draw "line 261,25 380,25" \
     *           img-0000.png out-0000.png
     */

    /* You can now produce a video from your image series using FFmpeg
     * <http://ffmpeg.org/>:
     *
     * $ ffmpeg -i img-%04d.png -b 200k video.mp4
     */

    return 0;
}
