/*
 * Copyright (C) 2013, 2014
 * Computer Graphics Group, University of Siegen
 * Written by Martin Lambers <martin.lambers@uni-siegen.de>
 * All rights reserved.
 */

#include <vector>
#include <cassert>
#include <cstddef>

#define GLM_FORCE_RADIANS
#define GLM_SWIZZLE
#include <glm/glm.hpp>
#include <glm/gtc/constants.hpp>
#include <glm/gtx/transform.hpp>

#include "geometries.hpp"

using namespace glm;


void geom_quad(
        std::vector<vec3>& positions,
        std::vector<vec3>& normals,
        std::vector<vec2>& texcoords,
        std::vector<unsigned int>& indices)
{
    static const float p[] = {
        -1.0f, +1.0f, 0.0f,
        +1.0f, +1.0f, 0.0f,
        +1.0f, -1.0f, 0.0f,
        -1.0f, -1.0f, 0.0f
    };
    static const float n[] = {
        0.0f, 0.0f, 1.0f,
        0.0f, 0.0f, 1.0f,
        0.0f, 0.0f, 1.0f,
        0.0f, 0.0f, 1.0f
    };
    static const float t[] = {
        0.0f, 1.0f,
        1.0f, 1.0f,
        1.0f, 0.0f,
        0.0f, 0.0f
    };
    static const unsigned int i[] = {
        0, 1, 3, 1, 2, 3
    };
    static const vec3 *vp = reinterpret_cast<const vec3*>(p);
    static const vec3 *vn = reinterpret_cast<const vec3*>(n);
    static const vec2 *vt = reinterpret_cast<const vec2*>(t);
    positions.assign(vp, vp + sizeof(p) / sizeof(vec3));
    normals.assign(vn, vn + sizeof(n) / sizeof(vec3));
    texcoords.assign(vt, vt + sizeof(t) / sizeof(vec2));
    indices.assign(i, i + sizeof(i) / sizeof(unsigned int));
}

void geom_cube(
        std::vector<vec3>& positions,
        std::vector<vec3>& normals,
        std::vector<vec2>& texcoords,
        std::vector<unsigned int>& indices)
{
    static const float p[] = {
        -1.0f, +1.0f, +1.0f,   +1.0f, +1.0f, +1.0f,   +1.0f, -1.0f, +1.0f,   -1.0f, -1.0f, +1.0f, // front
        -1.0f, +1.0f, -1.0f,   +1.0f, +1.0f, -1.0f,   +1.0f, -1.0f, -1.0f,   -1.0f, -1.0f, -1.0f, // back
        +1.0f, +1.0f, -1.0f,   +1.0f, +1.0f, +1.0f,   +1.0f, -1.0f, +1.0f,   +1.0f, -1.0f, -1.0f, // right
        -1.0f, +1.0f, -1.0f,   -1.0f, +1.0f, +1.0f,   -1.0f, -1.0f, +1.0f,   -1.0f, -1.0f, -1.0f, // left
        -1.0f, +1.0f, +1.0f,   +1.0f, +1.0f, +1.0f,   +1.0f, +1.0f, -1.0f,   -1.0f, +1.0f, -1.0f, // top
        -1.0f, -1.0f, +1.0f,   +1.0f, -1.0f, +1.0f,   +1.0f, -1.0f, -1.0f,   -1.0f, -1.0f, -1.0f  // bottom
    };
    static const float n[] = {
          0.0f, 0.0f, +1.0f,   0.0f, 0.0f, +1.0f,   0.0f, 0.0f, +1.0f,   0.0f, 0.0f, +1.0f, // front
          0.0f, 0.0f, -1.0f,   0.0f, 0.0f, -1.0f,   0.0f, 0.0f, -1.0f,   0.0f, 0.0f, -1.0f, // back
          +1.0f, 0.0f, 0.0f,   +1.0f, 0.0f, 0.0f,   +1.0f, 0.0f, 0.0f,   +1.0f, 0.0f, 0.0f, // right
          -1.0f, 0.0f, 0.0f,   -1.0f, 0.0f, 0.0f,   -1.0f, 0.0f, 0.0f,   -1.0f, 0.0f, 0.0f, // left
          0.0f, +1.0f, 0.0f,   0.0f, +1.0f, 0.0f,   0.0f, +1.0f, 0.0f,   0.0f, +1.0f, 0.0f, // top
          0.0f, -1.0f, 0.0f,   0.0f, -1.0f, 0.0f,   0.0f, -1.0f, 0.0f,   0.0f, -1.0f, 0.0f  // bottom
    };
    static const float t[] = {
          0.0f, 1.0f,   1.0f, 1.0f,   1.0f, 0.0f,   0.0f, 0.0f, // front
          1.0f, 1.0f,   0.0f, 1.0f,   0.0f, 0.0f,   1.0f, 0.0f, // back
          1.0f, 1.0f,   0.0f, 1.0f,   0.0f, 0.0f,   1.0f, 0.0f, // right
          0.0f, 1.0f,   1.0f, 1.0f,   1.0f, 0.0f,   0.0f, 0.0f, // left
          0.0f, 0.0f,   1.0f, 0.0f,   1.0f, 1.0f,   0.0f, 1.0f, // top
          0.0f, 1.0f,   1.0f, 1.0f,   1.0f, 0.0f,   0.0f, 0.0f, // bottom
    };
    static const unsigned int i[] = {
        0, 1, 3, 1, 2, 3, // front face
        4, 5, 7, 5, 6, 7, // back face
        8, 9, 11, 9, 10, 11, // right face
        12, 13, 15, 13, 14, 15, // left face
        16, 17, 19, 17, 18, 19, // top face
        20, 21, 23, 21, 22, 23, // bottom face
    };
    static const vec3 *vp = reinterpret_cast<const vec3*>(p);
    static const vec3 *vn = reinterpret_cast<const vec3*>(n);
    static const vec2 *vt = reinterpret_cast<const vec2*>(t);
    positions.assign(vp, vp + sizeof(p) / sizeof(vec3));
    normals.assign(vn, vn + sizeof(n) / sizeof(vec3));
    texcoords.assign(vt, vt + sizeof(t) / sizeof(vec2));
    indices.assign(i, i + sizeof(i) / sizeof(unsigned int));
}

void geom_disk(
        std::vector<vec3>& positions,
        std::vector<vec3>& normals,
        std::vector<vec2>& texcoords,
        std::vector<unsigned int>& indices,
        float inner_radius, int slices)
{
    const int loops = 1;

    assert(inner_radius >= 0.0f);
    assert(inner_radius <= 1.0f);
    assert(slices >= 4);
    assert(loops >= 1);

    positions.clear();
    normals.clear();
    texcoords.clear();
    indices.clear();

    for (int i = 0; i <= loops; i++) {
        float ty = static_cast<float>(i) / loops;
        float r = inner_radius + ty * (1.0f - inner_radius);
        for (int j = 0; j <= slices; j++) {
            float tx = static_cast<float>(j) / slices;
            float alpha = tx * 2.0f * pi<float>() + half_pi<float>();
            positions.push_back(vec3(r * cos(alpha), r * sin(alpha), 0.0f));
            normals.push_back(vec3(0.0f, 0.0f, 1.0f));
            texcoords.push_back(vec2(1.0f - tx, ty));
            if (i < loops && j < slices) {
                indices.push_back((i + 0) * (slices + 1) + (j + 0));
                indices.push_back((i + 0) * (slices + 1) + (j + 1));
                indices.push_back((i + 1) * (slices + 1) + (j + 0));
                indices.push_back((i + 0) * (slices + 1) + (j + 1));
                indices.push_back((i + 1) * (slices + 1) + (j + 1));
                indices.push_back((i + 1) * (slices + 1) + (j + 0));
            }
        }
    }
}

void geom_sphere(
        std::vector<vec3>& positions,
        std::vector<vec3>& normals,
        std::vector<vec2>& texcoords,
        std::vector<unsigned int>& indices,
        int slices, int stacks)
{
    assert(slices >= 4);
    assert(stacks >= 2);

    positions.clear();
    normals.clear();
    texcoords.clear();
    indices.clear();

    for (int i = 0; i <= stacks; i++) {
        float ty = static_cast<float>(i) / stacks;
        float lat = ty * pi<float>();
        for (int j = 0; j <= slices; j++) {
            float tx = static_cast<float>(j) / slices;
            float lon = tx * 2.0f * pi<float>() - half_pi<float>();
            float sinlat = sin(lat);
            float coslat = cos(lat);
            float sinlon = sin(lon);
            float coslon = cos(lon);
            vec3 pos = vec3(sinlat * coslon, coslat, sinlat * sinlon);
            positions.push_back(pos);
            normals.push_back(pos);
            texcoords.push_back(vec2(1.0f - tx, 1.0f - ty));
            if (i < stacks && j < slices) {
                indices.push_back((i + 0) * (slices + 1) + (j + 0));
                indices.push_back((i + 0) * (slices + 1) + (j + 1));
                indices.push_back((i + 1) * (slices + 1) + (j + 0));
                indices.push_back((i + 0) * (slices + 1) + (j + 1));
                indices.push_back((i + 1) * (slices + 1) + (j + 1));
                indices.push_back((i + 1) * (slices + 1) + (j + 0));
            }
        }
    }
}

void geom_cylinder(
        std::vector<vec3>& positions,
        std::vector<vec3>& normals,
        std::vector<vec2>& texcoords,
        std::vector<unsigned int>& indices,
        int slices)
{
    const int stacks = 1;

    positions.clear();
    normals.clear();
    texcoords.clear();
    indices.clear();

    for (int i = 0; i <= stacks; i++) {
        float ty = static_cast<float>(i) / stacks;
        for (int j = 0; j <= slices; j++) {
            float tx = static_cast<float>(j) / slices;
            float alpha = tx * 2.0f * pi<float>() - half_pi<float>();
            vec3 pos = vec3(cos(alpha), -(ty * 2.0f - 1.0f), sin(alpha));
            positions.push_back(pos);
            normals.push_back(vec3(pos.x, 0.0f, pos.z));
            texcoords.push_back(vec2(1.0f - tx, 1.0f - ty));
            if (i < stacks && j < slices) {
                indices.push_back((i + 0) * (slices + 1) + (j + 0));
                indices.push_back((i + 0) * (slices + 1) + (j + 1));
                indices.push_back((i + 1) * (slices + 1) + (j + 0));
                indices.push_back((i + 0) * (slices + 1) + (j + 1));
                indices.push_back((i + 1) * (slices + 1) + (j + 1));
                indices.push_back((i + 1) * (slices + 1) + (j + 0));
            }
        }
    }
}

void geom_cone(
        std::vector<vec3>& positions,
        std::vector<vec3>& normals,
        std::vector<vec2>& texcoords,
        std::vector<unsigned int>& indices,
        int slices, int stacks)
{
    assert(slices >= 4);
    assert(stacks >= 2);

    positions.clear();
    normals.clear();
    texcoords.clear();
    indices.clear();

    for (int i = 0; i <= stacks; i++) {
        float ty = static_cast<float>(i) / stacks;
        for (int j = 0; j <= slices; j++) {
            float tx = static_cast<float>(j) / slices;
            float alpha = tx * 2.0f * pi<float>() - half_pi<float>();
            vec3 pos = vec3(ty * cos(alpha), -(ty * 2.0f - 1.0f), ty * sin(alpha));
            positions.push_back(pos);
            normals.push_back(normalize(vec3(pos.x, 0.5f, pos.z)));
            texcoords.push_back(vec2(1.0f - tx, 1.0f - ty));
            if (i < stacks && j < slices) {
                indices.push_back((i + 0) * (slices + 1) + (j + 0));
                indices.push_back((i + 0) * (slices + 1) + (j + 1));
                indices.push_back((i + 1) * (slices + 1) + (j + 0));
                indices.push_back((i + 0) * (slices + 1) + (j + 1));
                indices.push_back((i + 1) * (slices + 1) + (j + 1));
                indices.push_back((i + 1) * (slices + 1) + (j + 0));
            }
        }
    }
}

void geom_torus(
        std::vector<vec3>& positions,
        std::vector<vec3>& normals,
        std::vector<vec2>& texcoords,
        std::vector<unsigned int>& indices,
        float inner_radius, int sides, int rings)
{
    assert(inner_radius >= 0.0f);
    assert(inner_radius < 1.0f);
    assert(sides >= 4);
    assert(rings >= 4);

    positions.clear();
    normals.clear();
    texcoords.clear();
    indices.clear();

    float ringscale = (1.0f - inner_radius) / 2.0f;
    float ringoffset = inner_radius + ringscale;

    for (int i = 0; i <= sides; i++) {
        float ty = static_cast<float>(i) / sides;
        float alpha = ty * 2.0f * pi<float>() - half_pi<float>();
        mat4 rot = rotate(alpha, vec3(0.0f, 0.0f, -1.0f));
        for (int j = 0; j <= rings; j++) {
            float tx = static_cast<float>(j) / rings;
            float beta = tx * 2.0f * pi<float>() - pi<float>();
            vec4 pos = rot * vec4(ringoffset + ringscale * cos(beta), 0.0f, ringscale * sin(beta), 1.0f);
            vec4 center = rot * vec4(ringoffset, 0.0f, 0.0f, 1.0f);
            positions.push_back(pos.xyz());
            normals.push_back(normalize((pos - center).xyz()));
            texcoords.push_back(vec2(1.0f - tx, 1.0f - ty));
            if (i < sides && j < rings) {
                indices.push_back((i + 0) * (rings + 1) + (j + 0));
                indices.push_back((i + 0) * (rings + 1) + (j + 1));
                indices.push_back((i + 1) * (rings + 1) + (j + 0));
                indices.push_back((i + 0) * (rings + 1) + (j + 1));
                indices.push_back((i + 1) * (rings + 1) + (j + 1));
                indices.push_back((i + 1) * (rings + 1) + (j + 0));
            }
        }
    }
}

void geom_teapot(
        std::vector<vec3>& positions,
        std::vector<vec3>& normals,
        std::vector<vec2>& texcoords,
        std::vector<unsigned int>& indices)
{
    // The teapot data was obtained as follows:
    // - download http://cggmwww.csie.nctu.edu.tw/courses/cgu/2002/prog1/teapot.obj
    // - import to meshlab
    // - apply 'remove duplicate faces', 'remove duplicate vertices, 'remove zero area faces'
    // - export to obj
    // - load with tiny object loader
    // - determine axis aligned bounding box
    // - translate y by -ymax/2
    // - scale everything by 1/xmax

    static const float teapot_positions[] = {
          +0.407688f, +0.240245f, -0.000000f,   +0.402437f, +0.240245f, +0.066893f,
          +0.398391f, +0.250564f, +0.066221f,   +0.403589f, +0.250564f, -0.000000f,
          +0.396803f, +0.258591f, +0.065957f,   +0.401980f, +0.258591f, -0.000000f,
          +0.397327f, +0.264324f, +0.066044f,   +0.402511f, +0.264324f, -0.000000f,
          +0.399620f, +0.267764f, +0.066425f,   +0.404834f, +0.267764f, -0.000000f,
          +0.403335f, +0.268910f, +0.067043f,   +0.408598f, +0.268910f, -0.000000f,
          +0.408128f, +0.267764f, +0.067839f,   +0.413454f, +0.267764f, -0.000000f,
          +0.413655f, +0.264324f, +0.068758f,   +0.419052f, +0.264324f, -0.000000f,
          +0.419569f, +0.258591f, +0.069741f,   +0.425044f, +0.258591f, -0.000000f,
          +0.425527f, +0.250564f, +0.070731f,   +0.431079f, +0.250564f, -0.000000f,
          +0.431182f, +0.240245f, +0.071672f,   +0.436808f, +0.240245f, -0.000000f,
          +0.387206f, +0.240245f, +0.130069f,   +0.383313f, +0.250564f, +0.128761f,
          +0.381785f, +0.258591f, +0.128248f,   +0.382290f, +0.264324f, +0.128417f,
          +0.384495f, +0.267764f, +0.129158f,   +0.388070f, +0.268910f, +0.130359f,
          +0.392681f, +0.267764f, +0.131908f,   +0.397999f, +0.264324f, +0.133695f,
          +0.403689f, +0.258591f, +0.135606f,   +0.409421f, +0.250564f, +0.137532f,
          +0.414863f, +0.240245f, +0.139359f,   +0.362777f, +0.240245f, +0.188743f,
          +0.359130f, +0.250564f, +0.186846f,   +0.357698f, +0.258591f, +0.186100f,
          +0.358171f, +0.264324f, +0.186347f,   +0.360238f, +0.267764f, +0.187422f,
          +0.363586f, +0.268910f, +0.189165f,   +0.367908f, +0.267764f, +0.191413f,
          +0.372889f, +0.264324f, +0.194005f,   +0.378221f, +0.258591f, +0.196778f,
          +0.383591f, +0.250564f, +0.199573f,   +0.388690f, +0.240245f, +0.202225f,
          +0.329934f, +0.240245f, +0.242134f,   +0.326616f, +0.250564f, +0.239699f,
          +0.325315f, +0.258591f, +0.238744f,   +0.325745f, +0.264324f, +0.239060f,
          +0.327624f, +0.267764f, +0.240439f,   +0.330670f, +0.268910f, +0.242674f,
          +0.334600f, +0.267764f, +0.245559f,   +0.339131f, +0.264324f, +0.248883f,
          +0.343979f, +0.258591f, +0.252441f,   +0.348863f, +0.250564f, +0.256027f,
          +0.353500f, +0.240245f, +0.259429f,   +0.289458f, +0.240245f, +0.289458f,
          +0.286549f, +0.250564f, +0.286549f,   +0.285406f, +0.258591f, +0.285406f,
          +0.285783f, +0.264324f, +0.285783f,   +0.287433f, +0.267764f, +0.287432f,
          +0.290105f, +0.268910f, +0.290105f,   +0.293552f, +0.267764f, +0.293552f,
          +0.297527f, +0.264324f, +0.297527f,   +0.301781f, +0.258591f, +0.301781f,
          +0.306066f, +0.250564f, +0.306066f,   +0.310134f, +0.240245f, +0.310134f,
          +0.242134f, +0.240245f, +0.329934f,   +0.239699f, +0.250564f, +0.326616f,
          +0.238744f, +0.258591f, +0.325315f,   +0.239060f, +0.264324f, +0.325745f,
          +0.240439f, +0.267764f, +0.327624f,   +0.242674f, +0.268910f, +0.330670f,
          +0.245559f, +0.267764f, +0.334600f,   +0.248883f, +0.264324f, +0.339131f,
          +0.252441f, +0.258591f, +0.343979f,   +0.256027f, +0.250564f, +0.348863f,
          +0.259429f, +0.240245f, +0.353500f,   +0.188743f, +0.240245f, +0.362777f,
          +0.186846f, +0.250564f, +0.359130f,   +0.186100f, +0.258591f, +0.357698f,
          +0.186347f, +0.264324f, +0.358171f,   +0.187422f, +0.267764f, +0.360238f,
          +0.189165f, +0.268910f, +0.363586f,   +0.191413f, +0.267764f, +0.367908f,
          +0.194005f, +0.264324f, +0.372890f,   +0.196778f, +0.258591f, +0.378221f,
          +0.199573f, +0.250564f, +0.383591f,   +0.202225f, +0.240245f, +0.388690f,
          +0.130069f, +0.240245f, +0.387206f,   +0.128761f, +0.250564f, +0.383313f,
          +0.128248f, +0.258591f, +0.381785f,   +0.128417f, +0.264324f, +0.382290f,
          +0.129158f, +0.267764f, +0.384495f,   +0.130359f, +0.268910f, +0.388070f,
          +0.131908f, +0.267764f, +0.392681f,   +0.133695f, +0.264324f, +0.397999f,
          +0.135606f, +0.258591f, +0.403689f,   +0.137532f, +0.250564f, +0.409421f,
          +0.139359f, +0.240245f, +0.414863f,   +0.066893f, +0.240245f, +0.402437f,
          +0.066221f, +0.250564f, +0.398391f,   +0.065957f, +0.258591f, +0.396803f,
          +0.066044f, +0.264324f, +0.397327f,   +0.066425f, +0.267764f, +0.399620f,
          +0.067043f, +0.268910f, +0.403335f,   +0.067839f, +0.267764f, +0.408128f,
          +0.068758f, +0.264324f, +0.413655f,   +0.069741f, +0.258591f, +0.419569f,
          +0.070731f, +0.250564f, +0.425527f,   +0.071671f, +0.240245f, +0.431182f,
          -0.000000f, +0.240245f, +0.407688f,   -0.000000f, +0.250564f, +0.403589f,
          -0.000000f, +0.258591f, +0.401980f,   -0.000000f, +0.264324f, +0.402511f,
          -0.000000f, +0.267764f, +0.404834f,   -0.000000f, +0.268910f, +0.408598f,
          -0.000000f, +0.267764f, +0.413454f,   -0.000000f, +0.264324f, +0.419052f,
          -0.000000f, +0.258591f, +0.425044f,   -0.000000f, +0.250564f, +0.431079f,
          -0.000000f, +0.240245f, +0.436808f,   -0.073262f, +0.240245f, +0.402437f,
          -0.070864f, +0.250564f, +0.398391f,   -0.069218f, +0.258591f, +0.396803f,
          -0.068229f, +0.264324f, +0.397327f,   -0.067801f, +0.267764f, +0.399620f,
          -0.067839f, +0.268910f, +0.403335f,   -0.068247f, +0.267764f, +0.408128f,
          -0.068930f, +0.264324f, +0.413655f,   -0.069792f, +0.258591f, +0.419569f,
          -0.070738f, +0.250564f, +0.425527f,   -0.071672f, +0.240245f, +0.431182f,
          -0.140133f, +0.240245f, +0.387206f,   -0.136098f, +0.250564f, +0.383313f,
          -0.133400f, +0.258591f, +0.381785f,   -0.131870f, +0.264324f, +0.382290f,
          -0.131333f, +0.267764f, +0.384495f,   -0.131617f, +0.268910f, +0.388070f,
          -0.132552f, +0.267764f, +0.392681f,   -0.133966f, +0.264324f, +0.397999f,
          -0.135687f, +0.258591f, +0.403689f,   -0.137541f, +0.250564f, +0.409421f,
          -0.139359f, +0.240245f, +0.414863f,   -0.200302f, +0.240245f, +0.362777f,
          -0.195271f, +0.250564f, +0.359130f,   -0.192019f, +0.258591f, +0.357698f,
          -0.190311f, +0.264324f, +0.358171f,   -0.189918f, +0.267764f, +0.360238f,
          -0.190609f, +0.268910f, +0.363586f,   -0.192153f, +0.267764f, +0.367908f,
          -0.194317f, +0.264324f, +0.372889f,   -0.196871f, +0.258591f, +0.378221f,
          -0.199584f, +0.250564f, +0.383591f,   -0.202225f, +0.240245f, +0.388690f,
          -0.253456f, +0.240245f, +0.329934f,   -0.247953f, +0.250564f, +0.326616f,
          -0.244540f, +0.258591f, +0.325315f,   -0.242944f, +0.264324f, +0.325745f,
          -0.242884f, +0.267764f, +0.327624f,   -0.244090f, +0.268910f, +0.330670f,
          -0.246283f, +0.267764f, +0.334600f,   -0.249189f, +0.264324f, +0.339131f,
          -0.252532f, +0.258591f, +0.343979f,   -0.256037f, +0.250564f, +0.348863f,
          -0.259429f, +0.240245f, +0.353500f,   -0.299287f, +0.240245f, +0.289458f,
          -0.293713f, +0.250564f, +0.286549f,   -0.290438f, +0.258591f, +0.285406f,
          -0.289154f, +0.264324f, +0.285783f,   -0.289555f, +0.267764f, +0.287432f,
          -0.291333f, +0.268910f, +0.290105f,   -0.294182f, +0.267764f, +0.293552f,
          -0.297793f, +0.264324f, +0.297527f,   -0.301859f, +0.258591f, +0.301781f,
          -0.306076f, +0.250564f, +0.306066f,   -0.310134f, +0.240245f, +0.310134f,
          -0.337482f, +0.240245f, +0.242134f,   -0.332119f, +0.250564f, +0.239699f,
          -0.329179f, +0.258591f, +0.238744f,   -0.328333f, +0.264324f, +0.239060f,
          -0.329255f, +0.267764f, +0.240439f,   -0.331613f, +0.268910f, +0.242674f,
          -0.335083f, +0.267764f, +0.245559f,   -0.339335f, +0.264324f, +0.248883f,
          -0.344040f, +0.258591f, +0.252441f,   -0.348871f, +0.250564f, +0.256027f,
          -0.353500f, +0.240245f, +0.259429f,   -0.367731f, +0.240245f, +0.188743f,
          -0.362741f, +0.250564f, +0.186846f,   -0.360234f, +0.258591f, +0.186100f,
          -0.359870f, +0.264324f, +0.186347f,   -0.361307f, +0.267764f, +0.187422f,
          -0.364206f, +0.268910f, +0.189165f,   -0.368225f, +0.267764f, +0.191413f,
          -0.373023f, +0.264324f, +0.194005f,   -0.378261f, +0.258591f, +0.196778f,
          -0.383596f, +0.250564f, +0.199573f,   -0.388690f, +0.240245f, +0.202225f,
          -0.389722f, +0.240245f, +0.130069f,   -0.385147f, +0.250564f, +0.128761f,
          -0.383073f, +0.258591f, +0.128248f,   -0.383152f, +0.264324f, +0.128417f,
          -0.385039f, +0.267764f, +0.129158f,   -0.388384f, +0.268910f, +0.130359f,
          -0.392843f, +0.267764f, +0.131908f,   -0.398066f, +0.264324f, +0.133695f,
          -0.403710f, +0.258591f, +0.135606f,   -0.409425f, +0.250564f, +0.137532f,
          -0.414863f, +0.240245f, +0.139359f,   -0.403144f, +0.240245f, +0.066893f,
          -0.398906f, +0.250564f, +0.066221f,   -0.397165f, +0.258591f, +0.065957f,
          -0.397570f, +0.264324f, +0.066044f,   -0.399773f, +0.267764f, +0.066425f,
          -0.403423f, +0.268910f, +0.067043f,   -0.408174f, +0.267764f, +0.067839f,
          -0.413674f, +0.264324f, +0.068758f,   -0.419575f, +0.258591f, +0.069741f,
          -0.425527f, +0.250564f, +0.070731f,   -0.431182f, +0.240245f, +0.071671f,
          -0.407688f, +0.240245f, -0.000000f,   -0.403589f, +0.250564f, -0.000000f,
          -0.401980f, +0.258591f, -0.000000f,   -0.402511f, +0.264324f, -0.000000f,
          -0.404834f, +0.267764f, -0.000000f,   -0.408598f, +0.268910f, -0.000000f,
          -0.413454f, +0.267764f, -0.000000f,   -0.419052f, +0.264324f, -0.000000f,
          -0.425044f, +0.258591f, -0.000000f,   -0.431079f, +0.250564f, -0.000000f,
          -0.436808f, +0.240245f, -0.000000f,   -0.402437f, +0.240245f, -0.066893f,
          -0.398391f, +0.250564f, -0.066221f,   -0.396803f, +0.258591f, -0.065957f,
          -0.397327f, +0.264324f, -0.066044f,   -0.399620f, +0.267764f, -0.066425f,
          -0.403335f, +0.268910f, -0.067043f,   -0.408128f, +0.267764f, -0.067839f,
          -0.413655f, +0.264324f, -0.068758f,   -0.419569f, +0.258591f, -0.069741f,
          -0.425527f, +0.250564f, -0.070732f,   -0.431182f, +0.240245f, -0.071672f,
          -0.387206f, +0.240245f, -0.130069f,   -0.383313f, +0.250564f, -0.128761f,
          -0.381785f, +0.258591f, -0.128248f,   -0.382290f, +0.264324f, -0.128417f,
          -0.384495f, +0.267764f, -0.129158f,   -0.388070f, +0.268910f, -0.130359f,
          -0.392681f, +0.267764f, -0.131908f,   -0.397999f, +0.264324f, -0.133695f,
          -0.403689f, +0.258591f, -0.135606f,   -0.409421f, +0.250564f, -0.137532f,
          -0.414863f, +0.240245f, -0.139359f,   -0.362777f, +0.240245f, -0.188743f,
          -0.359130f, +0.250564f, -0.186846f,   -0.357698f, +0.258591f, -0.186101f,
          -0.358171f, +0.264324f, -0.186347f,   -0.360238f, +0.267764f, -0.187422f,
          -0.363586f, +0.268910f, -0.189165f,   -0.367908f, +0.267764f, -0.191413f,
          -0.372889f, +0.264324f, -0.194005f,   -0.378221f, +0.258591f, -0.196778f,
          -0.383591f, +0.250564f, -0.199573f,   -0.388690f, +0.240245f, -0.202225f,
          -0.329934f, +0.240245f, -0.242134f,   -0.326616f, +0.250564f, -0.239699f,
          -0.325315f, +0.258591f, -0.238744f,   -0.325745f, +0.264324f, -0.239060f,
          -0.327624f, +0.267764f, -0.240439f,   -0.330670f, +0.268910f, -0.242674f,
          -0.334600f, +0.267764f, -0.245559f,   -0.339131f, +0.264324f, -0.248883f,
          -0.343979f, +0.258591f, -0.252441f,   -0.348863f, +0.250564f, -0.256027f,
          -0.353500f, +0.240245f, -0.259429f,   -0.289458f, +0.240245f, -0.289458f,
          -0.286549f, +0.250564f, -0.286549f,   -0.285406f, +0.258591f, -0.285406f,
          -0.285783f, +0.264324f, -0.285783f,   -0.287433f, +0.267764f, -0.287433f,
          -0.290105f, +0.268910f, -0.290105f,   -0.293552f, +0.267764f, -0.293553f,
          -0.297527f, +0.264324f, -0.297527f,   -0.301781f, +0.258591f, -0.301781f,
          -0.306066f, +0.250564f, -0.306066f,   -0.310134f, +0.240245f, -0.310134f,
          -0.242134f, +0.240245f, -0.329934f,   -0.239699f, +0.250564f, -0.326616f,
          -0.238744f, +0.258591f, -0.325315f,   -0.239060f, +0.264324f, -0.325745f,
          -0.240439f, +0.267764f, -0.327624f,   -0.242674f, +0.268910f, -0.330670f,
          -0.245559f, +0.267764f, -0.334600f,   -0.248883f, +0.264324f, -0.339131f,
          -0.252441f, +0.258591f, -0.343979f,   -0.256027f, +0.250564f, -0.348863f,
          -0.259429f, +0.240245f, -0.353500f,   -0.188743f, +0.240245f, -0.362777f,
          -0.186846f, +0.250564f, -0.359130f,   -0.186100f, +0.258591f, -0.357698f,
          -0.186347f, +0.264324f, -0.358171f,   -0.187422f, +0.267764f, -0.360238f,
          -0.189165f, +0.268910f, -0.363586f,   -0.191413f, +0.267764f, -0.367908f,
          -0.194005f, +0.264324f, -0.372890f,   -0.196778f, +0.258591f, -0.378221f,
          -0.199573f, +0.250564f, -0.383591f,   -0.202225f, +0.240245f, -0.388690f,
          -0.130069f, +0.240245f, -0.387206f,   -0.128761f, +0.250564f, -0.383313f,
          -0.128248f, +0.258591f, -0.381785f,   -0.128417f, +0.264324f, -0.382290f,
          -0.129158f, +0.267764f, -0.384495f,   -0.130359f, +0.268910f, -0.388070f,
          -0.131908f, +0.267764f, -0.392681f,   -0.133695f, +0.264324f, -0.397999f,
          -0.135606f, +0.258591f, -0.403689f,   -0.137532f, +0.250564f, -0.409421f,
          -0.139359f, +0.240245f, -0.414863f,   -0.066893f, +0.240245f, -0.402437f,
          -0.066221f, +0.250564f, -0.398391f,   -0.065957f, +0.258591f, -0.396803f,
          -0.066044f, +0.264324f, -0.397328f,   -0.066425f, +0.267764f, -0.399620f,
          -0.067043f, +0.268910f, -0.403335f,   -0.067839f, +0.267764f, -0.408128f,
          -0.068758f, +0.264324f, -0.413655f,   -0.069741f, +0.258591f, -0.419569f,
          -0.070731f, +0.250564f, -0.425527f,   -0.071671f, +0.240245f, -0.431182f,
          +0.000000f, +0.240245f, -0.407688f,   +0.000000f, +0.250564f, -0.403589f,
          +0.000000f, +0.258591f, -0.401980f,   +0.000000f, +0.264324f, -0.402511f,
          +0.000000f, +0.267764f, -0.404834f,   +0.000000f, +0.268910f, -0.408598f,
          +0.000000f, +0.267764f, -0.413454f,   +0.000000f, +0.264324f, -0.419052f,
          +0.000000f, +0.258591f, -0.425044f,   +0.000000f, +0.250564f, -0.431079f,
          +0.000000f, +0.240245f, -0.436808f,   +0.066893f, +0.240245f, -0.402437f,
          +0.066221f, +0.250564f, -0.398391f,   +0.065957f, +0.258591f, -0.396803f,
          +0.066044f, +0.264324f, -0.397328f,   +0.066425f, +0.267764f, -0.399620f,
          +0.067043f, +0.268910f, -0.403335f,   +0.067839f, +0.267764f, -0.408128f,
          +0.068758f, +0.264324f, -0.413655f,   +0.069741f, +0.258591f, -0.419569f,
          +0.070731f, +0.250564f, -0.425527f,   +0.071672f, +0.240245f, -0.431182f,
          +0.130069f, +0.240245f, -0.387206f,   +0.128761f, +0.250564f, -0.383313f,
          +0.128248f, +0.258591f, -0.381785f,   +0.128417f, +0.264324f, -0.382290f,
          +0.129158f, +0.267764f, -0.384495f,   +0.130359f, +0.268910f, -0.388070f,
          +0.131908f, +0.267764f, -0.392681f,   +0.133695f, +0.264324f, -0.397999f,
          +0.135606f, +0.258591f, -0.403689f,   +0.137532f, +0.250564f, -0.409421f,
          +0.139359f, +0.240245f, -0.414863f,   +0.188743f, +0.240245f, -0.362777f,
          +0.186846f, +0.250564f, -0.359130f,   +0.186101f, +0.258591f, -0.357698f,
          +0.186347f, +0.264324f, -0.358171f,   +0.187422f, +0.267764f, -0.360238f,
          +0.189165f, +0.268910f, -0.363586f,   +0.191413f, +0.267764f, -0.367908f,
          +0.194005f, +0.264324f, -0.372890f,   +0.196778f, +0.258591f, -0.378221f,
          +0.199573f, +0.250564f, -0.383591f,   +0.202225f, +0.240245f, -0.388690f,
          +0.242134f, +0.240245f, -0.329934f,   +0.239699f, +0.250564f, -0.326616f,
          +0.238744f, +0.258591f, -0.325315f,   +0.239060f, +0.264324f, -0.325745f,
          +0.240439f, +0.267764f, -0.327624f,   +0.242674f, +0.268910f, -0.330670f,
          +0.245559f, +0.267764f, -0.334600f,   +0.248883f, +0.264324f, -0.339131f,
          +0.252441f, +0.258591f, -0.343979f,   +0.256027f, +0.250564f, -0.348863f,
          +0.259429f, +0.240245f, -0.353500f,   +0.289458f, +0.240245f, -0.289458f,
          +0.286549f, +0.250564f, -0.286549f,   +0.285406f, +0.258591f, -0.285406f,
          +0.285783f, +0.264324f, -0.285783f,   +0.287433f, +0.267764f, -0.287433f,
          +0.290105f, +0.268910f, -0.290105f,   +0.293552f, +0.267764f, -0.293553f,
          +0.297527f, +0.264324f, -0.297527f,   +0.301781f, +0.258591f, -0.301781f,
          +0.306066f, +0.250564f, -0.306066f,   +0.310134f, +0.240245f, -0.310134f,
          +0.329934f, +0.240245f, -0.242134f,   +0.326616f, +0.250564f, -0.239699f,
          +0.325315f, +0.258591f, -0.238744f,   +0.325745f, +0.264324f, -0.239060f,
          +0.327624f, +0.267764f, -0.240439f,   +0.330670f, +0.268910f, -0.242674f,
          +0.334600f, +0.267764f, -0.245559f,   +0.339131f, +0.264324f, -0.248883f,
          +0.343979f, +0.258591f, -0.252441f,   +0.348863f, +0.250564f, -0.256027f,
          +0.353500f, +0.240245f, -0.259429f,   +0.362777f, +0.240245f, -0.188743f,
          +0.359130f, +0.250564f, -0.186846f,   +0.357698f, +0.258591f, -0.186101f,
          +0.358171f, +0.264324f, -0.186347f,   +0.360238f, +0.267764f, -0.187422f,
          +0.363586f, +0.268910f, -0.189165f,   +0.367908f, +0.267764f, -0.191413f,
          +0.372890f, +0.264324f, -0.194005f,   +0.378221f, +0.258591f, -0.196778f,
          +0.383591f, +0.250564f, -0.199573f,   +0.388690f, +0.240245f, -0.202225f,
          +0.387206f, +0.240245f, -0.130069f,   +0.383313f, +0.250564f, -0.128761f,
          +0.381785f, +0.258591f, -0.128248f,   +0.382290f, +0.264324f, -0.128417f,
          +0.384495f, +0.267764f, -0.129158f,   +0.388070f, +0.268910f, -0.130359f,
          +0.392681f, +0.267764f, -0.131908f,   +0.397999f, +0.264324f, -0.133695f,
          +0.403689f, +0.258591f, -0.135606f,   +0.409421f, +0.250564f, -0.137532f,
          +0.414863f, +0.240245f, -0.139359f,   +0.402437f, +0.240245f, -0.066893f,
          +0.398391f, +0.250564f, -0.066221f,   +0.396803f, +0.258591f, -0.065957f,
          +0.397327f, +0.264324f, -0.066044f,   +0.399620f, +0.267764f, -0.066425f,
          +0.403335f, +0.268910f, -0.067043f,   +0.408128f, +0.267764f, -0.067839f,
          +0.413655f, +0.264324f, -0.068758f,   +0.419569f, +0.258591f, -0.069741f,
          +0.425527f, +0.250564f, -0.070731f,   +0.431182f, +0.240245f, -0.071672f,
          +0.452670f, +0.194402f, +0.075243f,   +0.458575f, +0.194402f, -0.000000f,
          +0.473725f, +0.148690f, +0.078743f,   +0.479907f, +0.148690f, -0.000000f,
          +0.493920f, +0.103239f, +0.082100f,   +0.500365f, +0.103239f, -0.000000f,
          +0.512820f, +0.058183f, +0.085241f,   +0.519511f, +0.058183f, -0.000000f,
          +0.529995f, +0.013651f, +0.088096f,   +0.536911f, +0.013651f, -0.000000f,
          +0.545015f, -0.030227f, +0.090593f,   +0.552126f, -0.030227f, -0.000000f,
          +0.557447f, -0.073319f, +0.092659f,   +0.564720f, -0.073319f, -0.000000f,
          +0.566861f, -0.115492f, +0.094224f,   +0.574257f, -0.115492f, -0.000000f,
          +0.572826f, -0.156617f, +0.095216f,   +0.580299f, -0.156617f, -0.000000f,
          +0.574910f, -0.196564f, +0.095562f,   +0.582411f, -0.196564f, -0.000000f,
          +0.435538f, +0.194402f, +0.146304f,   +0.455796f, +0.148690f, +0.153109f,
          +0.475225f, +0.103239f, +0.159637f,   +0.493411f, +0.058183f, +0.165745f,
          +0.509936f, +0.013651f, +0.171295f,   +0.524387f, -0.030227f, +0.176150f,
          +0.536349f, -0.073319f, +0.180169f,   +0.545407f, -0.115492f, +0.183211f,
          +0.551146f, -0.156617f, +0.185139f,   +0.553151f, -0.196564f, +0.185812f,
          +0.408059f, +0.194402f, +0.212303f,   +0.427040f, +0.148690f, +0.222178f,
          +0.445244f, +0.103239f, +0.231648f,   +0.462282f, +0.058183f, +0.240513f,
          +0.477765f, +0.013651f, +0.248568f,   +0.491303f, -0.030227f, +0.255612f,
          +0.502511f, -0.073319f, +0.261443f,   +0.510997f, -0.115492f, +0.265858f,
          +0.516374f, -0.156617f, +0.268656f,   +0.518253f, -0.196564f, +0.269633f,
          +0.371116f, +0.194402f, +0.272358f,   +0.388379f, +0.148690f, +0.285026f,
          +0.404934f, +0.103239f, +0.297176f,   +0.420430f, +0.058183f, +0.308547f,
          +0.434511f, +0.013651f, +0.318882f,   +0.446825f, -0.030227f, +0.327918f,
          +0.457017f, -0.073319f, +0.335399f,   +0.464735f, -0.115492f, +0.341063f,
          +0.469625f, -0.156617f, +0.344652f,   +0.471334f, -0.196564f, +0.345906f,
          +0.325589f, +0.194402f, +0.325589f,   +0.340734f, +0.148690f, +0.340734f,
          +0.355258f, +0.103239f, +0.355258f,   +0.368853f, +0.058183f, +0.368853f,
          +0.381207f, +0.013651f, +0.381207f,   +0.392009f, -0.030227f, +0.392009f,
          +0.400952f, -0.073319f, +0.400952f,   +0.407723f, -0.115492f, +0.407723f,
          +0.412013f, -0.156617f, +0.412013f,   +0.413512f, -0.196564f, +0.413512f,
          +0.272358f, +0.194402f, +0.371116f,   +0.285026f, +0.148690f, +0.388379f,
          +0.297176f, +0.103239f, +0.404934f,   +0.308547f, +0.058183f, +0.420430f,
          +0.318882f, +0.013651f, +0.434510f,   +0.327918f, -0.030227f, +0.446825f,
          +0.335399f, -0.073319f, +0.457017f,   +0.341063f, -0.115492f, +0.464735f,
          +0.344652f, -0.156617f, +0.469625f,   +0.345906f, -0.196564f, +0.471334f,
          +0.212303f, +0.194402f, +0.408059f,   +0.222178f, +0.148690f, +0.427040f,
          +0.231648f, +0.103239f, +0.445244f,   +0.240513f, +0.058183f, +0.462282f,
          +0.248568f, +0.013651f, +0.477765f,   +0.255612f, -0.030227f, +0.491303f,
          +0.261443f, -0.073319f, +0.502511f,   +0.265858f, -0.115492f, +0.510997f,
          +0.268656f, -0.156617f, +0.516374f,   +0.269633f, -0.196564f, +0.518253f,
          +0.146304f, +0.194402f, +0.435538f,   +0.153109f, +0.148690f, +0.455796f,
          +0.159637f, +0.103239f, +0.475225f,   +0.165745f, +0.058183f, +0.493411f,
          +0.171295f, +0.013651f, +0.509936f,   +0.176150f, -0.030227f, +0.524387f,
          +0.180169f, -0.073319f, +0.536349f,   +0.183211f, -0.115492f, +0.545407f,
          +0.185139f, -0.156617f, +0.551146f,   +0.185812f, -0.196564f, +0.553151f,
          +0.075243f, +0.194402f, +0.452670f,   +0.078743f, +0.148690f, +0.473725f,
          +0.082100f, +0.103239f, +0.493920f,   +0.085241f, +0.058183f, +0.512820f,
          +0.088096f, +0.013651f, +0.529995f,   +0.090593f, -0.030227f, +0.545015f,
          +0.092659f, -0.073319f, +0.557447f,   +0.094224f, -0.115492f, +0.566861f,
          +0.095215f, -0.156617f, +0.572826f,   +0.095562f, -0.196564f, +0.574910f,
          -0.000000f, +0.194402f, +0.458575f,   -0.000000f, +0.148690f, +0.479907f,
          -0.000000f, +0.103239f, +0.500365f,   -0.000000f, +0.058183f, +0.519511f,
          -0.000000f, +0.013651f, +0.536911f,   -0.000000f, -0.030227f, +0.552126f,
          -0.000000f, -0.073319f, +0.564720f,   -0.000000f, -0.115492f, +0.574257f,
          -0.000000f, -0.156617f, +0.580299f,   -0.000000f, -0.196564f, +0.582411f,
          -0.075243f, +0.194402f, +0.452670f,   -0.078743f, +0.148690f, +0.473725f,
          -0.082100f, +0.103239f, +0.493920f,   -0.085241f, +0.058183f, +0.512820f,
          -0.088096f, +0.013651f, +0.529995f,   -0.090593f, -0.030227f, +0.545015f,
          -0.092659f, -0.073319f, +0.557447f,   -0.094224f, -0.115492f, +0.566861f,
          -0.095216f, -0.156617f, +0.572826f,   -0.095562f, -0.196564f, +0.574910f,
          -0.146304f, +0.194402f, +0.435538f,   -0.153109f, +0.148690f, +0.455796f,
          -0.159637f, +0.103239f, +0.475225f,   -0.165745f, +0.058183f, +0.493411f,
          -0.171296f, +0.013651f, +0.509936f,   -0.176150f, -0.030227f, +0.524387f,
          -0.180169f, -0.073319f, +0.536349f,   -0.183211f, -0.115492f, +0.545407f,
          -0.185139f, -0.156617f, +0.551146f,   -0.185812f, -0.196564f, +0.553151f,
          -0.212303f, +0.194402f, +0.408059f,   -0.222178f, +0.148690f, +0.427040f,
          -0.231648f, +0.103239f, +0.445244f,   -0.240513f, +0.058183f, +0.462282f,
          -0.248568f, +0.013651f, +0.477765f,   -0.255612f, -0.030227f, +0.491303f,
          -0.261443f, -0.073319f, +0.502511f,   -0.265858f, -0.115492f, +0.510997f,
          -0.268656f, -0.156617f, +0.516374f,   -0.269633f, -0.196564f, +0.518253f,
          -0.272358f, +0.194402f, +0.371116f,   -0.285026f, +0.148690f, +0.388379f,
          -0.297176f, +0.103239f, +0.404934f,   -0.308547f, +0.058183f, +0.420430f,
          -0.318882f, +0.013651f, +0.434510f,   -0.327918f, -0.030227f, +0.446825f,
          -0.335399f, -0.073319f, +0.457017f,   -0.341063f, -0.115492f, +0.464735f,
          -0.344652f, -0.156617f, +0.469625f,   -0.345906f, -0.196564f, +0.471334f,
          -0.325589f, +0.194402f, +0.325589f,   -0.340734f, +0.148690f, +0.340734f,
          -0.355258f, +0.103239f, +0.355258f,   -0.368853f, +0.058183f, +0.368853f,
          -0.381207f, +0.013651f, +0.381207f,   -0.392009f, -0.030227f, +0.392009f,
          -0.400952f, -0.073319f, +0.400952f,   -0.407723f, -0.115492f, +0.407723f,
          -0.412013f, -0.156617f, +0.412013f,   -0.413512f, -0.196564f, +0.413512f,
          -0.371116f, +0.194402f, +0.272358f,   -0.388379f, +0.148690f, +0.285026f,
          -0.404934f, +0.103239f, +0.297176f,   -0.420430f, +0.058183f, +0.308547f,
          -0.434511f, +0.013651f, +0.318882f,   -0.446825f, -0.030227f, +0.327918f,
          -0.457017f, -0.073319f, +0.335399f,   -0.464735f, -0.115492f, +0.341063f,
          -0.469625f, -0.156617f, +0.344652f,   -0.471334f, -0.196564f, +0.345906f,
          -0.408059f, +0.194402f, +0.212303f,   -0.427040f, +0.148690f, +0.222177f,
          -0.445244f, +0.103239f, +0.231648f,   -0.462282f, +0.058183f, +0.240513f,
          -0.477765f, +0.013651f, +0.248568f,   -0.491303f, -0.030227f, +0.255612f,
          -0.502511f, -0.073319f, +0.261443f,   -0.510997f, -0.115492f, +0.265858f,
          -0.516374f, -0.156617f, +0.268656f,   -0.518253f, -0.196564f, +0.269633f,
          -0.435538f, +0.194402f, +0.146304f,   -0.455796f, +0.148690f, +0.153109f,
          -0.475225f, +0.103239f, +0.159637f,   -0.493411f, +0.058183f, +0.165745f,
          -0.509936f, +0.013651f, +0.171295f,   -0.524387f, -0.030227f, +0.176150f,
          -0.536349f, -0.073319f, +0.180169f,   -0.545407f, -0.115492f, +0.183211f,
          -0.551146f, -0.156617f, +0.185139f,   -0.553151f, -0.196564f, +0.185812f,
          -0.452670f, +0.194402f, +0.075243f,   -0.473725f, +0.148690f, +0.078743f,
          -0.493920f, +0.103239f, +0.082100f,   -0.512820f, +0.058183f, +0.085241f,
          -0.529995f, +0.013651f, +0.088096f,   -0.545015f, -0.030227f, +0.090593f,
          -0.557447f, -0.073319f, +0.092659f,   -0.566861f, -0.115492f, +0.094224f,
          -0.572826f, -0.156617f, +0.095215f,   -0.574910f, -0.196564f, +0.095562f,
          -0.458575f, +0.194402f, -0.000000f,   -0.479907f, +0.148690f, -0.000000f,
          -0.500365f, +0.103239f, -0.000000f,   -0.519511f, +0.058183f, -0.000000f,
          -0.536911f, +0.013651f, -0.000000f,   -0.552126f, -0.030227f, -0.000000f,
          -0.564720f, -0.073319f, -0.000000f,   -0.574257f, -0.115492f, -0.000000f,
          -0.580299f, -0.156617f, -0.000000f,   -0.582411f, -0.196564f, -0.000000f,
          -0.452670f, +0.194402f, -0.075243f,   -0.473725f, +0.148690f, -0.078743f,
          -0.493920f, +0.103239f, -0.082100f,   -0.512820f, +0.058183f, -0.085241f,
          -0.529995f, +0.013651f, -0.088096f,   -0.545015f, -0.030227f, -0.090593f,
          -0.557447f, -0.073319f, -0.092659f,   -0.566861f, -0.115492f, -0.094224f,
          -0.572826f, -0.156617f, -0.095216f,   -0.574910f, -0.196564f, -0.095562f,
          -0.435538f, +0.194402f, -0.146304f,   -0.455796f, +0.148690f, -0.153110f,
          -0.475225f, +0.103239f, -0.159637f,   -0.493411f, +0.058183f, -0.165745f,
          -0.509936f, +0.013651f, -0.171296f,   -0.524387f, -0.030227f, -0.176150f,
          -0.536349f, -0.073319f, -0.180169f,   -0.545407f, -0.115492f, -0.183211f,
          -0.551146f, -0.156617f, -0.185139f,   -0.553151f, -0.196564f, -0.185812f,
          -0.408059f, +0.194402f, -0.212303f,   -0.427040f, +0.148690f, -0.222178f,
          -0.445244f, +0.103239f, -0.231648f,   -0.462282f, +0.058183f, -0.240513f,
          -0.477765f, +0.013651f, -0.248568f,   -0.491303f, -0.030227f, -0.255612f,
          -0.502511f, -0.073319f, -0.261443f,   -0.510997f, -0.115492f, -0.265858f,
          -0.516374f, -0.156617f, -0.268656f,   -0.518253f, -0.196564f, -0.269633f,
          -0.371116f, +0.194402f, -0.272358f,   -0.388379f, +0.148690f, -0.285026f,
          -0.404934f, +0.103239f, -0.297176f,   -0.420430f, +0.058183f, -0.308547f,
          -0.434511f, +0.013651f, -0.318882f,   -0.446825f, -0.030227f, -0.327918f,
          -0.457017f, -0.073319f, -0.335399f,   -0.464735f, -0.115492f, -0.341063f,
          -0.469625f, -0.156617f, -0.344652f,   -0.471334f, -0.196564f, -0.345906f,
          -0.325589f, +0.194402f, -0.325589f,   -0.340734f, +0.148690f, -0.340734f,
          -0.355258f, +0.103239f, -0.355258f,   -0.368853f, +0.058183f, -0.368853f,
          -0.381207f, +0.013651f, -0.381207f,   -0.392009f, -0.030227f, -0.392009f,
          -0.400952f, -0.073319f, -0.400952f,   -0.407723f, -0.115492f, -0.407723f,
          -0.412013f, -0.156617f, -0.412013f,   -0.413512f, -0.196564f, -0.413512f,
          -0.272358f, +0.194402f, -0.371116f,   -0.285026f, +0.148690f, -0.388379f,
          -0.297176f, +0.103239f, -0.404934f,   -0.308547f, +0.058183f, -0.420430f,
          -0.318882f, +0.013651f, -0.434511f,   -0.327918f, -0.030227f, -0.446825f,
          -0.335399f, -0.073319f, -0.457017f,   -0.341063f, -0.115492f, -0.464735f,
          -0.344652f, -0.156617f, -0.469625f,   -0.345906f, -0.196564f, -0.471334f,
          -0.212303f, +0.194402f, -0.408059f,   -0.222178f, +0.148690f, -0.427040f,
          -0.231648f, +0.103239f, -0.445244f,   -0.240513f, +0.058183f, -0.462282f,
          -0.248568f, +0.013651f, -0.477765f,   -0.255612f, -0.030227f, -0.491303f,
          -0.261443f, -0.073319f, -0.502511f,   -0.265858f, -0.115492f, -0.510997f,
          -0.268656f, -0.156617f, -0.516374f,   -0.269633f, -0.196564f, -0.518253f,
          -0.146304f, +0.194402f, -0.435538f,   -0.153109f, +0.148690f, -0.455796f,
          -0.159637f, +0.103239f, -0.475225f,   -0.165745f, +0.058183f, -0.493411f,
          -0.171295f, +0.013651f, -0.509936f,   -0.176150f, -0.030227f, -0.524387f,
          -0.180169f, -0.073319f, -0.536349f,   -0.183211f, -0.115492f, -0.545407f,
          -0.185139f, -0.156617f, -0.551146f,   -0.185812f, -0.196564f, -0.553151f,
          -0.075243f, +0.194402f, -0.452670f,   -0.078743f, +0.148690f, -0.473725f,
          -0.082100f, +0.103239f, -0.493920f,   -0.085241f, +0.058183f, -0.512820f,
          -0.088096f, +0.013651f, -0.529995f,   -0.090593f, -0.030227f, -0.545015f,
          -0.092659f, -0.073319f, -0.557447f,   -0.094224f, -0.115492f, -0.566861f,
          -0.095215f, -0.156617f, -0.572826f,   -0.095562f, -0.196564f, -0.574910f,
          +0.000000f, +0.194402f, -0.458577f,   +0.000000f, +0.148690f, -0.479907f,
          +0.000000f, +0.103239f, -0.500365f,   +0.000000f, +0.058183f, -0.519511f,
          +0.000000f, +0.013651f, -0.536911f,   +0.000000f, -0.030227f, -0.552126f,
          +0.000000f, -0.073319f, -0.564720f,   +0.000000f, -0.115492f, -0.574257f,
          +0.000000f, -0.156617f, -0.580299f,   +0.000000f, -0.196564f, -0.582411f,
          +0.075243f, +0.194402f, -0.452670f,   +0.078743f, +0.148690f, -0.473725f,
          +0.082100f, +0.103239f, -0.493920f,   +0.085241f, +0.058183f, -0.512820f,
          +0.088096f, +0.013651f, -0.529995f,   +0.090593f, -0.030227f, -0.545015f,
          +0.092659f, -0.073319f, -0.557447f,   +0.094224f, -0.115492f, -0.566861f,
          +0.095216f, -0.156617f, -0.572826f,   +0.095562f, -0.196564f, -0.574910f,
          +0.146304f, +0.194402f, -0.435538f,   +0.153109f, +0.148690f, -0.455796f,
          +0.159637f, +0.103239f, -0.475225f,   +0.165745f, +0.058183f, -0.493411f,
          +0.171296f, +0.013651f, -0.509936f,   +0.176150f, -0.030227f, -0.524387f,
          +0.180169f, -0.073319f, -0.536349f,   +0.183211f, -0.115492f, -0.545407f,
          +0.185139f, -0.156617f, -0.551146f,   +0.185812f, -0.196564f, -0.553151f,
          +0.212303f, +0.194402f, -0.408059f,   +0.222178f, +0.148690f, -0.427040f,
          +0.231648f, +0.103239f, -0.445244f,   +0.240513f, +0.058183f, -0.462282f,
          +0.248568f, +0.013651f, -0.477765f,   +0.255612f, -0.030227f, -0.491303f,
          +0.261443f, -0.073319f, -0.502511f,   +0.265858f, -0.115492f, -0.510997f,
          +0.268656f, -0.156617f, -0.516374f,   +0.269633f, -0.196564f, -0.518253f,
          +0.272358f, +0.194402f, -0.371116f,   +0.285026f, +0.148690f, -0.388379f,
          +0.297176f, +0.103239f, -0.404934f,   +0.308547f, +0.058183f, -0.420430f,
          +0.318882f, +0.013651f, -0.434511f,   +0.327918f, -0.030227f, -0.446825f,
          +0.335399f, -0.073319f, -0.457017f,   +0.341063f, -0.115492f, -0.464735f,
          +0.344652f, -0.156617f, -0.469625f,   +0.345906f, -0.196564f, -0.471334f,
          +0.325589f, +0.194402f, -0.325589f,   +0.340734f, +0.148690f, -0.340734f,
          +0.355258f, +0.103239f, -0.355258f,   +0.368853f, +0.058183f, -0.368853f,
          +0.381207f, +0.013651f, -0.381207f,   +0.392009f, -0.030227f, -0.392009f,
          +0.400952f, -0.073319f, -0.400952f,   +0.407723f, -0.115492f, -0.407723f,
          +0.412013f, -0.156617f, -0.412013f,   +0.413512f, -0.196564f, -0.413512f,
          +0.371116f, +0.194402f, -0.272358f,   +0.388379f, +0.148690f, -0.285026f,
          +0.404934f, +0.103239f, -0.297176f,   +0.420430f, +0.058183f, -0.308547f,
          +0.434511f, +0.013651f, -0.318882f,   +0.446825f, -0.030227f, -0.327918f,
          +0.457017f, -0.073319f, -0.335399f,   +0.464735f, -0.115492f, -0.341063f,
          +0.469625f, -0.156617f, -0.344652f,   +0.471334f, -0.196564f, -0.345906f,
          +0.408059f, +0.194402f, -0.212303f,   +0.427040f, +0.148690f, -0.222178f,
          +0.445244f, +0.103239f, -0.231648f,   +0.462282f, +0.058183f, -0.240513f,
          +0.477765f, +0.013651f, -0.248568f,   +0.491303f, -0.030227f, -0.255612f,
          +0.502511f, -0.073319f, -0.261443f,   +0.510997f, -0.115492f, -0.265858f,
          +0.516374f, -0.156617f, -0.268656f,   +0.518253f, -0.196564f, -0.269633f,
          +0.435538f, +0.194402f, -0.146304f,   +0.455796f, +0.148690f, -0.153109f,
          +0.475225f, +0.103239f, -0.159637f,   +0.493411f, +0.058183f, -0.165745f,
          +0.509936f, +0.013651f, -0.171295f,   +0.524387f, -0.030227f, -0.176150f,
          +0.536349f, -0.073319f, -0.180169f,   +0.545407f, -0.115492f, -0.183211f,
          +0.551146f, -0.156617f, -0.185139f,   +0.553151f, -0.196564f, -0.185812f,
          +0.452670f, +0.194402f, -0.075243f,   +0.473725f, +0.148690f, -0.078743f,
          +0.493920f, +0.103239f, -0.082100f,   +0.512820f, +0.058183f, -0.085241f,
          +0.529995f, +0.013651f, -0.088096f,   +0.545015f, -0.030227f, -0.090593f,
          +0.557447f, -0.073319f, -0.092659f,   +0.566861f, -0.115492f, -0.094224f,
          +0.572826f, -0.156617f, -0.095215f,   +0.574910f, -0.196564f, -0.095562f,
          +0.570885f, -0.233932f, +0.094893f,   +0.578334f, -0.233932f, -0.000000f,
          +0.559962f, -0.267501f, +0.093077f,   +0.567268f, -0.267501f, -0.000000f,
          +0.543865f, -0.297401f, +0.090402f,   +0.550961f, -0.297401f, -0.000000f,
          +0.524318f, -0.323762f, +0.087153f,   +0.531159f, -0.323762f, -0.000000f,
          +0.503046f, -0.346717f, +0.083617f,   +0.509610f, -0.346717f, -0.000000f,
          +0.481774f, -0.366395f, +0.080081f,   +0.488061f, -0.366395f, -0.000000f,
          +0.462227f, -0.382928f, +0.076832f,   +0.468259f, -0.382928f, -0.000000f,
          +0.446130f, -0.396447f, +0.074156f,   +0.451951f, -0.396447f, -0.000000f,
          +0.435207f, -0.407084f, +0.072340f,   +0.440885f, -0.407084f, -0.000000f,
          +0.431182f, -0.414968f, +0.071672f,   +0.436808f, -0.414968f, -0.000000f,
          +0.549279f, -0.233932f, +0.184511f,   +0.538769f, -0.267501f, +0.180981f,
          +0.523280f, -0.297401f, +0.175779f,   +0.504474f, -0.323762f, +0.169461f,
          +0.484007f, -0.346717f, +0.162586f,   +0.463540f, -0.366395f, +0.155711f,
          +0.444734f, -0.382928f, +0.149393f,   +0.429245f, -0.396447f, +0.144191f,
          +0.418735f, -0.407084f, +0.140660f,   +0.414863f, -0.414968f, +0.139359f,
          +0.514625f, -0.233932f, +0.267746f,   +0.504778f, -0.267501f, +0.262623f,
          +0.490267f, -0.297401f, +0.255073f,   +0.472646f, -0.323762f, +0.245906f,
          +0.453471f, -0.346717f, +0.235929f,   +0.434296f, -0.366395f, +0.225952f,
          +0.416676f, -0.382928f, +0.216785f,   +0.402164f, -0.396447f, +0.209235f,
          +0.392317f, -0.407084f, +0.204112f,   +0.388690f, -0.414968f, +0.202225f,
          +0.468034f, -0.233932f, +0.343484f,   +0.459079f, -0.267501f, +0.336912f,
          +0.445881f, -0.297401f, +0.327227f,   +0.429857f, -0.323762f, +0.315467f,
          +0.412417f, -0.346717f, +0.302667f,   +0.394977f, -0.366395f, +0.289868f,
          +0.378953f, -0.382928f, +0.278108f,   +0.365755f, -0.396447f, +0.268423f,
          +0.356799f, -0.407084f, +0.261851f,   +0.353500f, -0.414968f, +0.259429f,
          +0.410617f, -0.233932f, +0.410617f,   +0.402761f, -0.267501f, +0.402761f,
          +0.391182f, -0.297401f, +0.391182f,   +0.377123f, -0.323762f, +0.377123f,
          +0.361823f, -0.346717f, +0.361823f,   +0.346523f, -0.366395f, +0.346523f,
          +0.332464f, -0.382928f, +0.332464f,   +0.320885f, -0.396447f, +0.320885f,
          +0.313029f, -0.407084f, +0.313029f,   +0.310134f, -0.414968f, +0.310134f,
          +0.343484f, -0.233932f, +0.468035f,   +0.336912f, -0.267501f, +0.459079f,
          +0.327227f, -0.297401f, +0.445881f,   +0.315467f, -0.323762f, +0.429857f,
          +0.302667f, -0.346717f, +0.412417f,   +0.289868f, -0.366395f, +0.394977f,
          +0.278108f, -0.382928f, +0.378953f,   +0.268423f, -0.396447f, +0.365755f,
          +0.261851f, -0.407084f, +0.356799f,   +0.259429f, -0.414968f, +0.353500f,
          +0.267746f, -0.233932f, +0.514625f,   +0.262623f, -0.267501f, +0.504778f,
          +0.255073f, -0.297401f, +0.490267f,   +0.245906f, -0.323762f, +0.472646f,
          +0.235929f, -0.346717f, +0.453471f,   +0.225952f, -0.366395f, +0.434296f,
          +0.216785f, -0.382928f, +0.416676f,   +0.209235f, -0.396447f, +0.402164f,
          +0.204112f, -0.407084f, +0.392317f,   +0.202225f, -0.414968f, +0.388690f,
          +0.184511f, -0.233932f, +0.549279f,   +0.180981f, -0.267501f, +0.538769f,
          +0.175779f, -0.297401f, +0.523280f,   +0.169461f, -0.323762f, +0.504474f,
          +0.162586f, -0.346717f, +0.484007f,   +0.155711f, -0.366395f, +0.463540f,
          +0.149393f, -0.382928f, +0.444734f,   +0.144190f, -0.396447f, +0.429245f,
          +0.140660f, -0.407084f, +0.418735f,   +0.139359f, -0.414968f, +0.414863f,
          +0.094893f, -0.233932f, +0.570885f,   +0.093077f, -0.267501f, +0.559962f,
          +0.090402f, -0.297401f, +0.543865f,   +0.087152f, -0.323762f, +0.524318f,
          +0.083617f, -0.346717f, +0.503046f,   +0.080081f, -0.366395f, +0.481774f,
          +0.076832f, -0.382928f, +0.462227f,   +0.074156f, -0.396447f, +0.446130f,
          +0.072340f, -0.407084f, +0.435207f,   +0.071671f, -0.414968f, +0.431182f,
          -0.000000f, -0.233932f, +0.578334f,   -0.000000f, -0.267501f, +0.567268f,
          -0.000000f, -0.297401f, +0.550961f,   -0.000000f, -0.323762f, +0.531159f,
          -0.000000f, -0.346717f, +0.509610f,   -0.000000f, -0.366395f, +0.488061f,
          -0.000000f, -0.382928f, +0.468259f,   -0.000000f, -0.396447f, +0.451951f,
          -0.000000f, -0.407084f, +0.440885f,   -0.000000f, -0.414968f, +0.436808f,
          -0.094893f, -0.233932f, +0.570885f,   -0.093077f, -0.267501f, +0.559962f,
          -0.090402f, -0.297401f, +0.543865f,   -0.087153f, -0.323762f, +0.524318f,
          -0.083617f, -0.346717f, +0.503046f,   -0.080081f, -0.366395f, +0.481774f,
          -0.076832f, -0.382928f, +0.462227f,   -0.074156f, -0.396447f, +0.446130f,
          -0.072340f, -0.407084f, +0.435207f,   -0.071672f, -0.414968f, +0.431182f,
          -0.184511f, -0.233932f, +0.549279f,   -0.180981f, -0.267501f, +0.538769f,
          -0.175779f, -0.297401f, +0.523280f,   -0.169461f, -0.323762f, +0.504474f,
          -0.162586f, -0.346717f, +0.484007f,   -0.155711f, -0.366395f, +0.463540f,
          -0.149393f, -0.382928f, +0.444734f,   -0.144191f, -0.396447f, +0.429245f,
          -0.140660f, -0.407084f, +0.418735f,   -0.139359f, -0.414968f, +0.414863f,
          -0.267746f, -0.233932f, +0.514625f,   -0.262623f, -0.267501f, +0.504778f,
          -0.255073f, -0.297401f, +0.490267f,   -0.245906f, -0.323762f, +0.472646f,
          -0.235929f, -0.346717f, +0.453471f,   -0.225952f, -0.366395f, +0.434296f,
          -0.216785f, -0.382928f, +0.416676f,   -0.209235f, -0.396447f, +0.402164f,
          -0.204112f, -0.407084f, +0.392317f,   -0.202225f, -0.414968f, +0.388690f,
          -0.343484f, -0.233932f, +0.468034f,   -0.336912f, -0.267501f, +0.459079f,
          -0.327227f, -0.297401f, +0.445881f,   -0.315467f, -0.323762f, +0.429857f,
          -0.302667f, -0.346717f, +0.412417f,   -0.289868f, -0.366395f, +0.394977f,
          -0.278108f, -0.382928f, +0.378953f,   -0.268423f, -0.396447f, +0.365755f,
          -0.261851f, -0.407084f, +0.356799f,   -0.259429f, -0.414968f, +0.353500f,
          -0.410617f, -0.233932f, +0.410617f,   -0.402761f, -0.267501f, +0.402761f,
          -0.391182f, -0.297401f, +0.391182f,   -0.377123f, -0.323762f, +0.377123f,
          -0.361823f, -0.346717f, +0.361823f,   -0.346523f, -0.366395f, +0.346523f,
          -0.332464f, -0.382928f, +0.332464f,   -0.320885f, -0.396447f, +0.320885f,
          -0.313029f, -0.407084f, +0.313029f,   -0.310134f, -0.414968f, +0.310134f,
          -0.468035f, -0.233932f, +0.343484f,   -0.459079f, -0.267501f, +0.336912f,
          -0.445881f, -0.297401f, +0.327227f,   -0.429857f, -0.323762f, +0.315467f,
          -0.412417f, -0.346717f, +0.302667f,   -0.394977f, -0.366395f, +0.289868f,
          -0.378953f, -0.382928f, +0.278108f,   -0.365755f, -0.396447f, +0.268423f,
          -0.356799f, -0.407084f, +0.261851f,   -0.353500f, -0.414968f, +0.259429f,
          -0.514625f, -0.233932f, +0.267746f,   -0.504778f, -0.267501f, +0.262623f,
          -0.490267f, -0.297401f, +0.255073f,   -0.472646f, -0.323762f, +0.245906f,
          -0.453471f, -0.346717f, +0.235929f,   -0.434296f, -0.366395f, +0.225952f,
          -0.416676f, -0.382928f, +0.216785f,   -0.402164f, -0.396447f, +0.209235f,
          -0.392317f, -0.407084f, +0.204112f,   -0.388690f, -0.414968f, +0.202225f,
          -0.549279f, -0.233932f, +0.184511f,   -0.538769f, -0.267501f, +0.180981f,
          -0.523280f, -0.297401f, +0.175779f,   -0.504474f, -0.323762f, +0.169461f,
          -0.484007f, -0.346717f, +0.162586f,   -0.463540f, -0.366395f, +0.155711f,
          -0.444734f, -0.382928f, +0.149393f,   -0.429245f, -0.396447f, +0.144190f,
          -0.418735f, -0.407084f, +0.140660f,   -0.414863f, -0.414968f, +0.139359f,
          -0.570885f, -0.233932f, +0.094893f,   -0.559962f, -0.267501f, +0.093077f,
          -0.543865f, -0.297401f, +0.090402f,   -0.524318f, -0.323762f, +0.087152f,
          -0.503046f, -0.346717f, +0.083617f,   -0.481774f, -0.366395f, +0.080081f,
          -0.462227f, -0.382928f, +0.076832f,   -0.446130f, -0.396447f, +0.074156f,
          -0.435207f, -0.407084f, +0.072340f,   -0.431182f, -0.414968f, +0.071671f,
          -0.578334f, -0.233932f, -0.000000f,   -0.567268f, -0.267501f, -0.000000f,
          -0.550961f, -0.297401f, -0.000000f,   -0.531159f, -0.323762f, -0.000000f,
          -0.509610f, -0.346717f, -0.000000f,   -0.488061f, -0.366395f, -0.000000f,
          -0.468259f, -0.382928f, -0.000000f,   -0.451951f, -0.396447f, -0.000000f,
          -0.440885f, -0.407084f, -0.000000f,   -0.436808f, -0.414968f, -0.000000f,
          -0.570885f, -0.233932f, -0.094893f,   -0.559962f, -0.267501f, -0.093077f,
          -0.543865f, -0.297401f, -0.090402f,   -0.524318f, -0.323762f, -0.087153f,
          -0.503046f, -0.346717f, -0.083617f,   -0.481774f, -0.366395f, -0.080081f,
          -0.462227f, -0.382928f, -0.076832f,   -0.446130f, -0.396447f, -0.074156f,
          -0.435207f, -0.407084f, -0.072340f,   -0.431182f, -0.414968f, -0.071672f,
          -0.549279f, -0.233932f, -0.184511f,   -0.538769f, -0.267501f, -0.180982f,
          -0.523280f, -0.297401f, -0.175779f,   -0.504474f, -0.323762f, -0.169461f,
          -0.484007f, -0.346717f, -0.162586f,   -0.463540f, -0.366395f, -0.155711f,
          -0.444734f, -0.382928f, -0.149393f,   -0.429245f, -0.396447f, -0.144191f,
          -0.418735f, -0.407084f, -0.140660f,   -0.414863f, -0.414968f, -0.139359f,
          -0.514625f, -0.233932f, -0.267746f,   -0.504778f, -0.267501f, -0.262623f,
          -0.490267f, -0.297401f, -0.255073f,   -0.472646f, -0.323762f, -0.245906f,
          -0.453471f, -0.346717f, -0.235929f,   -0.434296f, -0.366395f, -0.225952f,
          -0.416676f, -0.382928f, -0.216785f,   -0.402164f, -0.396447f, -0.209235f,
          -0.392317f, -0.407084f, -0.204112f,   -0.388690f, -0.414968f, -0.202225f,
          -0.468034f, -0.233932f, -0.343484f,   -0.459079f, -0.267501f, -0.336912f,
          -0.445881f, -0.297401f, -0.327227f,   -0.429857f, -0.323762f, -0.315467f,
          -0.412417f, -0.346717f, -0.302667f,   -0.394977f, -0.366395f, -0.289868f,
          -0.378953f, -0.382928f, -0.278108f,   -0.365755f, -0.396447f, -0.268423f,
          -0.356799f, -0.407084f, -0.261851f,   -0.353500f, -0.414968f, -0.259429f,
          -0.410617f, -0.233932f, -0.410617f,   -0.402761f, -0.267501f, -0.402761f,
          -0.391182f, -0.297401f, -0.391182f,   -0.377123f, -0.323762f, -0.377123f,
          -0.361823f, -0.346717f, -0.361823f,   -0.346523f, -0.366395f, -0.346523f,
          -0.332464f, -0.382928f, -0.332464f,   -0.320885f, -0.396447f, -0.320885f,
          -0.313029f, -0.407084f, -0.313029f,   -0.310134f, -0.414968f, -0.310134f,
          -0.343484f, -0.233932f, -0.468035f,   -0.336912f, -0.267501f, -0.459079f,
          -0.327227f, -0.297401f, -0.445881f,   -0.315467f, -0.323762f, -0.429857f,
          -0.302667f, -0.346717f, -0.412417f,   -0.289868f, -0.366395f, -0.394977f,
          -0.278108f, -0.382928f, -0.378953f,   -0.268423f, -0.396447f, -0.365755f,
          -0.261851f, -0.407084f, -0.356799f,   -0.259429f, -0.414968f, -0.353500f,
          -0.267746f, -0.233932f, -0.514625f,   -0.262623f, -0.267501f, -0.504778f,
          -0.255073f, -0.297401f, -0.490267f,   -0.245906f, -0.323762f, -0.472646f,
          -0.235929f, -0.346717f, -0.453471f,   -0.225952f, -0.366395f, -0.434296f,
          -0.216785f, -0.382928f, -0.416676f,   -0.209235f, -0.396447f, -0.402164f,
          -0.204112f, -0.407084f, -0.392317f,   -0.202225f, -0.414968f, -0.388690f,
          -0.184511f, -0.233932f, -0.549279f,   -0.180981f, -0.267501f, -0.538769f,
          -0.175779f, -0.297401f, -0.523280f,   -0.169461f, -0.323762f, -0.504474f,
          -0.162586f, -0.346717f, -0.484007f,   -0.155711f, -0.366395f, -0.463540f,
          -0.149393f, -0.382928f, -0.444734f,   -0.144190f, -0.396447f, -0.429245f,
          -0.140660f, -0.407084f, -0.418735f,   -0.139359f, -0.414968f, -0.414863f,
          -0.094893f, -0.233932f, -0.570885f,   -0.093077f, -0.267501f, -0.559962f,
          -0.090402f, -0.297401f, -0.543865f,   -0.087152f, -0.323762f, -0.524318f,
          -0.083617f, -0.346717f, -0.503046f,   -0.080081f, -0.366395f, -0.481774f,
          -0.076832f, -0.382928f, -0.462227f,   -0.074156f, -0.396447f, -0.446130f,
          -0.072340f, -0.407084f, -0.435207f,   -0.071671f, -0.414968f, -0.431182f,
          +0.000000f, -0.233932f, -0.578334f,   +0.000000f, -0.267501f, -0.567268f,
          +0.000000f, -0.297401f, -0.550961f,   +0.000000f, -0.323762f, -0.531159f,
          +0.000000f, -0.346717f, -0.509610f,   +0.000000f, -0.366395f, -0.488061f,
          +0.000000f, -0.382928f, -0.468259f,   +0.000000f, -0.396447f, -0.451951f,
          +0.000000f, -0.407084f, -0.440885f,   +0.000000f, -0.414968f, -0.436808f,
          +0.094893f, -0.233932f, -0.570885f,   +0.093077f, -0.267501f, -0.559962f,
          +0.090402f, -0.297401f, -0.543865f,   +0.087153f, -0.323762f, -0.524318f,
          +0.083617f, -0.346717f, -0.503046f,   +0.080081f, -0.366395f, -0.481774f,
          +0.076832f, -0.382928f, -0.462227f,   +0.074156f, -0.396447f, -0.446130f,
          +0.072340f, -0.407084f, -0.435207f,   +0.071672f, -0.414968f, -0.431182f,
          +0.184511f, -0.233932f, -0.549279f,   +0.180981f, -0.267501f, -0.538769f,
          +0.175779f, -0.297401f, -0.523280f,   +0.169461f, -0.323762f, -0.504474f,
          +0.162586f, -0.346717f, -0.484007f,   +0.155711f, -0.366395f, -0.463540f,
          +0.149393f, -0.382928f, -0.444734f,   +0.144191f, -0.396447f, -0.429245f,
          +0.140660f, -0.407084f, -0.418735f,   +0.139359f, -0.414968f, -0.414863f,
          +0.267746f, -0.233932f, -0.514625f,   +0.262623f, -0.267501f, -0.504778f,
          +0.255073f, -0.297401f, -0.490267f,   +0.245906f, -0.323762f, -0.472646f,
          +0.235929f, -0.346717f, -0.453471f,   +0.225952f, -0.366395f, -0.434296f,
          +0.216785f, -0.382928f, -0.416676f,   +0.209235f, -0.396447f, -0.402164f,
          +0.204112f, -0.407084f, -0.392317f,   +0.202225f, -0.414968f, -0.388690f,
          +0.343484f, -0.233932f, -0.468034f,   +0.336912f, -0.267501f, -0.459079f,
          +0.327227f, -0.297401f, -0.445881f,   +0.315467f, -0.323762f, -0.429857f,
          +0.302667f, -0.346717f, -0.412417f,   +0.289868f, -0.366395f, -0.394977f,
          +0.278108f, -0.382928f, -0.378953f,   +0.268423f, -0.396447f, -0.365755f,
          +0.261851f, -0.407084f, -0.356799f,   +0.259429f, -0.414968f, -0.353500f,
          +0.410617f, -0.233932f, -0.410617f,   +0.402761f, -0.267501f, -0.402761f,
          +0.391182f, -0.297401f, -0.391182f,   +0.377123f, -0.323762f, -0.377123f,
          +0.361823f, -0.346717f, -0.361823f,   +0.346523f, -0.366395f, -0.346523f,
          +0.332464f, -0.382928f, -0.332464f,   +0.320885f, -0.396447f, -0.320885f,
          +0.313029f, -0.407084f, -0.313029f,   +0.310134f, -0.414968f, -0.310134f,
          +0.468035f, -0.233932f, -0.343484f,   +0.459079f, -0.267501f, -0.336912f,
          +0.445881f, -0.297401f, -0.327227f,   +0.429857f, -0.323762f, -0.315467f,
          +0.412417f, -0.346717f, -0.302667f,   +0.394977f, -0.366395f, -0.289868f,
          +0.378953f, -0.382928f, -0.278108f,   +0.365755f, -0.396447f, -0.268423f,
          +0.356799f, -0.407084f, -0.261851f,   +0.353500f, -0.414968f, -0.259429f,
          +0.514625f, -0.233932f, -0.267746f,   +0.504778f, -0.267501f, -0.262623f,
          +0.490267f, -0.297401f, -0.255073f,   +0.472646f, -0.323762f, -0.245906f,
          +0.453471f, -0.346717f, -0.235929f,   +0.434296f, -0.366395f, -0.225952f,
          +0.416676f, -0.382928f, -0.216785f,   +0.402164f, -0.396447f, -0.209235f,
          +0.392317f, -0.407084f, -0.204112f,   +0.388690f, -0.414968f, -0.202225f,
          +0.549279f, -0.233932f, -0.184511f,   +0.538769f, -0.267501f, -0.180981f,
          +0.523280f, -0.297401f, -0.175779f,   +0.504474f, -0.323762f, -0.169461f,
          +0.484007f, -0.346717f, -0.162586f,   +0.463540f, -0.366395f, -0.155711f,
          +0.444734f, -0.382928f, -0.149393f,   +0.429245f, -0.396447f, -0.144190f,
          +0.418735f, -0.407084f, -0.140660f,   +0.414863f, -0.414968f, -0.139359f,
          +0.570885f, -0.233932f, -0.094893f,   +0.559962f, -0.267501f, -0.093077f,
          +0.543865f, -0.297401f, -0.090402f,   +0.524318f, -0.323762f, -0.087152f,
          +0.503046f, -0.346717f, -0.083617f,   +0.481774f, -0.366395f, -0.080081f,
          +0.462227f, -0.382928f, -0.076832f,   +0.446130f, -0.396447f, -0.074156f,
          +0.435207f, -0.407084f, -0.072340f,   +0.431182f, -0.414968f, -0.071671f,
          +0.430169f, -0.421498f, +0.071503f,   +0.435782f, -0.421498f, -0.000000f,
          +0.425663f, -0.427898f, +0.070754f,   +0.431217f, -0.427898f, -0.000000f,
          +0.415465f, -0.434035f, +0.069059f,   +0.420886f, -0.434035f, -0.000000f,
          +0.397378f, -0.439779f, +0.066052f,   +0.402563f, -0.439779f, -0.000000f,
          +0.369200f, -0.444998f, +0.061369f,   +0.374017f, -0.444998f, -0.000000f,
          +0.328734f, -0.449563f, +0.054642f,   +0.333023f, -0.449563f, -0.000000f,
          +0.273779f, -0.453342f, +0.045508f,   +0.277351f, -0.453342f, -0.000000f,
          +0.202139f, -0.456203f, +0.033600f,   +0.204776f, -0.456203f, -0.000000f,
          +0.111611f, -0.458016f, +0.018552f,   +0.113068f, -0.458016f, -0.000000f,
          +0.000000f, -0.458649f, +0.000000f,   +0.413888f, -0.421498f, +0.139032f,
          +0.409553f, -0.427898f, +0.137576f,   +0.399741f, -0.434035f, +0.134280f,
          +0.382338f, -0.439779f, +0.128433f,   +0.355227f, -0.444998f, +0.119327f,
          +0.316291f, -0.449563f, +0.106248f,   +0.263418f, -0.453342f, +0.088486f,
          +0.194488f, -0.456203f, +0.065332f,   +0.107387f, -0.458016f, +0.036073f,
          +0.387776f, -0.421498f, +0.201750f,   +0.383715f, -0.427898f, +0.199637f,
          +0.374522f, -0.434035f, +0.194854f,   +0.358217f, -0.439779f, +0.186370f,
          +0.332815f, -0.444998f, +0.173156f,   +0.296337f, -0.449563f, +0.154176f,
          +0.246798f, -0.453342f, +0.128403f,   +0.182218f, -0.456203f, +0.094803f,
          +0.100612f, -0.458015f, +0.052346f,   +0.352670f, -0.421498f, +0.258820f,
          +0.348975f, -0.427898f, +0.256108f,   +0.340615f, -0.434035f, +0.249973f,
          +0.325786f, -0.439779f, +0.239090f,   +0.302685f, -0.444998f, +0.222136f,
          +0.269508f, -0.449563f, +0.197789f,   +0.224455f, -0.453342f, +0.164725f,
          +0.165720f, -0.456203f, +0.121620f,   +0.091503f, -0.458015f, +0.067153f,
          +0.309405f, -0.421498f, +0.309405f,   +0.306164f, -0.427898f, +0.306164f,
          +0.298829f, -0.434035f, +0.298829f,   +0.285819f, -0.439779f, +0.285819f,
          +0.265553f, -0.444998f, +0.265553f,   +0.236446f, -0.449563f, +0.236446f,
          +0.196919f, -0.453342f, +0.196919f,   +0.145391f, -0.456203f, +0.145391f,
          +0.080278f, -0.458015f, +0.080278f,   +0.258820f, -0.421498f, +0.352670f,
          +0.256108f, -0.427898f, +0.348975f,   +0.249973f, -0.434035f, +0.340615f,
          +0.239090f, -0.439779f, +0.325786f,   +0.222136f, -0.444998f, +0.302685f,
          +0.197789f, -0.449563f, +0.269508f,   +0.164725f, -0.453342f, +0.224455f,
          +0.121620f, -0.456203f, +0.165720f,   +0.067153f, -0.458015f, +0.091503f,
          +0.201750f, -0.421498f, +0.387776f,   +0.199637f, -0.427898f, +0.383715f,
          +0.194854f, -0.434035f, +0.374522f,   +0.186370f, -0.439779f, +0.358217f,
          +0.173156f, -0.444998f, +0.332815f,   +0.154176f, -0.449563f, +0.296337f,
          +0.128403f, -0.453342f, +0.246798f,   +0.094803f, -0.456203f, +0.182218f,
          +0.052346f, -0.458015f, +0.100612f,   +0.139032f, -0.421498f, +0.413888f,
          +0.137576f, -0.427898f, +0.409553f,   +0.134280f, -0.434035f, +0.399741f,
          +0.128433f, -0.439779f, +0.382338f,   +0.119327f, -0.444998f, +0.355227f,
          +0.106248f, -0.449563f, +0.316291f,   +0.088486f, -0.453342f, +0.263418f,
          +0.065332f, -0.456203f, +0.194488f,   +0.036073f, -0.458015f, +0.107387f,
          +0.071503f, -0.421498f, +0.430169f,   +0.070754f, -0.427898f, +0.425663f,
          +0.069059f, -0.434035f, +0.415465f,   +0.066052f, -0.439779f, +0.397378f,
          +0.061369f, -0.444998f, +0.369200f,   +0.054642f, -0.449563f, +0.328734f,
          +0.045508f, -0.453342f, +0.273779f,   +0.033600f, -0.456203f, +0.202139f,
          +0.018552f, -0.458015f, +0.111611f,   -0.000000f, -0.421498f, +0.435782f,
          -0.000000f, -0.427898f, +0.431217f,   -0.000000f, -0.434035f, +0.420886f,
          -0.000000f, -0.439779f, +0.402563f,   -0.000000f, -0.444998f, +0.374017f,
          -0.000000f, -0.449563f, +0.333023f,   -0.000000f, -0.453342f, +0.277351f,
          -0.000000f, -0.456203f, +0.204776f,   -0.000000f, -0.458015f, +0.113068f,
          -0.071503f, -0.421498f, +0.430169f,   -0.070754f, -0.427898f, +0.425663f,
          -0.069059f, -0.434035f, +0.415465f,   -0.066052f, -0.439779f, +0.397378f,
          -0.061369f, -0.444998f, +0.369200f,   -0.054642f, -0.449563f, +0.328734f,
          -0.045508f, -0.453342f, +0.273779f,   -0.033600f, -0.456203f, +0.202139f,
          -0.018552f, -0.458015f, +0.111611f,   -0.139032f, -0.421498f, +0.413888f,
          -0.137576f, -0.427898f, +0.409553f,   -0.134280f, -0.434035f, +0.399741f,
          -0.128433f, -0.439779f, +0.382338f,   -0.119327f, -0.444998f, +0.355227f,
          -0.106248f, -0.449563f, +0.316291f,   -0.088486f, -0.453342f, +0.263418f,
          -0.065332f, -0.456203f, +0.194488f,   -0.036073f, -0.458015f, +0.107387f,
          -0.201750f, -0.421498f, +0.387776f,   -0.199637f, -0.427898f, +0.383715f,
          -0.194854f, -0.434035f, +0.374522f,   -0.186370f, -0.439779f, +0.358217f,
          -0.173156f, -0.444998f, +0.332815f,   -0.154176f, -0.449563f, +0.296337f,
          -0.128403f, -0.453342f, +0.246798f,   -0.094803f, -0.456203f, +0.182218f,
          -0.052346f, -0.458015f, +0.100612f,   -0.258820f, -0.421498f, +0.352670f,
          -0.256108f, -0.427898f, +0.348975f,   -0.249973f, -0.434035f, +0.340615f,
          -0.239090f, -0.439779f, +0.325786f,   -0.222136f, -0.444998f, +0.302685f,
          -0.197789f, -0.449563f, +0.269508f,   -0.164725f, -0.453342f, +0.224455f,
          -0.121620f, -0.456203f, +0.165720f,   -0.067153f, -0.458015f, +0.091503f,
          -0.309405f, -0.421498f, +0.309405f,   -0.306164f, -0.427898f, +0.306164f,
          -0.298829f, -0.434035f, +0.298829f,   -0.285819f, -0.439779f, +0.285819f,
          -0.265553f, -0.444998f, +0.265553f,   -0.236446f, -0.449563f, +0.236446f,
          -0.196919f, -0.453342f, +0.196919f,   -0.145391f, -0.456203f, +0.145391f,
          -0.080278f, -0.458015f, +0.080278f,   -0.352670f, -0.421498f, +0.258820f,
          -0.348975f, -0.427898f, +0.256108f,   -0.340615f, -0.434035f, +0.249973f,
          -0.325786f, -0.439779f, +0.239090f,   -0.302685f, -0.444998f, +0.222136f,
          -0.269508f, -0.449563f, +0.197789f,   -0.224455f, -0.453342f, +0.164725f,
          -0.165720f, -0.456203f, +0.121620f,   -0.091503f, -0.458015f, +0.067153f,
          -0.387776f, -0.421498f, +0.201750f,   -0.383715f, -0.427898f, +0.199637f,
          -0.374522f, -0.434035f, +0.194854f,   -0.358217f, -0.439779f, +0.186370f,
          -0.332815f, -0.444998f, +0.173156f,   -0.296337f, -0.449563f, +0.154176f,
          -0.246798f, -0.453342f, +0.128403f,   -0.182218f, -0.456203f, +0.094803f,
          -0.100612f, -0.458015f, +0.052346f,   -0.413888f, -0.421498f, +0.139032f,
          -0.409553f, -0.427898f, +0.137576f,   -0.399741f, -0.434035f, +0.134280f,
          -0.382338f, -0.439779f, +0.128433f,   -0.355227f, -0.444998f, +0.119327f,
          -0.316291f, -0.449563f, +0.106248f,   -0.263418f, -0.453342f, +0.088486f,
          -0.194488f, -0.456203f, +0.065332f,   -0.107387f, -0.458016f, +0.036073f,
          -0.430169f, -0.421498f, +0.071503f,   -0.425663f, -0.427898f, +0.070754f,
          -0.415465f, -0.434035f, +0.069059f,   -0.397378f, -0.439779f, +0.066052f,
          -0.369200f, -0.444998f, +0.061369f,   -0.328734f, -0.449563f, +0.054642f,
          -0.273779f, -0.453342f, +0.045508f,   -0.202139f, -0.456203f, +0.033600f,
          -0.111611f, -0.458016f, +0.018552f,   -0.435782f, -0.421498f, -0.000000f,
          -0.431217f, -0.427898f, -0.000000f,   -0.420886f, -0.434035f, -0.000000f,
          -0.402563f, -0.439779f, -0.000000f,   -0.374017f, -0.444998f, -0.000000f,
          -0.333023f, -0.449563f, -0.000000f,   -0.277351f, -0.453342f, -0.000000f,
          -0.204776f, -0.456203f, -0.000000f,   -0.113068f, -0.458016f, -0.000000f,
          -0.430169f, -0.421498f, -0.071503f,   -0.425663f, -0.427898f, -0.070754f,
          -0.415465f, -0.434035f, -0.069059f,   -0.397378f, -0.439779f, -0.066052f,
          -0.369200f, -0.444999f, -0.061369f,   -0.328734f, -0.449563f, -0.054642f,
          -0.273779f, -0.453342f, -0.045508f,   -0.202139f, -0.456203f, -0.033600f,
          -0.111611f, -0.458016f, -0.018552f,   -0.413888f, -0.421498f, -0.139032f,
          -0.409553f, -0.427898f, -0.137576f,   -0.399741f, -0.434035f, -0.134280f,
          -0.382338f, -0.439779f, -0.128433f,   -0.355227f, -0.444999f, -0.119327f,
          -0.316291f, -0.449563f, -0.106248f,   -0.263418f, -0.453342f, -0.088486f,
          -0.194488f, -0.456203f, -0.065332f,   -0.107387f, -0.458016f, -0.036073f,
          -0.387776f, -0.421498f, -0.201750f,   -0.383715f, -0.427898f, -0.199637f,
          -0.374522f, -0.434035f, -0.194854f,   -0.358217f, -0.439779f, -0.186370f,
          -0.332815f, -0.444999f, -0.173156f,   -0.296337f, -0.449563f, -0.154176f,
          -0.246798f, -0.453342f, -0.128403f,   -0.182218f, -0.456203f, -0.094803f,
          -0.100612f, -0.458016f, -0.052346f,   -0.352670f, -0.421498f, -0.258820f,
          -0.348975f, -0.427898f, -0.256108f,   -0.340615f, -0.434035f, -0.249973f,
          -0.325786f, -0.439779f, -0.239090f,   -0.302685f, -0.444999f, -0.222136f,
          -0.269508f, -0.449563f, -0.197789f,   -0.224455f, -0.453342f, -0.164725f,
          -0.165720f, -0.456203f, -0.121620f,   -0.091503f, -0.458016f, -0.067153f,
          -0.309405f, -0.421498f, -0.309405f,   -0.306164f, -0.427898f, -0.306164f,
          -0.298829f, -0.434035f, -0.298829f,   -0.285819f, -0.439779f, -0.285819f,
          -0.265553f, -0.444999f, -0.265553f,   -0.236446f, -0.449563f, -0.236446f,
          -0.196919f, -0.453342f, -0.196919f,   -0.145391f, -0.456203f, -0.145391f,
          -0.080278f, -0.458016f, -0.080278f,   -0.258820f, -0.421498f, -0.352670f,
          -0.256108f, -0.427898f, -0.348975f,   -0.249973f, -0.434035f, -0.340615f,
          -0.239090f, -0.439779f, -0.325786f,   -0.222136f, -0.444999f, -0.302685f,
          -0.197789f, -0.449563f, -0.269508f,   -0.164725f, -0.453342f, -0.224455f,
          -0.121620f, -0.456203f, -0.165720f,   -0.067153f, -0.458016f, -0.091503f,
          -0.201750f, -0.421498f, -0.387776f,   -0.199637f, -0.427898f, -0.383715f,
          -0.194854f, -0.434035f, -0.374522f,   -0.186370f, -0.439779f, -0.358217f,
          -0.173156f, -0.444999f, -0.332815f,   -0.154176f, -0.449563f, -0.296337f,
          -0.128403f, -0.453342f, -0.246798f,   -0.094803f, -0.456203f, -0.182218f,
          -0.052346f, -0.458016f, -0.100612f,   -0.139032f, -0.421498f, -0.413888f,
          -0.137576f, -0.427898f, -0.409553f,   -0.134280f, -0.434035f, -0.399741f,
          -0.128433f, -0.439779f, -0.382338f,   -0.119327f, -0.444999f, -0.355227f,
          -0.106248f, -0.449563f, -0.316291f,   -0.088486f, -0.453342f, -0.263418f,
          -0.065332f, -0.456203f, -0.194488f,   -0.036073f, -0.458016f, -0.107387f,
          -0.071503f, -0.421498f, -0.430169f,   -0.070754f, -0.427898f, -0.425663f,
          -0.069059f, -0.434035f, -0.415465f,   -0.066052f, -0.439779f, -0.397378f,
          -0.061369f, -0.444999f, -0.369200f,   -0.054642f, -0.449563f, -0.328734f,
          -0.045508f, -0.453342f, -0.273779f,   -0.033600f, -0.456203f, -0.202139f,
          -0.018552f, -0.458016f, -0.111611f,   +0.000000f, -0.421498f, -0.435782f,
          +0.000000f, -0.427898f, -0.431217f,   +0.000000f, -0.434035f, -0.420886f,
          +0.000000f, -0.439779f, -0.402563f,   +0.000000f, -0.444999f, -0.374017f,
          +0.000000f, -0.449563f, -0.333023f,   +0.000000f, -0.453342f, -0.277351f,
          +0.000000f, -0.456203f, -0.204776f,   +0.000000f, -0.458016f, -0.113068f,
          +0.071503f, -0.421498f, -0.430169f,   +0.070754f, -0.427898f, -0.425663f,
          +0.069059f, -0.434035f, -0.415465f,   +0.066052f, -0.439779f, -0.397378f,
          +0.061369f, -0.444999f, -0.369200f,   +0.054642f, -0.449563f, -0.328734f,
          +0.045508f, -0.453342f, -0.273779f,   +0.033600f, -0.456203f, -0.202139f,
          +0.018552f, -0.458016f, -0.111611f,   +0.139032f, -0.421498f, -0.413888f,
          +0.137576f, -0.427898f, -0.409553f,   +0.134280f, -0.434035f, -0.399741f,
          +0.128433f, -0.439779f, -0.382338f,   +0.119327f, -0.444999f, -0.355227f,
          +0.106248f, -0.449563f, -0.316291f,   +0.088486f, -0.453342f, -0.263418f,
          +0.065332f, -0.456203f, -0.194488f,   +0.036073f, -0.458016f, -0.107387f,
          +0.201750f, -0.421498f, -0.387776f,   +0.199637f, -0.427898f, -0.383715f,
          +0.194854f, -0.434035f, -0.374522f,   +0.186370f, -0.439779f, -0.358217f,
          +0.173156f, -0.444999f, -0.332815f,   +0.154176f, -0.449563f, -0.296337f,
          +0.128403f, -0.453342f, -0.246798f,   +0.094803f, -0.456203f, -0.182218f,
          +0.052346f, -0.458016f, -0.100612f,   +0.258820f, -0.421498f, -0.352670f,
          +0.256108f, -0.427898f, -0.348975f,   +0.249973f, -0.434035f, -0.340615f,
          +0.239090f, -0.439779f, -0.325786f,   +0.222136f, -0.444999f, -0.302685f,
          +0.197789f, -0.449563f, -0.269508f,   +0.164725f, -0.453342f, -0.224455f,
          +0.121620f, -0.456203f, -0.165720f,   +0.067153f, -0.458016f, -0.091503f,
          +0.309405f, -0.421498f, -0.309405f,   +0.306164f, -0.427898f, -0.306164f,
          +0.298829f, -0.434035f, -0.298829f,   +0.285819f, -0.439779f, -0.285819f,
          +0.265553f, -0.444999f, -0.265553f,   +0.236446f, -0.449563f, -0.236446f,
          +0.196919f, -0.453342f, -0.196919f,   +0.145391f, -0.456203f, -0.145391f,
          +0.080278f, -0.458016f, -0.080278f,   +0.352670f, -0.421498f, -0.258820f,
          +0.348975f, -0.427898f, -0.256108f,   +0.340615f, -0.434035f, -0.249973f,
          +0.325786f, -0.439779f, -0.239090f,   +0.302685f, -0.444999f, -0.222136f,
          +0.269508f, -0.449563f, -0.197789f,   +0.224455f, -0.453342f, -0.164725f,
          +0.165720f, -0.456203f, -0.121620f,   +0.091503f, -0.458016f, -0.067153f,
          +0.387776f, -0.421498f, -0.201750f,   +0.383715f, -0.427898f, -0.199637f,
          +0.374522f, -0.434035f, -0.194854f,   +0.358217f, -0.439779f, -0.186370f,
          +0.332815f, -0.444999f, -0.173156f,   +0.296337f, -0.449563f, -0.154176f,
          +0.246798f, -0.453342f, -0.128403f,   +0.182218f, -0.456203f, -0.094803f,
          +0.100612f, -0.458016f, -0.052346f,   +0.413888f, -0.421498f, -0.139032f,
          +0.409553f, -0.427898f, -0.137576f,   +0.399741f, -0.434035f, -0.134280f,
          +0.382338f, -0.439779f, -0.128433f,   +0.355227f, -0.444999f, -0.119327f,
          +0.316291f, -0.449563f, -0.106248f,   +0.263418f, -0.453342f, -0.088486f,
          +0.194488f, -0.456203f, -0.065332f,   +0.107387f, -0.458016f, -0.036073f,
          +0.430169f, -0.421498f, -0.071503f,   +0.425663f, -0.427898f, -0.070754f,
          +0.415465f, -0.434035f, -0.069059f,   +0.397378f, -0.439779f, -0.066052f,
          +0.369200f, -0.444999f, -0.061369f,   +0.328734f, -0.449563f, -0.054642f,
          +0.273779f, -0.453342f, -0.045508f,   +0.202139f, -0.456203f, -0.033600f,
          +0.111611f, -0.458016f, -0.018552f,   -0.465929f, +0.131043f, -0.000000f,
          -0.465114f, +0.132877f, +0.023588f,   -0.524303f, +0.132810f, +0.023588f,
          -0.524432f, +0.130977f, -0.000000f,   -0.577982f, +0.132338f, +0.023588f,
          -0.577519f, +0.130518f, -0.000000f,   -0.625983f, +0.131059f, +0.023588f,
          -0.625015f, +0.129273f, -0.000000f,   -0.668134f, +0.128566f, +0.023588f,
          -0.666744f, +0.126849f, -0.000000f,   -0.704266f, +0.124458f, +0.023588f,
          -0.702533f, +0.122853f, -0.000000f,   -0.734210f, +0.118328f, +0.023588f,
          -0.732207f, +0.116890f, -0.000000f,   -0.757795f, +0.109774f, +0.023588f,
          -0.755591f, +0.108568f, -0.000000f,   -0.774852f, +0.098391f, +0.023588f,
          -0.772510f, +0.097496f, -0.000000f,   -0.785210f, +0.083775f, +0.023588f,
          -0.782790f, +0.083278f, -0.000000f,   -0.788701f, +0.065521f, +0.023588f,
          -0.786255f, +0.065521f, -0.000000f,   -0.462900f, +0.137857f, +0.041934f,
          -0.523951f, +0.137784f, +0.041934f,   -0.579239f, +0.137278f, +0.041934f,
          -0.628609f, +0.135903f, +0.041934f,   -0.671905f, +0.133228f, +0.041934f,
          -0.708969f, +0.128815f, +0.041934f,   -0.739646f, +0.122232f, +0.041934f,
          -0.763778f, +0.113046f, +0.041934f,   -0.781208f, +0.100821f, +0.041934f,
          -0.791781f, +0.085124f, +0.041934f,   -0.795341f, +0.065521f, +0.041934f,
          -0.459639f, +0.145195f, +0.055038f,   -0.523432f, +0.145116f, +0.055038f,
          -0.581091f, +0.144558f, +0.055038f,   -0.632481f, +0.143044f, +0.055038f,
          -0.677463f, +0.140096f, +0.055038f,   -0.715900f, +0.135236f, +0.055038f,
          -0.747655f, +0.127986f, +0.055038f,   -0.772593f, +0.117867f, +0.055038f,
          -0.790575f, +0.104402f, +0.055038f,   -0.801465f, +0.087112f, +0.055038f,
          -0.805125f, +0.065521f, +0.055038f,   -0.455678f, +0.154106f, +0.062900f,
          -0.522803f, +0.154017f, +0.062900f,   -0.583341f, +0.153398f, +0.062900f,
          -0.637182f, +0.151715f, +0.062900f,   -0.684211f, +0.148437f, +0.062900f,
          -0.724316f, +0.143033f, +0.062900f,   -0.757383f, +0.134971f, +0.062900f,
          -0.783298f, +0.123722f, +0.062900f,   -0.801950f, +0.108750f, +0.062900f,
          -0.813223f, +0.089528f, +0.062900f,   -0.817006f, +0.065521f, +0.062900f,
          -0.451369f, +0.163803f, +0.065521f,   -0.522118f, +0.163705f, +0.065521f,
          -0.585789f, +0.163017f, +0.065521f,   -0.642298f, +0.161150f, +0.065521f,
          -0.691555f, +0.157513f, +0.065521f,   -0.733475f, +0.151518f, +0.065521f,
          -0.767967f, +0.142574f, +0.065521f,   -0.794948f, +0.130092f, +0.065521f,
          -0.814327f, +0.113483f, +0.065521f,   -0.826019f, +0.092156f, +0.065521f,
          -0.829936f, +0.065521f, +0.065521f,   -0.447059f, +0.173500f, +0.062900f,
          -0.521432f, +0.173392f, +0.062900f,   -0.588238f, +0.172636f, +0.062900f,
          -0.647413f, +0.170585f, +0.062900f,   -0.698899f, +0.166589f, +0.062900f,
          -0.742633f, +0.160002f, +0.062900f,   -0.778552f, +0.150177f, +0.062900f,
          -0.806598f, +0.136464f, +0.062900f,   -0.826705f, +0.118216f, +0.062900f,
          -0.838815f, +0.094784f, +0.062900f,   -0.842865f, +0.065521f, +0.062900f,
          -0.443098f, +0.182411f, +0.055038f,   -0.520803f, +0.182295f, +0.055038f,
          -0.590487f, +0.181476f, +0.055038f,   -0.652114f, +0.179256f, +0.055038f,
          -0.705647f, +0.174931f, +0.055038f,   -0.751048f, +0.167800f, +0.055038f,
          -0.788280f, +0.157162f, +0.055038f,   -0.817302f, +0.142318f, +0.055038f,
          -0.838079f, +0.122564f, +0.055038f,   -0.850573f, +0.097199f, +0.055038f,
          -0.854747f, +0.065521f, +0.055038f,   -0.439837f, +0.189750f, +0.041934f,
          -0.520283f, +0.189625f, +0.041934f,   -0.592339f, +0.188756f, +0.041934f,
          -0.655986f, +0.186395f, +0.041934f,   -0.711206f, +0.181798f, +0.041934f,
          -0.757979f, +0.174221f, +0.041934f,   -0.796289f, +0.162917f, +0.041934f,
          -0.826118f, +0.147139f, +0.041934f,   -0.847447f, +0.126144f, +0.041934f,
          -0.860257f, +0.099187f, +0.041934f,   -0.864531f, +0.065521f, +0.041934f,
          -0.437624f, +0.194729f, +0.023588f,   -0.519931f, +0.194600f, +0.023588f,
          -0.593596f, +0.193696f, +0.023588f,   -0.658613f, +0.191241f, +0.023588f,
          -0.714976f, +0.186460f, +0.023588f,   -0.762683f, +0.178578f, +0.023588f,
          -0.801725f, +0.166820f, +0.023588f,   -0.832100f, +0.150411f, +0.023588f,
          -0.853803f, +0.128574f, +0.023588f,   -0.866828f, +0.100537f, +0.023588f,
          -0.871171f, +0.065521f, +0.023588f,   -0.436808f, +0.196564f, -0.000000f,
          -0.519802f, +0.196433f, -0.000000f,   -0.594059f, +0.195515f, -0.000000f,
          -0.659581f, +0.193025f, -0.000000f,   -0.716366f, +0.188177f, -0.000000f,
          -0.764415f, +0.180184f, -0.000000f,   -0.803727f, +0.168259f, -0.000000f,
          -0.834304f, +0.151616f, -0.000000f,   -0.856144f, +0.129470f, -0.000000f,
          -0.869249f, +0.101033f, -0.000000f,   -0.873617f, +0.065521f, -0.000000f,
          -0.437624f, +0.194729f, -0.023588f,   -0.519931f, +0.194600f, -0.023588f,
          -0.593596f, +0.193696f, -0.023588f,   -0.658613f, +0.191241f, -0.023588f,
          -0.714976f, +0.186460f, -0.023588f,   -0.762683f, +0.178578f, -0.023588f,
          -0.801725f, +0.166820f, -0.023588f,   -0.832100f, +0.150411f, -0.023588f,
          -0.853803f, +0.128574f, -0.023588f,   -0.866828f, +0.100537f, -0.023588f,
          -0.871171f, +0.065521f, -0.023588f,   -0.439837f, +0.189750f, -0.041934f,
          -0.520283f, +0.189625f, -0.041934f,   -0.592339f, +0.188756f, -0.041934f,
          -0.655986f, +0.186395f, -0.041934f,   -0.711206f, +0.181798f, -0.041934f,
          -0.757979f, +0.174221f, -0.041934f,   -0.796289f, +0.162917f, -0.041934f,
          -0.826118f, +0.147139f, -0.041934f,   -0.847447f, +0.126144f, -0.041934f,
          -0.860257f, +0.099187f, -0.041934f,   -0.864531f, +0.065521f, -0.041934f,
          -0.443098f, +0.182411f, -0.055038f,   -0.520803f, +0.182295f, -0.055038f,
          -0.590487f, +0.181476f, -0.055038f,   -0.652114f, +0.179256f, -0.055038f,
          -0.705647f, +0.174931f, -0.055038f,   -0.751048f, +0.167800f, -0.055038f,
          -0.788280f, +0.157162f, -0.055038f,   -0.817302f, +0.142318f, -0.055038f,
          -0.838079f, +0.122564f, -0.055038f,   -0.850573f, +0.097199f, -0.055038f,
          -0.854747f, +0.065521f, -0.055038f,   -0.447059f, +0.173500f, -0.062900f,
          -0.521432f, +0.173392f, -0.062900f,   -0.588238f, +0.172636f, -0.062900f,
          -0.647413f, +0.170585f, -0.062900f,   -0.698899f, +0.166589f, -0.062900f,
          -0.742633f, +0.160002f, -0.062900f,   -0.778552f, +0.150177f, -0.062900f,
          -0.806598f, +0.136464f, -0.062900f,   -0.826705f, +0.118216f, -0.062900f,
          -0.838815f, +0.094784f, -0.062900f,   -0.842865f, +0.065521f, -0.062900f,
          -0.451369f, +0.163803f, -0.065521f,   -0.522118f, +0.163705f, -0.065521f,
          -0.585789f, +0.163017f, -0.065521f,   -0.642298f, +0.161150f, -0.065521f,
          -0.691555f, +0.157513f, -0.065521f,   -0.733475f, +0.151518f, -0.065521f,
          -0.767967f, +0.142574f, -0.065521f,   -0.794948f, +0.130092f, -0.065521f,
          -0.814327f, +0.113483f, -0.065521f,   -0.826019f, +0.092156f, -0.065521f,
          -0.829936f, +0.065521f, -0.065521f,   -0.455678f, +0.154106f, -0.062900f,
          -0.522803f, +0.154017f, -0.062900f,   -0.583341f, +0.153398f, -0.062900f,
          -0.637182f, +0.151715f, -0.062900f,   -0.684211f, +0.148437f, -0.062900f,
          -0.724316f, +0.143033f, -0.062900f,   -0.757383f, +0.134971f, -0.062900f,
          -0.783298f, +0.123722f, -0.062900f,   -0.801950f, +0.108750f, -0.062900f,
          -0.813223f, +0.089528f, -0.062900f,   -0.817006f, +0.065521f, -0.062900f,
          -0.459639f, +0.145195f, -0.055038f,   -0.523432f, +0.145116f, -0.055038f,
          -0.581091f, +0.144558f, -0.055038f,   -0.632481f, +0.143044f, -0.055038f,
          -0.677463f, +0.140096f, -0.055038f,   -0.715900f, +0.135236f, -0.055038f,
          -0.747655f, +0.127986f, -0.055038f,   -0.772593f, +0.117867f, -0.055038f,
          -0.790575f, +0.104402f, -0.055038f,   -0.801465f, +0.087112f, -0.055038f,
          -0.805125f, +0.065521f, -0.055038f,   -0.462900f, +0.137857f, -0.041934f,
          -0.523951f, +0.137784f, -0.041934f,   -0.579239f, +0.137278f, -0.041934f,
          -0.628609f, +0.135903f, -0.041934f,   -0.671905f, +0.133228f, -0.041934f,
          -0.708969f, +0.128815f, -0.041934f,   -0.739646f, +0.122232f, -0.041934f,
          -0.763778f, +0.113046f, -0.041934f,   -0.781208f, +0.100821f, -0.041934f,
          -0.791781f, +0.085124f, -0.041934f,   -0.795341f, +0.065521f, -0.041934f,
          -0.465114f, +0.132877f, -0.023588f,   -0.524303f, +0.132810f, -0.023588f,
          -0.577982f, +0.132338f, -0.023588f,   -0.625983f, +0.131059f, -0.023588f,
          -0.668134f, +0.128566f, -0.023588f,   -0.704266f, +0.124458f, -0.023588f,
          -0.734210f, +0.118328f, -0.023588f,   -0.757795f, +0.109774f, -0.023588f,
          -0.774852f, +0.098391f, -0.023588f,   -0.785210f, +0.083775f, -0.023588f,
          -0.788701f, +0.065521f, -0.023588f,   -0.786889f, +0.043541f, +0.023588f,
          -0.784479f, +0.044030f, -0.000000f,   -0.781336f, +0.018523f, +0.023588f,
          -0.779033f, +0.019394f, -0.000000f,   -0.771871f, -0.008764f, +0.023588f,
          -0.769744f, -0.007600f, -0.000000f,   -0.758320f, -0.037557f, +0.023588f,
          -0.756436f, -0.036168f, -0.000000f,   -0.740514f, -0.067088f, +0.023588f,
          -0.738934f, -0.065521f, -0.000000f,   -0.718278f, -0.096592f, +0.023588f,
          -0.717065f, -0.094875f, -0.000000f,   -0.691441f, -0.125302f, +0.023588f,
          -0.690652f, -0.123442f, -0.000000f,   -0.659829f, -0.152452f, +0.023588f,
          -0.659522f, -0.150437f, -0.000000f,   -0.623272f, -0.177277f, +0.023588f,
          -0.623500f, -0.175073f, -0.000000f,   -0.581596f, -0.199010f, +0.023588f,
          -0.582411f, -0.196564f, -0.000000f,   -0.793429f, +0.042212f, +0.041934f,
          -0.787585f, +0.016160f, +0.041934f,   -0.777644f, -0.011924f, +0.041934f,
          -0.763437f, -0.041328f, +0.041934f,   -0.744801f, -0.071342f, +0.041934f,
          -0.721571f, -0.101253f, +0.041934f,   -0.693579f, -0.130351f, +0.041934f,
          -0.660662f, -0.157923f, +0.041934f,   -0.622651f, -0.183260f, +0.041934f,
          -0.579383f, -0.205649f, +0.041934f,   -0.803069f, +0.040254f, +0.055038f,
          -0.796797f, +0.012677f, +0.055038f,   -0.786151f, -0.016580f, +0.055038f,
          -0.770978f, -0.046886f, +0.055038f,   -0.751121f, -0.077610f, +0.055038f,
          -0.726424f, -0.108121f, +0.055038f,   -0.696732f, -0.137790f, +0.055038f,
          -0.661887f, -0.165986f, +0.055038f,   -0.621736f, -0.192077f, +0.055038f,
          -0.576121f, -0.215434f, +0.055038f,   -0.814773f, +0.037877f, +0.062900f,
          -0.807980f, +0.008447f, +0.062900f,   -0.796482f, -0.022234f, +0.062900f,
          -0.780135f, -0.053634f, +0.062900f,   -0.758794f, -0.085221f, +0.062900f,
          -0.732317f, -0.116462f, +0.062900f,   -0.700559f, -0.146825f, +0.062900f,
          -0.663377f, -0.175776f, +0.062900f,   -0.620626f, -0.202784f, +0.062900f,
          -0.572161f, -0.227315f, +0.062900f,   -0.827512f, +0.035288f, +0.065521f,
          -0.820151f, +0.003844f, +0.065521f,   -0.807724f, -0.028387f, +0.065521f,
          -0.790099f, -0.060978f, +0.065521f,   -0.767145f, -0.093504f, +0.065521f,
          -0.738730f, -0.125539f, +0.065521f,   -0.704724f, -0.156656f, +0.065521f,
          -0.664997f, -0.186430f, +0.065521f,   -0.619416f, -0.214434f, +0.065521f,
          -0.567851f, -0.240245f, +0.065521f,   -0.840249f, +0.032701f, +0.062900f,
          -0.832323f, -0.000759f, +0.062900f,   -0.818967f, -0.034540f, +0.062900f,
          -0.800063f, -0.068323f, +0.062900f,   -0.775496f, -0.101787f, +0.062900f,
          -0.745144f, -0.134615f, +0.062900f,   -0.708890f, -0.166487f, +0.062900f,
          -0.666617f, -0.197083f, +0.062900f,   -0.618207f, -0.226086f, +0.062900f,
          -0.563541f, -0.253174f, +0.062900f,   -0.851955f, +0.030324f, +0.055038f,
          -0.843506f, -0.004989f, +0.055038f,   -0.829298f, -0.040193f, +0.055038f,
          -0.809220f, -0.075070f, +0.055038f,   -0.783168f, -0.109399f, +0.055038f,
          -0.751037f, -0.142956f, +0.055038f,   -0.712718f, -0.175521f, +0.055038f,
          -0.668107f, -0.206874f, +0.055038f,   -0.617096f, -0.236792f, +0.055038f,
          -0.559581f, -0.265055f, +0.055038f,   -0.861593f, +0.028366f, +0.041934f,
          -0.852718f, -0.008472f, +0.041934f,   -0.837804f, -0.044850f, +0.041934f,
          -0.816761f, -0.080629f, +0.041934f,   -0.789487f, -0.115667f, +0.041934f,
          -0.755889f, -0.149824f, +0.041934f,   -0.715871f, -0.182961f, +0.041934f,
          -0.669333f, -0.214937f, +0.041934f,   -0.616182f, -0.245610f, +0.041934f,
          -0.556319f, -0.274840f, +0.041934f,   -0.868135f, +0.027037f, +0.023588f,
          -0.858967f, -0.010835f, +0.023588f,   -0.843578f, -0.048009f, +0.023588f,
          -0.821878f, -0.084400f, +0.023588f,   -0.793775f, -0.119921f, +0.023588f,
          -0.759182f, -0.154486f, +0.023588f,   -0.718009f, -0.188009f, +0.023588f,
          -0.670165f, -0.220408f, +0.023588f,   -0.615561f, -0.251592f, +0.023588f,
          -0.554106f, -0.281479f, +0.023588f,   -0.870545f, +0.026547f, -0.000000f,
          -0.861270f, -0.011706f, -0.000000f,   -0.845705f, -0.049174f, -0.000000f,
          -0.823762f, -0.085789f, -0.000000f,   -0.795355f, -0.121487f, -0.000000f,
          -0.760396f, -0.156203f, -0.000000f,   -0.718797f, -0.189870f, -0.000000f,
          -0.670472f, -0.222423f, -0.000000f,   -0.615331f, -0.253796f, -0.000000f,
          -0.553291f, -0.283925f, -0.000000f,   -0.868135f, +0.027037f, -0.023588f,
          -0.858967f, -0.010835f, -0.023588f,   -0.843578f, -0.048009f, -0.023588f,
          -0.821878f, -0.084400f, -0.023588f,   -0.793775f, -0.119921f, -0.023588f,
          -0.759182f, -0.154486f, -0.023588f,   -0.718009f, -0.188009f, -0.023588f,
          -0.670165f, -0.220408f, -0.023588f,   -0.615561f, -0.251592f, -0.023588f,
          -0.554106f, -0.281479f, -0.023588f,   -0.861593f, +0.028366f, -0.041934f,
          -0.852718f, -0.008472f, -0.041934f,   -0.837804f, -0.044850f, -0.041934f,
          -0.816761f, -0.080629f, -0.041934f,   -0.789487f, -0.115667f, -0.041934f,
          -0.755889f, -0.149824f, -0.041934f,   -0.715871f, -0.182961f, -0.041934f,
          -0.669333f, -0.214937f, -0.041934f,   -0.616182f, -0.245610f, -0.041934f,
          -0.556319f, -0.274840f, -0.041934f,   -0.851955f, +0.030324f, -0.055038f,
          -0.843506f, -0.004989f, -0.055038f,   -0.829298f, -0.040193f, -0.055038f,
          -0.809220f, -0.075070f, -0.055038f,   -0.783168f, -0.109399f, -0.055038f,
          -0.751037f, -0.142956f, -0.055038f,   -0.712718f, -0.175521f, -0.055038f,
          -0.668107f, -0.206874f, -0.055038f,   -0.617096f, -0.236792f, -0.055038f,
          -0.559581f, -0.265055f, -0.055038f,   -0.840249f, +0.032701f, -0.062900f,
          -0.832323f, -0.000759f, -0.062900f,   -0.818967f, -0.034540f, -0.062900f,
          -0.800063f, -0.068323f, -0.062900f,   -0.775496f, -0.101787f, -0.062900f,
          -0.745144f, -0.134615f, -0.062900f,   -0.708890f, -0.166487f, -0.062900f,
          -0.666617f, -0.197083f, -0.062900f,   -0.618207f, -0.226086f, -0.062900f,
          -0.563541f, -0.253174f, -0.062900f,   -0.827512f, +0.035288f, -0.065521f,
          -0.820151f, +0.003844f, -0.065521f,   -0.807724f, -0.028387f, -0.065521f,
          -0.790099f, -0.060978f, -0.065521f,   -0.767145f, -0.093504f, -0.065521f,
          -0.738730f, -0.125539f, -0.065521f,   -0.704724f, -0.156656f, -0.065521f,
          -0.664997f, -0.186430f, -0.065521f,   -0.619416f, -0.214434f, -0.065521f,
          -0.567851f, -0.240245f, -0.065521f,   -0.814773f, +0.037877f, -0.062900f,
          -0.807980f, +0.008447f, -0.062900f,   -0.796482f, -0.022234f, -0.062900f,
          -0.780135f, -0.053634f, -0.062900f,   -0.758794f, -0.085221f, -0.062900f,
          -0.732317f, -0.116462f, -0.062900f,   -0.700559f, -0.146825f, -0.062900f,
          -0.663377f, -0.175776f, -0.062900f,   -0.620626f, -0.202784f, -0.062900f,
          -0.572161f, -0.227315f, -0.062900f,   -0.803069f, +0.040254f, -0.055038f,
          -0.796797f, +0.012677f, -0.055038f,   -0.786151f, -0.016580f, -0.055038f,
          -0.770978f, -0.046886f, -0.055038f,   -0.751121f, -0.077610f, -0.055038f,
          -0.726424f, -0.108121f, -0.055038f,   -0.696732f, -0.137790f, -0.055038f,
          -0.661887f, -0.165986f, -0.055038f,   -0.621736f, -0.192077f, -0.055038f,
          -0.576121f, -0.215434f, -0.055038f,   -0.793429f, +0.042212f, -0.041934f,
          -0.787585f, +0.016160f, -0.041934f,   -0.777644f, -0.011924f, -0.041934f,
          -0.763437f, -0.041328f, -0.041934f,   -0.744801f, -0.071342f, -0.041934f,
          -0.721571f, -0.101253f, -0.041934f,   -0.693579f, -0.130351f, -0.041934f,
          -0.660662f, -0.157923f, -0.041934f,   -0.622651f, -0.183260f, -0.041934f,
          -0.579383f, -0.205649f, -0.041934f,   -0.786889f, +0.043541f, -0.023588f,
          -0.781336f, +0.018523f, -0.023588f,   -0.771871f, -0.008764f, -0.023588f,
          -0.758320f, -0.037557f, -0.023588f,   -0.740514f, -0.067088f, -0.023588f,
          -0.718278f, -0.096592f, -0.023588f,   -0.691441f, -0.125302f, -0.023588f,
          -0.659829f, -0.152452f, -0.023588f,   -0.623272f, -0.177277f, -0.023588f,
          -0.581596f, -0.199010f, -0.023588f,   +0.495050f, -0.043681f, -0.000000f,
          +0.495050f, -0.050408f, +0.051893f,   +0.564763f, -0.044199f, +0.050990f,
          +0.563745f, -0.038090f, -0.000000f,   +0.616476f, -0.027921f, +0.048540f,
          +0.614793f, -0.022539f, -0.000000f,   +0.653598f, -0.003444f, +0.044930f,
          +0.651514f, +0.001136f, -0.000000f,   +0.679536f, +0.027358f, +0.040546f,
          +0.677228f, +0.031101f, -0.000000f,   +0.697699f, +0.062616f, +0.035775f,
          +0.695253f, +0.065521f, -0.000000f,   +0.711494f, +0.100459f, +0.031004f,
          +0.708911f, +0.102563f, -0.000000f,   +0.724328f, +0.139015f, +0.026619f,
          +0.721520f, +0.140390f, -0.000000f,   +0.739610f, +0.176411f, +0.023009f,
          +0.736401f, +0.177169f, -0.000000f,   +0.760747f, +0.210778f, +0.020559f,
          +0.756872f, +0.211066f, -0.000000f,   +0.791147f, +0.240245f, +0.019656f,
          +0.786255f, +0.240245f, -0.000000f,   +0.495050f, -0.068666f, +0.092254f,
          +0.567525f, -0.060782f, +0.090649f,   +0.621044f, -0.042528f, +0.086294f,
          +0.659256f, -0.015877f, +0.079875f,   +0.685804f, +0.017200f, +0.072081f,
          +0.704339f, +0.054732f, +0.063599f,   +0.718506f, +0.094749f, +0.055118f,
          +0.731950f, +0.135280f, +0.047323f,   +0.748321f, +0.174353f, +0.040905f,
          +0.771264f, +0.209999f, +0.036549f,   +0.804426f, +0.240245f, +0.034945f,
          +0.495050f, -0.095574f, +0.121083f,   +0.571595f, -0.085218f, +0.118977f,
          +0.627776f, -0.064054f, +0.113261f,   +0.667591f, -0.034199f, +0.104836f,
          +0.695041f, +0.002229f, +0.094606f,   +0.714123f, +0.043112f, +0.083474f,
          +0.728837f, +0.086334f, +0.072342f,   +0.743183f, +0.129775f, +0.062112f,
          +0.761158f, +0.171320f, +0.053688f,   +0.786763f, +0.208849f, +0.047971f,
          +0.823995f, +0.240245f, +0.045865f,   +0.495050f, -0.128247f, +0.138381f,
          +0.576537f, -0.114891f, +0.135974f,   +0.635950f, -0.090192f, +0.129441f,
          +0.677715f, -0.056446f, +0.119812f,   +0.706257f, -0.015949f, +0.108122f,
          +0.726005f, +0.029004f, +0.095399f,   +0.741384f, +0.076116f, +0.082676f,
          +0.756822f, +0.123093f, +0.070985f,   +0.776747f, +0.167637f, +0.061357f,
          +0.805583f, +0.207453f, +0.054824f,   +0.847758f, +0.240245f, +0.052417f,
          +0.495050f, -0.163803f, +0.144147f,   +0.581916f, -0.147182f, +0.141639f,
          +0.644846f, -0.118637f, +0.134834f,   +0.688730f, -0.080657f, +0.124805f,
          +0.718462f, -0.035731f, +0.112627f,   +0.738934f, +0.013651f, +0.099374f,
          +0.755038f, +0.064997f, +0.086121f,   +0.771666f, +0.115819f, +0.073943f,
          +0.793710f, +0.163628f, +0.063914f,   +0.826063f, +0.205934f, +0.057108f,
          +0.873617f, +0.240245f, +0.054601f,   +0.495050f, -0.199359f, +0.138381f,
          +0.587295f, -0.179475f, +0.135974f,   +0.653741f, -0.147082f, +0.129441f,
          +0.699746f, -0.104868f, +0.119812f,   +0.730667f, -0.055513f, +0.108122f,
          +0.751864f, -0.001704f, +0.095399f,   +0.768692f, +0.053878f, +0.082676f,
          +0.786509f, +0.108547f, +0.070985f,   +0.810673f, +0.159620f, +0.061357f,
          +0.846543f, +0.204415f, +0.054824f,   +0.899476f, +0.240245f, +0.052417f,
          +0.495050f, -0.232033f, +0.121083f,   +0.592238f, -0.209147f, +0.118977f,
          +0.661915f, -0.173221f, +0.113261f,   +0.709870f, -0.127115f, +0.104836f,
          +0.741884f, -0.073691f, +0.094606f,   +0.763745f, -0.015812f, +0.083474f,
          +0.781238f, +0.043660f, +0.072342f,   +0.800148f, +0.101864f, +0.062112f,
          +0.826262f, +0.155937f, +0.053688f,   +0.865363f, +0.203018f, +0.047971f,
          +0.923238f, +0.240245f, +0.045865f,   +0.495050f, -0.258940f, +0.092254f,
          +0.596308f, -0.233584f, +0.090649f,   +0.668648f, -0.194747f, +0.086294f,
          +0.718205f, -0.145436f, +0.079875f,   +0.751121f, -0.088662f, +0.072081f,
          +0.773529f, -0.027432f, +0.063599f,   +0.791570f, +0.035245f, +0.055118f,
          +0.811382f, +0.096360f, +0.047323f,   +0.839098f, +0.152904f, +0.040905f,
          +0.880862f, +0.201868f, +0.036549f,   +0.942807f, +0.240245f, +0.034945f,
          +0.495050f, -0.277199f, +0.051893f,   +0.599069f, -0.250167f, +0.050990f,
          +0.673215f, -0.209354f, +0.048540f,   +0.723863f, -0.157868f, +0.044930f,
          +0.757388f, -0.098820f, +0.040546f,   +0.780169f, -0.035316f, +0.035775f,
          +0.798581f, +0.029535f, +0.031004f,   +0.819003f, +0.092625f, +0.026619f,
          +0.847810f, +0.150846f, +0.023009f,   +0.891379f, +0.201088f, +0.020559f,
          +0.956086f, +0.240245f, +0.019656f,   +0.495050f, -0.283925f, -0.000000f,
          +0.600087f, -0.256275f, -0.000000f,   +0.674898f, -0.214735f, -0.000000f,
          +0.725946f, -0.162448f, -0.000000f,   +0.759697f, -0.102563f, -0.000000f,
          +0.782615f, -0.038220f, -0.000000f,   +0.801165f, +0.027432f, -0.000000f,
          +0.821811f, +0.091250f, -0.000000f,   +0.851019f, +0.150087f, -0.000000f,
          +0.895253f, +0.200801f, -0.000000f,   +0.960978f, +0.240245f, -0.000000f,
          +0.495050f, -0.277199f, -0.051893f,   +0.599069f, -0.250167f, -0.050990f,
          +0.673215f, -0.209354f, -0.048540f,   +0.723863f, -0.157868f, -0.044930f,
          +0.757388f, -0.098820f, -0.040546f,   +0.780169f, -0.035316f, -0.035775f,
          +0.798581f, +0.029535f, -0.031004f,   +0.819003f, +0.092625f, -0.026619f,
          +0.847810f, +0.150846f, -0.023009f,   +0.891379f, +0.201088f, -0.020559f,
          +0.956086f, +0.240245f, -0.019656f,   +0.495050f, -0.258940f, -0.092254f,
          +0.596308f, -0.233584f, -0.090649f,   +0.668648f, -0.194747f, -0.086294f,
          +0.718205f, -0.145436f, -0.079875f,   +0.751121f, -0.088662f, -0.072081f,
          +0.773529f, -0.027432f, -0.063599f,   +0.791570f, +0.035245f, -0.055118f,
          +0.811382f, +0.096360f, -0.047323f,   +0.839098f, +0.152904f, -0.040905f,
          +0.880862f, +0.201868f, -0.036549f,   +0.942807f, +0.240245f, -0.034945f,
          +0.495050f, -0.232033f, -0.121083f,   +0.592238f, -0.209147f, -0.118977f,
          +0.661915f, -0.173221f, -0.113261f,   +0.709870f, -0.127115f, -0.104836f,
          +0.741884f, -0.073691f, -0.094606f,   +0.763745f, -0.015812f, -0.083474f,
          +0.781238f, +0.043660f, -0.072342f,   +0.800148f, +0.101864f, -0.062112f,
          +0.826262f, +0.155937f, -0.053688f,   +0.865363f, +0.203018f, -0.047971f,
          +0.923238f, +0.240245f, -0.045865f,   +0.495050f, -0.199359f, -0.138381f,
          +0.587295f, -0.179475f, -0.135974f,   +0.653741f, -0.147082f, -0.129441f,
          +0.699746f, -0.104868f, -0.119812f,   +0.730667f, -0.055513f, -0.108122f,
          +0.751864f, -0.001704f, -0.095399f,   +0.768692f, +0.053878f, -0.082676f,
          +0.786509f, +0.108547f, -0.070985f,   +0.810673f, +0.159620f, -0.061357f,
          +0.846543f, +0.204415f, -0.054824f,   +0.899476f, +0.240245f, -0.052417f,
          +0.495050f, -0.163803f, -0.144147f,   +0.581916f, -0.147182f, -0.141640f,
          +0.644846f, -0.118637f, -0.134834f,   +0.688730f, -0.080657f, -0.124805f,
          +0.718462f, -0.035731f, -0.112627f,   +0.738934f, +0.013651f, -0.099374f,
          +0.755038f, +0.064997f, -0.086121f,   +0.771666f, +0.115819f, -0.073943f,
          +0.793710f, +0.163628f, -0.063914f,   +0.826063f, +0.205934f, -0.057108f,
          +0.873617f, +0.240245f, -0.054601f,   +0.495050f, -0.128247f, -0.138381f,
          +0.576537f, -0.114891f, -0.135974f,   +0.635950f, -0.090192f, -0.129441f,
          +0.677715f, -0.056446f, -0.119812f,   +0.706257f, -0.015949f, -0.108122f,
          +0.726005f, +0.029004f, -0.095399f,   +0.741384f, +0.076116f, -0.082676f,
          +0.756822f, +0.123093f, -0.070985f,   +0.776747f, +0.167637f, -0.061357f,
          +0.805583f, +0.207453f, -0.054824f,   +0.847758f, +0.240245f, -0.052417f,
          +0.495050f, -0.095574f, -0.121083f,   +0.571595f, -0.085217f, -0.118977f,
          +0.627776f, -0.064054f, -0.113261f,   +0.667591f, -0.034199f, -0.104836f,
          +0.695041f, +0.002229f, -0.094606f,   +0.714123f, +0.043112f, -0.083474f,
          +0.728837f, +0.086334f, -0.072342f,   +0.743183f, +0.129776f, -0.062112f,
          +0.761158f, +0.171320f, -0.053688f,   +0.786763f, +0.208849f, -0.047971f,
          +0.823995f, +0.240245f, -0.045865f,   +0.495050f, -0.068666f, -0.092254f,
          +0.567525f, -0.060780f, -0.090649f,   +0.621044f, -0.042528f, -0.086294f,
          +0.659256f, -0.015877f, -0.079875f,   +0.685804f, +0.017200f, -0.072081f,
          +0.704339f, +0.054733f, -0.063599f,   +0.718506f, +0.094749f, -0.055118f,
          +0.731950f, +0.135280f, -0.047323f,   +0.748321f, +0.174353f, -0.040905f,
          +0.771264f, +0.209999f, -0.036549f,   +0.804426f, +0.240245f, -0.034945f,
          +0.495050f, -0.050408f, -0.051893f,   +0.564763f, -0.044199f, -0.050990f,
          +0.616476f, -0.027921f, -0.048540f,   +0.653598f, -0.003444f, -0.044930f,
          +0.679536f, +0.027358f, -0.040546f,   +0.697699f, +0.062616f, -0.035775f,
          +0.711494f, +0.100459f, -0.031004f,   +0.724328f, +0.139015f, -0.026619f,
          +0.739610f, +0.176411f, -0.023009f,   +0.760747f, +0.210778f, -0.020559f,
          +0.791147f, +0.240245f, -0.019656f,   +0.800061f, +0.246186f, +0.019436f,
          +0.794933f, +0.246141f, -0.000000f,   +0.808493f, +0.250817f, +0.018839f,
          +0.803261f, +0.250728f, -0.000000f,   +0.816112f, +0.254129f, +0.017958f,
          +0.810891f, +0.254004f, -0.000000f,   +0.822584f, +0.256123f, +0.016889f,
          +0.817472f, +0.255970f, -0.000000f,   +0.827574f, +0.256797f, +0.015725f,
          +0.822656f, +0.256624f, -0.000000f,   +0.830749f, +0.256146f, +0.014561f,
          +0.826092f, +0.255970f, -0.000000f,   +0.831778f, +0.254168f, +0.013492f,
          +0.827432f, +0.254004f, -0.000000f,   +0.830324f, +0.250860f, +0.012612f,
          +0.826325f, +0.250728f, -0.000000f,   +0.826055f, +0.246220f, +0.012014f,
          +0.822423f, +0.246141f, -0.000000f,   +0.818637f, +0.240245f, +0.011794f,
          +0.815376f, +0.240245f, -0.000000f,   +0.813978f, +0.246310f, +0.034553f,
          +0.822693f, +0.251055f, +0.033491f,   +0.830282f, +0.254469f, +0.031925f,
          +0.836455f, +0.256542f, +0.030024f,   +0.840921f, +0.257264f, +0.027956f,
          +0.843391f, +0.256624f, +0.025887f,   +0.843573f, +0.254613f, +0.023986f,
          +0.841176f, +0.251218f, +0.022421f,   +0.835913f, +0.246433f, +0.021358f,
          +0.827490f, +0.240245f, +0.020967f,   +0.834487f, +0.246492f, +0.045351f,
          +0.843618f, +0.251407f, +0.043957f,   +0.851165f, +0.254970f, +0.041902f,
          +0.856898f, +0.257159f, +0.039407f,   +0.860593f, +0.257952f, +0.036692f,
          +0.862021f, +0.257328f, +0.033977f,   +0.860956f, +0.255267f, +0.031482f,
          +0.857172f, +0.251747f, +0.029427f,   +0.850440f, +0.246747f, +0.028033f,
          +0.840536f, +0.240245f, +0.027519f,   +0.859391f, +0.246713f, +0.051830f,
          +0.869029f, +0.251835f, +0.050236f,   +0.876522f, +0.255578f, +0.047888f,
          +0.881722f, +0.257907f, +0.045037f,   +0.884479f, +0.258787f, +0.041934f,
          +0.884643f, +0.258184f, +0.038831f,   +0.882064f, +0.256063f, +0.035979f,
          +0.876593f, +0.252389f, +0.033631f,   +0.868080f, +0.247128f, +0.032037f,
          +0.856377f, +0.240245f, +0.031450f,   +0.886492f, +0.246953f, +0.053990f,
          +0.896680f, +0.252301f, +0.052330f,   +0.904116f, +0.256240f, +0.049884f,
          +0.908736f, +0.258721f, +0.046913f,   +0.910473f, +0.259696f, +0.043681f,
          +0.909260f, +0.259115f, +0.040448f,   +0.905034f, +0.256928f, +0.037478f,
          +0.897729f, +0.253087f, +0.035032f,   +0.887278f, +0.247542f, +0.033372f,
          +0.873617f, +0.240245f, +0.032761f,   +0.913593f, +0.247193f, +0.051830f,
          +0.924332f, +0.252766f, +0.050236f,   +0.931712f, +0.256902f, +0.047888f,
          +0.935751f, +0.259536f, +0.045037f,   +0.936466f, +0.260606f, +0.041934f,
          +0.933878f, +0.260045f, +0.038831f,   +0.928005f, +0.257794f, +0.035979f,
          +0.918864f, +0.253785f, +0.033631f,   +0.906475f, +0.247957f, +0.032037f,
          +0.890856f, +0.240245f, +0.031450f,   +0.938497f, +0.247413f, +0.045351f,
          +0.949743f, +0.253194f, +0.043957f,   +0.957069f, +0.257510f, +0.041902f,
          +0.960574f, +0.260284f, +0.039407f,   +0.960353f, +0.261441f, +0.036692f,
          +0.956500f, +0.260902f, +0.033977f,   +0.949112f, +0.258588f, +0.031482f,
          +0.938285f, +0.254426f, +0.029427f,   +0.924115f, +0.248338f, +0.028033f,
          +0.906698f, +0.240245f, +0.027519f,   +0.959006f, +0.247595f, +0.034553f,
          +0.970667f, +0.253546f, +0.033491f,   +0.977952f, +0.258011f, +0.031925f,
          +0.981017f, +0.260902f, +0.030024f,   +0.980023f, +0.262129f, +0.027956f,
          +0.975130f, +0.261605f, +0.025887f,   +0.966495f, +0.259244f, +0.023986f,
          +0.954281f, +0.254955f, +0.022421f,   +0.938643f, +0.248651f, +0.021358f,
          +0.919744f, +0.240245f, +0.020967f,   +0.972923f, +0.247718f, +0.019436f,
          +0.984868f, +0.253785f, +0.018839f,   +0.992122f, +0.258351f, +0.017958f,
          +0.994889f, +0.261320f, +0.016889f,   +0.993371f, +0.262595f, +0.015725f,
          +0.987772f, +0.262084f, +0.014561f,   +0.978291f, +0.259688f, +0.013492f,
          +0.965133f, +0.255314f, +0.012612f,   +0.948501f, +0.248864f, +0.012014f,
          +0.928596f, +0.240245f, +0.011794f,   +0.978050f, +0.247764f, -0.000000f,
          +0.990099f, +0.253873f, -0.000000f,   +0.997343f, +0.258476f, -0.000000f,
          +1.000000f, +0.261474f, -0.000000f,   +0.998289f, +0.262768f, -0.000000f,
          +0.992429f, +0.262260f, -0.000000f,   +0.982637f, +0.259852f, -0.000000f,
          +0.969132f, +0.255446f, -0.000000f,   +0.952133f, +0.248942f, -0.000000f,
          +0.931858f, +0.240245f, -0.000000f,   +0.972923f, +0.247718f, -0.019436f,
          +0.984868f, +0.253785f, -0.018839f,   +0.992122f, +0.258351f, -0.017958f,
          +0.994889f, +0.261320f, -0.016889f,   +0.993371f, +0.262595f, -0.015725f,
          +0.987772f, +0.262084f, -0.014561f,   +0.978291f, +0.259688f, -0.013492f,
          +0.965133f, +0.255314f, -0.012612f,   +0.948501f, +0.248864f, -0.012014f,
          +0.928596f, +0.240245f, -0.011794f,   +0.959006f, +0.247595f, -0.034553f,
          +0.970667f, +0.253546f, -0.033491f,   +0.977952f, +0.258011f, -0.031925f,
          +0.981017f, +0.260902f, -0.030024f,   +0.980023f, +0.262129f, -0.027956f,
          +0.975130f, +0.261605f, -0.025887f,   +0.966495f, +0.259244f, -0.023986f,
          +0.954281f, +0.254955f, -0.022421f,   +0.938643f, +0.248651f, -0.021358f,
          +0.919744f, +0.240245f, -0.020967f,   +0.938497f, +0.247413f, -0.045351f,
          +0.949743f, +0.253194f, -0.043957f,   +0.957069f, +0.257510f, -0.041902f,
          +0.960574f, +0.260284f, -0.039407f,   +0.960353f, +0.261441f, -0.036692f,
          +0.956500f, +0.260902f, -0.033977f,   +0.949112f, +0.258588f, -0.031482f,
          +0.938285f, +0.254426f, -0.029427f,   +0.924115f, +0.248338f, -0.028033f,
          +0.906698f, +0.240245f, -0.027519f,   +0.913593f, +0.247193f, -0.051830f,
          +0.924332f, +0.252766f, -0.050236f,   +0.931712f, +0.256902f, -0.047888f,
          +0.935751f, +0.259536f, -0.045037f,   +0.936466f, +0.260606f, -0.041934f,
          +0.933878f, +0.260045f, -0.038831f,   +0.928005f, +0.257794f, -0.035979f,
          +0.918864f, +0.253785f, -0.033631f,   +0.906475f, +0.247957f, -0.032037f,
          +0.890856f, +0.240245f, -0.031450f,   +0.886492f, +0.246953f, -0.053990f,
          +0.896680f, +0.252301f, -0.052330f,   +0.904116f, +0.256240f, -0.049884f,
          +0.908736f, +0.258721f, -0.046913f,   +0.910473f, +0.259696f, -0.043681f,
          +0.909260f, +0.259115f, -0.040448f,   +0.905034f, +0.256928f, -0.037478f,
          +0.897729f, +0.253087f, -0.035032f,   +0.887278f, +0.247542f, -0.033372f,
          +0.873617f, +0.240245f, -0.032761f,   +0.859391f, +0.246713f, -0.051830f,
          +0.869029f, +0.251835f, -0.050236f,   +0.876522f, +0.255578f, -0.047888f,
          +0.881722f, +0.257907f, -0.045037f,   +0.884479f, +0.258787f, -0.041934f,
          +0.884643f, +0.258184f, -0.038831f,   +0.882064f, +0.256063f, -0.035979f,
          +0.876593f, +0.252389f, -0.033631f,   +0.868080f, +0.247128f, -0.032037f,
          +0.856377f, +0.240245f, -0.031450f,   +0.834487f, +0.246492f, -0.045351f,
          +0.843618f, +0.251407f, -0.043957f,   +0.851165f, +0.254970f, -0.041902f,
          +0.856898f, +0.257159f, -0.039407f,   +0.860593f, +0.257952f, -0.036692f,
          +0.862021f, +0.257328f, -0.033977f,   +0.860956f, +0.255267f, -0.031482f,
          +0.857172f, +0.251747f, -0.029427f,   +0.850440f, +0.246747f, -0.028033f,
          +0.840536f, +0.240245f, -0.027519f,   +0.813977f, +0.246310f, -0.034553f,
          +0.822693f, +0.251055f, -0.033491f,   +0.830282f, +0.254469f, -0.031925f,
          +0.836455f, +0.256542f, -0.030024f,   +0.840921f, +0.257264f, -0.027956f,
          +0.843391f, +0.256624f, -0.025887f,   +0.843573f, +0.254613f, -0.023986f,
          +0.841176f, +0.251218f, -0.022421f,   +0.835913f, +0.246433f, -0.021358f,
          +0.827490f, +0.240245f, -0.020967f,   +0.800061f, +0.246186f, -0.019436f,
          +0.808493f, +0.250817f, -0.018839f,   +0.816112f, +0.254129f, -0.017958f,
          +0.822584f, +0.256123f, -0.016889f,   +0.827574f, +0.256797f, -0.015725f,
          +0.830749f, +0.256146f, -0.014561f,   +0.831778f, +0.254168f, -0.013492f,
          +0.830324f, +0.250860f, -0.012612f,   +0.826055f, +0.246220f, -0.012014f,
          +0.818637f, +0.240245f, -0.011794f,   +0.055943f, +0.456160f, +0.009333f,
          +0.056669f, +0.456160f, -0.000000f,   +0.000000f, +0.458649f, -0.000000f,
          +0.088772f, +0.449214f, +0.014809f,   +0.089924f, +0.449214f, -0.000000f,
          +0.102973f, +0.438600f, +0.017178f,   +0.104310f, +0.438600f, -0.000000f,
          +0.103031f, +0.425102f, +0.017186f,   +0.104368f, +0.425102f, -0.000000f,
          +0.093429f, +0.409507f, +0.015582f,   +0.094642f, +0.409507f, -0.000000f,
          +0.078652f, +0.392603f, +0.013114f,   +0.079674f, +0.392603f, -0.000000f,
          +0.063186f, +0.375174f, +0.010529f,   +0.064007f, +0.375174f, -0.000000f,
          +0.051513f, +0.358008f, +0.008576f,   +0.052184f, +0.358008f, -0.000000f,
          +0.048120f, +0.341889f, +0.008002f,   +0.048748f, +0.341889f, -0.000000f,
          +0.057491f, +0.327606f, +0.009556f,   +0.058241f, +0.327606f, -0.000000f,
          +0.053835f, +0.456160f, +0.018134f,   +0.085428f, +0.449214f, +0.028775f,
          +0.099094f, +0.438600f, +0.033378f,   +0.099149f, +0.425102f, +0.033394f,
          +0.089908f, +0.409507f, +0.030278f,   +0.075687f, +0.392603f, +0.025484f,
          +0.060802f, +0.375174f, +0.020463f,   +0.049568f, +0.358008f, +0.016670f,
          +0.046300f, +0.341889f, +0.015559f,   +0.055315f, +0.327606f, +0.018581f,
          +0.050453f, +0.456160f, +0.026298f,   +0.080060f, +0.449214f, +0.041730f,
          +0.092868f, +0.438600f, +0.048405f,   +0.092918f, +0.425102f, +0.048429f,
          +0.084257f, +0.409507f, +0.043912f,   +0.070929f, +0.392603f, +0.036960f,
          +0.056977f, +0.375174f, +0.029681f,   +0.046446f, +0.358008f, +0.024184f,
          +0.043381f, +0.341889f, +0.022575f,   +0.051825f, +0.327606f, +0.026963f,
          +0.045902f, +0.456160f, +0.033718f,   +0.072838f, +0.449214f, +0.053504f,
          +0.084490f, +0.438600f, +0.062063f,   +0.084535f, +0.425102f, +0.062095f,
          +0.076655f, +0.409507f, +0.056304f,   +0.064527f, +0.392603f, +0.047392f,
          +0.051831f, +0.375174f, +0.038063f,   +0.042248f, +0.358008f, +0.031017f,
          +0.039455f, +0.341889f, +0.028959f,   +0.047133f, +0.327606f, +0.034591f,
          +0.040288f, +0.456160f, +0.040288f,   +0.063930f, +0.449214f, +0.063930f,
          +0.074156f, +0.438600f, +0.074156f,   +0.074196f, +0.425102f, +0.074196f,
          +0.067278f, +0.409507f, +0.067278f,   +0.056631f, +0.392603f, +0.056631f,
          +0.045486f, +0.375174f, +0.045486f,   +0.037072f, +0.358008f, +0.037072f,
          +0.034617f, +0.341889f, +0.034617f,   +0.041351f, +0.327606f, +0.041351f,
          +0.033718f, +0.456160f, +0.045901f,   +0.053504f, +0.449214f, +0.072838f,
          +0.062063f, +0.438600f, +0.084490f,   +0.062095f, +0.425102f, +0.084535f,
          +0.056304f, +0.409507f, +0.076655f,   +0.047392f, +0.392603f, +0.064527f,
          +0.038063f, +0.375174f, +0.051831f,   +0.031017f, +0.358008f, +0.042248f,
          +0.028959f, +0.341889f, +0.039455f,   +0.034591f, +0.327606f, +0.047133f,
          +0.026298f, +0.456160f, +0.050453f,   +0.041730f, +0.449214f, +0.080060f,
          +0.048405f, +0.438600f, +0.092868f,   +0.048429f, +0.425102f, +0.092918f,
          +0.043912f, +0.409507f, +0.084257f,   +0.036960f, +0.392603f, +0.070929f,
          +0.029681f, +0.375174f, +0.056977f,   +0.024184f, +0.358008f, +0.046446f,
          +0.022575f, +0.341889f, +0.043381f,   +0.026963f, +0.327606f, +0.051825f,
          +0.018134f, +0.456160f, +0.053835f,   +0.028775f, +0.449214f, +0.085428f,
          +0.033378f, +0.438600f, +0.099094f,   +0.033394f, +0.425102f, +0.099149f,
          +0.030278f, +0.409507f, +0.089908f,   +0.025484f, +0.392603f, +0.075687f,
          +0.020463f, +0.375174f, +0.060802f,   +0.016670f, +0.358008f, +0.049568f,
          +0.015559f, +0.341889f, +0.046300f,   +0.018581f, +0.327606f, +0.055315f,
          +0.009333f, +0.456160f, +0.055942f,   +0.014809f, +0.449214f, +0.088772f,
          +0.017178f, +0.438600f, +0.102973f,   +0.017186f, +0.425102f, +0.103031f,
          +0.015582f, +0.409507f, +0.093429f,   +0.013114f, +0.392603f, +0.078652f,
          +0.010529f, +0.375174f, +0.063186f,   +0.008576f, +0.358008f, +0.051513f,
          +0.008002f, +0.341889f, +0.048120f,   +0.009556f, +0.327606f, +0.057491f,
          -0.000000f, +0.456160f, +0.056669f,   -0.000000f, +0.449214f, +0.089924f,
          -0.000000f, +0.438600f, +0.104310f,   -0.000000f, +0.425102f, +0.104368f,
          -0.000000f, +0.409507f, +0.094642f,   -0.000000f, +0.392603f, +0.079674f,
          -0.000000f, +0.375174f, +0.064007f,   -0.000000f, +0.358008f, +0.052184f,
          -0.000000f, +0.341889f, +0.048748f,   -0.000000f, +0.327606f, +0.058241f,
          -0.009333f, +0.456160f, +0.055942f,   -0.014809f, +0.449214f, +0.088772f,
          -0.017178f, +0.438600f, +0.102973f,   -0.017186f, +0.425102f, +0.103031f,
          -0.015582f, +0.409507f, +0.093429f,   -0.013114f, +0.392603f, +0.078652f,
          -0.010529f, +0.375174f, +0.063186f,   -0.008576f, +0.358008f, +0.051513f,
          -0.008002f, +0.341889f, +0.048120f,   -0.009556f, +0.327606f, +0.057491f,
          -0.018134f, +0.456160f, +0.053835f,   -0.028775f, +0.449214f, +0.085428f,
          -0.033378f, +0.438600f, +0.099094f,   -0.033394f, +0.425102f, +0.099149f,
          -0.030278f, +0.409507f, +0.089908f,   -0.025484f, +0.392603f, +0.075687f,
          -0.020463f, +0.375174f, +0.060802f,   -0.016670f, +0.358008f, +0.049568f,
          -0.015559f, +0.341889f, +0.046300f,   -0.018581f, +0.327606f, +0.055315f,
          -0.026298f, +0.456160f, +0.050453f,   -0.041730f, +0.449214f, +0.080060f,
          -0.048405f, +0.438600f, +0.092868f,   -0.048429f, +0.425102f, +0.092918f,
          -0.043912f, +0.409507f, +0.084257f,   -0.036960f, +0.392603f, +0.070929f,
          -0.029681f, +0.375174f, +0.056977f,   -0.024184f, +0.358008f, +0.046446f,
          -0.022575f, +0.341889f, +0.043381f,   -0.026963f, +0.327606f, +0.051825f,
          -0.033718f, +0.456160f, +0.045901f,   -0.053504f, +0.449214f, +0.072838f,
          -0.062063f, +0.438600f, +0.084490f,   -0.062095f, +0.425102f, +0.084535f,
          -0.056304f, +0.409507f, +0.076655f,   -0.047392f, +0.392603f, +0.064527f,
          -0.038063f, +0.375174f, +0.051831f,   -0.031017f, +0.358008f, +0.042248f,
          -0.028959f, +0.341889f, +0.039455f,   -0.034591f, +0.327606f, +0.047133f,
          -0.040288f, +0.456160f, +0.040288f,   -0.063930f, +0.449214f, +0.063930f,
          -0.074156f, +0.438600f, +0.074156f,   -0.074196f, +0.425102f, +0.074196f,
          -0.067278f, +0.409507f, +0.067278f,   -0.056631f, +0.392603f, +0.056631f,
          -0.045486f, +0.375174f, +0.045486f,   -0.037072f, +0.358008f, +0.037072f,
          -0.034617f, +0.341889f, +0.034617f,   -0.041351f, +0.327606f, +0.041351f,
          -0.045902f, +0.456160f, +0.033718f,   -0.072838f, +0.449214f, +0.053504f,
          -0.084490f, +0.438600f, +0.062063f,   -0.084535f, +0.425102f, +0.062095f,
          -0.076655f, +0.409507f, +0.056304f,   -0.064527f, +0.392603f, +0.047392f,
          -0.051831f, +0.375174f, +0.038063f,   -0.042248f, +0.358008f, +0.031017f,
          -0.039455f, +0.341889f, +0.028959f,   -0.047133f, +0.327606f, +0.034591f,
          -0.050453f, +0.456160f, +0.026298f,   -0.080060f, +0.449214f, +0.041730f,
          -0.092868f, +0.438600f, +0.048405f,   -0.092918f, +0.425102f, +0.048429f,
          -0.084257f, +0.409507f, +0.043912f,   -0.070929f, +0.392603f, +0.036960f,
          -0.056977f, +0.375174f, +0.029681f,   -0.046446f, +0.358008f, +0.024184f,
          -0.043381f, +0.341889f, +0.022575f,   -0.051825f, +0.327606f, +0.026963f,
          -0.053835f, +0.456160f, +0.018134f,   -0.085428f, +0.449214f, +0.028775f,
          -0.099094f, +0.438600f, +0.033378f,   -0.099149f, +0.425102f, +0.033394f,
          -0.089908f, +0.409507f, +0.030278f,   -0.075687f, +0.392603f, +0.025484f,
          -0.060802f, +0.375174f, +0.020463f,   -0.049568f, +0.358008f, +0.016670f,
          -0.046300f, +0.341889f, +0.015559f,   -0.055315f, +0.327606f, +0.018581f,
          -0.055943f, +0.456160f, +0.009333f,   -0.088772f, +0.449214f, +0.014809f,
          -0.102973f, +0.438600f, +0.017178f,   -0.103031f, +0.425102f, +0.017186f,
          -0.093429f, +0.409507f, +0.015582f,   -0.078652f, +0.392603f, +0.013114f,
          -0.063186f, +0.375174f, +0.010529f,   -0.051513f, +0.358008f, +0.008576f,
          -0.048120f, +0.341889f, +0.008002f,   -0.057491f, +0.327606f, +0.009556f,
          -0.056669f, +0.456160f, -0.000000f,   -0.089924f, +0.449214f, -0.000000f,
          -0.104310f, +0.438600f, -0.000000f,   -0.104368f, +0.425102f, -0.000000f,
          -0.094642f, +0.409507f, -0.000000f,   -0.079674f, +0.392603f, -0.000000f,
          -0.064007f, +0.375174f, -0.000000f,   -0.052184f, +0.358008f, -0.000000f,
          -0.048748f, +0.341889f, -0.000000f,   -0.058241f, +0.327606f, -0.000000f,
          -0.055943f, +0.456160f, -0.009333f,   -0.088772f, +0.449214f, -0.014809f,
          -0.102973f, +0.438600f, -0.017178f,   -0.103031f, +0.425102f, -0.017186f,
          -0.093429f, +0.409507f, -0.015582f,   -0.078652f, +0.392603f, -0.013114f,
          -0.063186f, +0.375174f, -0.010529f,   -0.051513f, +0.358008f, -0.008576f,
          -0.048120f, +0.341889f, -0.008002f,   -0.057491f, +0.327606f, -0.009556f,
          -0.053835f, +0.456160f, -0.018134f,   -0.085428f, +0.449214f, -0.028775f,
          -0.099094f, +0.438600f, -0.033378f,   -0.099149f, +0.425102f, -0.033394f,
          -0.089908f, +0.409507f, -0.030278f,   -0.075687f, +0.392603f, -0.025484f,
          -0.060802f, +0.375174f, -0.020463f,   -0.049568f, +0.358008f, -0.016670f,
          -0.046300f, +0.341889f, -0.015559f,   -0.055315f, +0.327606f, -0.018581f,
          -0.050453f, +0.456160f, -0.026298f,   -0.080060f, +0.449214f, -0.041730f,
          -0.092868f, +0.438600f, -0.048405f,   -0.092918f, +0.425102f, -0.048429f,
          -0.084257f, +0.409507f, -0.043912f,   -0.070929f, +0.392603f, -0.036960f,
          -0.056977f, +0.375174f, -0.029681f,   -0.046446f, +0.358008f, -0.024184f,
          -0.043381f, +0.341889f, -0.022575f,   -0.051825f, +0.327606f, -0.026963f,
          -0.045902f, +0.456160f, -0.033718f,   -0.072838f, +0.449214f, -0.053504f,
          -0.084490f, +0.438600f, -0.062063f,   -0.084535f, +0.425102f, -0.062095f,
          -0.076655f, +0.409507f, -0.056304f,   -0.064527f, +0.392603f, -0.047392f,
          -0.051831f, +0.375174f, -0.038063f,   -0.042248f, +0.358008f, -0.031017f,
          -0.039455f, +0.341889f, -0.028959f,   -0.047133f, +0.327606f, -0.034591f,
          -0.040288f, +0.456160f, -0.040288f,   -0.063930f, +0.449214f, -0.063930f,
          -0.074156f, +0.438600f, -0.074156f,   -0.074196f, +0.425102f, -0.074196f,
          -0.067278f, +0.409507f, -0.067278f,   -0.056631f, +0.392603f, -0.056631f,
          -0.045486f, +0.375174f, -0.045486f,   -0.037072f, +0.358008f, -0.037072f,
          -0.034617f, +0.341889f, -0.034617f,   -0.041351f, +0.327606f, -0.041351f,
          -0.033718f, +0.456160f, -0.045902f,   -0.053504f, +0.449214f, -0.072838f,
          -0.062063f, +0.438600f, -0.084490f,   -0.062095f, +0.425102f, -0.084535f,
          -0.056304f, +0.409507f, -0.076655f,   -0.047392f, +0.392603f, -0.064527f,
          -0.038063f, +0.375174f, -0.051831f,   -0.031017f, +0.358008f, -0.042248f,
          -0.028959f, +0.341889f, -0.039455f,   -0.034591f, +0.327606f, -0.047133f,
          -0.026298f, +0.456160f, -0.050453f,   -0.041730f, +0.449214f, -0.080061f,
          -0.048405f, +0.438600f, -0.092868f,   -0.048429f, +0.425102f, -0.092918f,
          -0.043912f, +0.409507f, -0.084257f,   -0.036960f, +0.392603f, -0.070929f,
          -0.029681f, +0.375174f, -0.056977f,   -0.024184f, +0.358008f, -0.046446f,
          -0.022575f, +0.341889f, -0.043381f,   -0.026963f, +0.327606f, -0.051825f,
          -0.018134f, +0.456160f, -0.053835f,   -0.028775f, +0.449214f, -0.085428f,
          -0.033378f, +0.438600f, -0.099094f,   -0.033394f, +0.425102f, -0.099149f,
          -0.030278f, +0.409507f, -0.089908f,   -0.025484f, +0.392603f, -0.075687f,
          -0.020463f, +0.375174f, -0.060802f,   -0.016670f, +0.358008f, -0.049568f,
          -0.015559f, +0.341889f, -0.046300f,   -0.018581f, +0.327606f, -0.055315f,
          -0.009333f, +0.456160f, -0.055943f,   -0.014809f, +0.449214f, -0.088772f,
          -0.017178f, +0.438600f, -0.102973f,   -0.017186f, +0.425102f, -0.103031f,
          -0.015582f, +0.409507f, -0.093429f,   -0.013114f, +0.392603f, -0.078652f,
          -0.010529f, +0.375174f, -0.063186f,   -0.008576f, +0.358008f, -0.051513f,
          -0.008002f, +0.341889f, -0.048120f,   -0.009556f, +0.327606f, -0.057491f,
          +0.000000f, +0.456160f, -0.056669f,   +0.000000f, +0.449214f, -0.089924f,
          +0.000000f, +0.438600f, -0.104310f,   +0.000000f, +0.425102f, -0.104368f,
          +0.000000f, +0.409507f, -0.094642f,   +0.000000f, +0.392603f, -0.079674f,
          +0.000000f, +0.375174f, -0.064007f,   +0.000000f, +0.358008f, -0.052184f,
          +0.000000f, +0.341889f, -0.048748f,   +0.000000f, +0.327606f, -0.058241f,
          +0.009333f, +0.456160f, -0.055943f,   +0.014809f, +0.449214f, -0.088772f,
          +0.017178f, +0.438600f, -0.102973f,   +0.017186f, +0.425102f, -0.103031f,
          +0.015582f, +0.409507f, -0.093429f,   +0.013114f, +0.392603f, -0.078652f,
          +0.010529f, +0.375174f, -0.063186f,   +0.008576f, +0.358008f, -0.051513f,
          +0.008002f, +0.341889f, -0.048120f,   +0.009556f, +0.327606f, -0.057491f,
          +0.018134f, +0.456160f, -0.053835f,   +0.028775f, +0.449214f, -0.085428f,
          +0.033378f, +0.438600f, -0.099094f,   +0.033394f, +0.425102f, -0.099149f,
          +0.030278f, +0.409507f, -0.089908f,   +0.025484f, +0.392603f, -0.075687f,
          +0.020463f, +0.375174f, -0.060802f,   +0.016670f, +0.358008f, -0.049568f,
          +0.015559f, +0.341889f, -0.046300f,   +0.018581f, +0.327606f, -0.055315f,
          +0.026298f, +0.456160f, -0.050453f,   +0.041730f, +0.449214f, -0.080061f,
          +0.048405f, +0.438600f, -0.092868f,   +0.048429f, +0.425102f, -0.092918f,
          +0.043912f, +0.409507f, -0.084257f,   +0.036960f, +0.392603f, -0.070929f,
          +0.029681f, +0.375174f, -0.056977f,   +0.024184f, +0.358008f, -0.046446f,
          +0.022575f, +0.341889f, -0.043381f,   +0.026963f, +0.327606f, -0.051825f,
          +0.033718f, +0.456160f, -0.045902f,   +0.053504f, +0.449214f, -0.072838f,
          +0.062063f, +0.438600f, -0.084490f,   +0.062095f, +0.425102f, -0.084535f,
          +0.056304f, +0.409507f, -0.076655f,   +0.047392f, +0.392603f, -0.064527f,
          +0.038063f, +0.375174f, -0.051831f,   +0.031017f, +0.358008f, -0.042248f,
          +0.028959f, +0.341889f, -0.039455f,   +0.034591f, +0.327606f, -0.047133f,
          +0.040288f, +0.456160f, -0.040288f,   +0.063930f, +0.449214f, -0.063930f,
          +0.074156f, +0.438600f, -0.074156f,   +0.074196f, +0.425102f, -0.074196f,
          +0.067278f, +0.409507f, -0.067278f,   +0.056631f, +0.392603f, -0.056631f,
          +0.045486f, +0.375174f, -0.045486f,   +0.037072f, +0.358008f, -0.037072f,
          +0.034617f, +0.341889f, -0.034617f,   +0.041351f, +0.327606f, -0.041351f,
          +0.045902f, +0.456160f, -0.033718f,   +0.072838f, +0.449214f, -0.053504f,
          +0.084490f, +0.438600f, -0.062063f,   +0.084535f, +0.425102f, -0.062095f,
          +0.076655f, +0.409507f, -0.056304f,   +0.064527f, +0.392603f, -0.047392f,
          +0.051831f, +0.375174f, -0.038063f,   +0.042248f, +0.358008f, -0.031017f,
          +0.039455f, +0.341889f, -0.028959f,   +0.047133f, +0.327606f, -0.034591f,
          +0.050453f, +0.456160f, -0.026298f,   +0.080060f, +0.449214f, -0.041730f,
          +0.092868f, +0.438600f, -0.048405f,   +0.092918f, +0.425102f, -0.048429f,
          +0.084257f, +0.409507f, -0.043912f,   +0.070929f, +0.392603f, -0.036960f,
          +0.056977f, +0.375174f, -0.029681f,   +0.046446f, +0.358008f, -0.024184f,
          +0.043381f, +0.341889f, -0.022575f,   +0.051825f, +0.327606f, -0.026963f,
          +0.053835f, +0.456160f, -0.018134f,   +0.085428f, +0.449214f, -0.028775f,
          +0.099094f, +0.438600f, -0.033378f,   +0.099149f, +0.425102f, -0.033394f,
          +0.089908f, +0.409507f, -0.030278f,   +0.075687f, +0.392603f, -0.025484f,
          +0.060802f, +0.375174f, -0.020463f,   +0.049568f, +0.358008f, -0.016670f,
          +0.046300f, +0.341889f, -0.015559f,   +0.055315f, +0.327606f, -0.018581f,
          +0.055943f, +0.456160f, -0.009333f,   +0.088772f, +0.449214f, -0.014809f,
          +0.102973f, +0.438600f, -0.017178f,   +0.103031f, +0.425102f, -0.017186f,
          +0.093429f, +0.409507f, -0.015582f,   +0.078652f, +0.392603f, -0.013114f,
          +0.063186f, +0.375174f, -0.010529f,   +0.051513f, +0.358008f, -0.008576f,
          +0.048120f, +0.341889f, -0.008002f,   +0.057491f, +0.327606f, -0.009556f,
          +0.080315f, +0.315725f, +0.013350f,   +0.081363f, +0.315725f, -0.000000f,
          +0.112452f, +0.305941f, +0.018692f,   +0.113920f, +0.305941f, -0.000000f,
          +0.151144f, +0.297729f, +0.025123f,   +0.153116f, +0.297729f, -0.000000f,
          +0.193630f, +0.290565f, +0.032185f,   +0.196156f, +0.290565f, -0.000000f,
          +0.237150f, +0.283925f, +0.039419f,   +0.240245f, +0.283925f, -0.000000f,
          +0.278946f, +0.277286f, +0.046367f,   +0.282586f, +0.277286f, -0.000000f,
          +0.316257f, +0.270122f, +0.052569f,   +0.320384f, +0.270122f, -0.000000f,
          +0.346326f, +0.261910f, +0.057567f,   +0.350845f, +0.261910f, -0.000000f,
          +0.366390f, +0.252126f, +0.060902f,   +0.371171f, +0.252126f, -0.000000f,
          +0.373691f, +0.240245f, +0.062115f,   +0.378567f, +0.240245f, -0.000000f,
          +0.077275f, +0.315725f, +0.025958f,   +0.108196f, +0.305941f, +0.036345f,
          +0.145423f, +0.297729f, +0.048850f,   +0.186302f, +0.290565f, +0.062582f,
          +0.228175f, +0.283925f, +0.076648f,   +0.268389f, +0.277286f, +0.090156f,
          +0.304288f, +0.270122f, +0.102215f,   +0.333218f, +0.261910f, +0.111933f,
          +0.352523f, +0.252126f, +0.118418f,   +0.359548f, +0.240245f, +0.120778f,
          +0.072400f, +0.315725f, +0.037668f,   +0.101370f, +0.305941f, +0.052740f,
          +0.136248f, +0.297729f, +0.070887f,   +0.174547f, +0.290565f, +0.090812f,
          +0.213779f, +0.283925f, +0.111224f,   +0.251456f, +0.277286f, +0.130826f,
          +0.285090f, +0.270122f, +0.148325f,   +0.312196f, +0.261910f, +0.162427f,
          +0.330283f, +0.252126f, +0.171837f,   +0.336864f, +0.240245f, +0.175261f,
          +0.065845f, +0.315725f, +0.048323f,   +0.092193f, +0.305941f, +0.067659f,
          +0.123914f, +0.297729f, +0.090939f,   +0.158745f, +0.290565f, +0.116501f,
          +0.194425f, +0.283925f, +0.142686f,   +0.228691f, +0.277286f, +0.167833f,
          +0.259280f, +0.270122f, +0.190283f,   +0.283931f, +0.261910f, +0.208374f,
          +0.300381f, +0.252126f, +0.220446f,   +0.306367f, +0.240245f, +0.224839f,
          +0.057768f, +0.315725f, +0.057768f,   +0.080883f, +0.305941f, +0.080883f,
          +0.108712f, +0.297729f, +0.108712f,   +0.139271f, +0.290565f, +0.139271f,
          +0.170574f, +0.283925f, +0.170573f,   +0.200636f, +0.277286f, +0.200636f,
          +0.227472f, +0.270122f, +0.227472f,   +0.249100f, +0.261910f, +0.249100f,
          +0.263531f, +0.252126f, +0.263531f,   +0.268783f, +0.240245f, +0.268783f,
          +0.048323f, +0.315725f, +0.065845f,   +0.067659f, +0.305941f, +0.092193f,
          +0.090939f, +0.297729f, +0.123914f,   +0.116501f, +0.290565f, +0.158745f,
          +0.142686f, +0.283925f, +0.194425f,   +0.167833f, +0.277286f, +0.228691f,
          +0.190283f, +0.270122f, +0.259280f,   +0.208374f, +0.261910f, +0.283931f,
          +0.220446f, +0.252126f, +0.300381f,   +0.224839f, +0.240245f, +0.306367f,
          +0.037668f, +0.315725f, +0.072400f,   +0.052740f, +0.305941f, +0.101370f,
          +0.070887f, +0.297729f, +0.136248f,   +0.090812f, +0.290565f, +0.174547f,
          +0.111224f, +0.283925f, +0.213779f,   +0.130826f, +0.277286f, +0.251456f,
          +0.148325f, +0.270122f, +0.285090f,   +0.162427f, +0.261910f, +0.312196f,
          +0.171837f, +0.252126f, +0.330283f,   +0.175261f, +0.240245f, +0.336864f,
          +0.025958f, +0.315725f, +0.077275f,   +0.036345f, +0.305941f, +0.108196f,
          +0.048850f, +0.297729f, +0.145423f,   +0.062582f, +0.290565f, +0.186302f,
          +0.076648f, +0.283925f, +0.228175f,   +0.090156f, +0.277286f, +0.268389f,
          +0.102215f, +0.270122f, +0.304288f,   +0.111933f, +0.261910f, +0.333218f,
          +0.118418f, +0.252126f, +0.352523f,   +0.120778f, +0.240245f, +0.359548f,
          +0.013350f, +0.315725f, +0.080315f,   +0.018692f, +0.305941f, +0.112452f,
          +0.025123f, +0.297729f, +0.151144f,   +0.032185f, +0.290565f, +0.193630f,
          +0.039419f, +0.283925f, +0.237150f,   +0.046367f, +0.277286f, +0.278946f,
          +0.052569f, +0.270122f, +0.316257f,   +0.057566f, +0.261910f, +0.346326f,
          +0.060902f, +0.252126f, +0.366390f,   +0.062115f, +0.240245f, +0.373691f,
          -0.000000f, +0.315725f, +0.081363f,   -0.000000f, +0.305941f, +0.113920f,
          -0.000000f, +0.297729f, +0.153116f,   -0.000000f, +0.290565f, +0.196156f,
          -0.000000f, +0.283925f, +0.240245f,   -0.000000f, +0.277286f, +0.282586f,
          -0.000000f, +0.270122f, +0.320384f,   -0.000000f, +0.261910f, +0.350845f,
          -0.000000f, +0.252126f, +0.371171f,   -0.000000f, +0.240245f, +0.378567f,
          -0.013350f, +0.315725f, +0.080315f,   -0.018692f, +0.305941f, +0.112452f,
          -0.025123f, +0.297729f, +0.151144f,   -0.032185f, +0.290565f, +0.193630f,
          -0.039419f, +0.283925f, +0.237150f,   -0.046367f, +0.277286f, +0.278946f,
          -0.052569f, +0.270122f, +0.316257f,   -0.057567f, +0.261910f, +0.346326f,
          -0.060902f, +0.252126f, +0.366390f,   -0.062115f, +0.240245f, +0.373691f,
          -0.025958f, +0.315725f, +0.077275f,   -0.036345f, +0.305941f, +0.108196f,
          -0.048850f, +0.297729f, +0.145423f,   -0.062582f, +0.290565f, +0.186302f,
          -0.076648f, +0.283925f, +0.228175f,   -0.090156f, +0.277286f, +0.268389f,
          -0.102215f, +0.270122f, +0.304288f,   -0.111933f, +0.261910f, +0.333218f,
          -0.118418f, +0.252126f, +0.352523f,   -0.120778f, +0.240245f, +0.359548f,
          -0.037668f, +0.315725f, +0.072400f,   -0.052740f, +0.305941f, +0.101370f,
          -0.070887f, +0.297729f, +0.136248f,   -0.090812f, +0.290565f, +0.174547f,
          -0.111224f, +0.283925f, +0.213779f,   -0.130826f, +0.277286f, +0.251456f,
          -0.148325f, +0.270122f, +0.285090f,   -0.162427f, +0.261910f, +0.312196f,
          -0.171837f, +0.252126f, +0.330282f,   -0.175261f, +0.240245f, +0.336864f,
          -0.048323f, +0.315725f, +0.065845f,   -0.067659f, +0.305941f, +0.092193f,
          -0.090939f, +0.297729f, +0.123914f,   -0.116501f, +0.290565f, +0.158745f,
          -0.142686f, +0.283925f, +0.194425f,   -0.167833f, +0.277286f, +0.228691f,
          -0.190283f, +0.270122f, +0.259280f,   -0.208374f, +0.261910f, +0.283931f,
          -0.220446f, +0.252126f, +0.300381f,   -0.224839f, +0.240245f, +0.306367f,
          -0.057768f, +0.315725f, +0.057768f,   -0.080883f, +0.305941f, +0.080883f,
          -0.108712f, +0.297729f, +0.108712f,   -0.139271f, +0.290565f, +0.139271f,
          -0.170574f, +0.283925f, +0.170573f,   -0.200636f, +0.277286f, +0.200636f,
          -0.227472f, +0.270122f, +0.227472f,   -0.249100f, +0.261910f, +0.249100f,
          -0.263531f, +0.252126f, +0.263531f,   -0.268783f, +0.240245f, +0.268783f,
          -0.065845f, +0.315725f, +0.048323f,   -0.092193f, +0.305941f, +0.067659f,
          -0.123914f, +0.297729f, +0.090939f,   -0.158745f, +0.290565f, +0.116501f,
          -0.194425f, +0.283925f, +0.142686f,   -0.228691f, +0.277286f, +0.167833f,
          -0.259280f, +0.270122f, +0.190283f,   -0.283931f, +0.261910f, +0.208374f,
          -0.300381f, +0.252126f, +0.220446f,   -0.306367f, +0.240245f, +0.224839f,
          -0.072400f, +0.315725f, +0.037668f,   -0.101370f, +0.305941f, +0.052740f,
          -0.136248f, +0.297729f, +0.070887f,   -0.174547f, +0.290565f, +0.090812f,
          -0.213780f, +0.283925f, +0.111224f,   -0.251456f, +0.277286f, +0.130826f,
          -0.285091f, +0.270122f, +0.148325f,   -0.312196f, +0.261910f, +0.162427f,
          -0.330283f, +0.252126f, +0.171837f,   -0.336864f, +0.240245f, +0.175261f,
          -0.077275f, +0.315725f, +0.025958f,   -0.108196f, +0.305941f, +0.036345f,
          -0.145423f, +0.297729f, +0.048850f,   -0.186302f, +0.290565f, +0.062582f,
          -0.228175f, +0.283925f, +0.076648f,   -0.268389f, +0.277286f, +0.090156f,
          -0.304289f, +0.270122f, +0.102215f,   -0.333218f, +0.261910f, +0.111933f,
          -0.352523f, +0.252126f, +0.118418f,   -0.359548f, +0.240245f, +0.120778f,
          -0.080315f, +0.315725f, +0.013350f,   -0.112452f, +0.305941f, +0.018692f,
          -0.151144f, +0.297729f, +0.025123f,   -0.193630f, +0.290565f, +0.032185f,
          -0.237150f, +0.283925f, +0.039419f,   -0.278946f, +0.277286f, +0.046367f,
          -0.316257f, +0.270122f, +0.052569f,   -0.346326f, +0.261910f, +0.057566f,
          -0.366390f, +0.252126f, +0.060902f,   -0.373691f, +0.240245f, +0.062115f,
          -0.081363f, +0.315725f, -0.000000f,   -0.113920f, +0.305941f, -0.000000f,
          -0.153116f, +0.297729f, -0.000000f,   -0.196156f, +0.290565f, -0.000000f,
          -0.240245f, +0.283925f, -0.000000f,   -0.282586f, +0.277286f, -0.000000f,
          -0.320384f, +0.270122f, -0.000000f,   -0.350845f, +0.261910f, -0.000000f,
          -0.371171f, +0.252126f, -0.000000f,   -0.378567f, +0.240245f, -0.000000f,
          -0.080315f, +0.315725f, -0.013350f,   -0.112452f, +0.305941f, -0.018692f,
          -0.151144f, +0.297729f, -0.025123f,   -0.193630f, +0.290565f, -0.032185f,
          -0.237150f, +0.283925f, -0.039419f,   -0.278946f, +0.277286f, -0.046367f,
          -0.316257f, +0.270122f, -0.052569f,   -0.346326f, +0.261910f, -0.057567f,
          -0.366390f, +0.252126f, -0.060902f,   -0.373691f, +0.240245f, -0.062115f,
          -0.077275f, +0.315725f, -0.025958f,   -0.108196f, +0.305941f, -0.036345f,
          -0.145423f, +0.297729f, -0.048850f,   -0.186302f, +0.290565f, -0.062582f,
          -0.228175f, +0.283925f, -0.076648f,   -0.268389f, +0.277286f, -0.090156f,
          -0.304288f, +0.270122f, -0.102215f,   -0.333218f, +0.261910f, -0.111933f,
          -0.352523f, +0.252126f, -0.118418f,   -0.359548f, +0.240245f, -0.120778f,
          -0.072400f, +0.315725f, -0.037668f,   -0.101370f, +0.305941f, -0.052740f,
          -0.136248f, +0.297729f, -0.070887f,   -0.174547f, +0.290565f, -0.090812f,
          -0.213779f, +0.283925f, -0.111224f,   -0.251456f, +0.277286f, -0.130826f,
          -0.285090f, +0.270122f, -0.148325f,   -0.312196f, +0.261910f, -0.162427f,
          -0.330283f, +0.252126f, -0.171837f,   -0.336864f, +0.240245f, -0.175261f,
          -0.065845f, +0.315725f, -0.048323f,   -0.092193f, +0.305941f, -0.067659f,
          -0.123914f, +0.297729f, -0.090939f,   -0.158745f, +0.290565f, -0.116501f,
          -0.194425f, +0.283925f, -0.142686f,   -0.228691f, +0.277286f, -0.167833f,
          -0.259280f, +0.270122f, -0.190283f,   -0.283931f, +0.261910f, -0.208374f,
          -0.300381f, +0.252126f, -0.220446f,   -0.306367f, +0.240245f, -0.224839f,
          -0.057768f, +0.315725f, -0.057768f,   -0.080883f, +0.305941f, -0.080883f,
          -0.108712f, +0.297729f, -0.108712f,   -0.139271f, +0.290565f, -0.139271f,
          -0.170574f, +0.283925f, -0.170574f,   -0.200636f, +0.277286f, -0.200636f,
          -0.227472f, +0.270122f, -0.227473f,   -0.249100f, +0.261910f, -0.249100f,
          -0.263531f, +0.252126f, -0.263532f,   -0.268783f, +0.240245f, -0.268783f,
          -0.048323f, +0.315725f, -0.065845f,   -0.067659f, +0.305941f, -0.092193f,
          -0.090939f, +0.297729f, -0.123914f,   -0.116501f, +0.290565f, -0.158745f,
          -0.142686f, +0.283925f, -0.194425f,   -0.167833f, +0.277286f, -0.228691f,
          -0.190283f, +0.270122f, -0.259280f,   -0.208374f, +0.261910f, -0.283931f,
          -0.220446f, +0.252126f, -0.300381f,   -0.224839f, +0.240245f, -0.306367f,
          -0.037668f, +0.315725f, -0.072400f,   -0.052740f, +0.305941f, -0.101370f,
          -0.070887f, +0.297729f, -0.136249f,   -0.090812f, +0.290565f, -0.174547f,
          -0.111224f, +0.283925f, -0.213780f,   -0.130826f, +0.277286f, -0.251456f,
          -0.148325f, +0.270122f, -0.285091f,   -0.162427f, +0.261910f, -0.312196f,
          -0.171837f, +0.252126f, -0.330283f,   -0.175261f, +0.240245f, -0.336864f,
          -0.025958f, +0.315725f, -0.077275f,   -0.036345f, +0.305941f, -0.108196f,
          -0.048850f, +0.297729f, -0.145423f,   -0.062582f, +0.290565f, -0.186302f,
          -0.076648f, +0.283925f, -0.228175f,   -0.090156f, +0.277286f, -0.268389f,
          -0.102215f, +0.270122f, -0.304289f,   -0.111933f, +0.261910f, -0.333218f,
          -0.118418f, +0.252126f, -0.352523f,   -0.120778f, +0.240245f, -0.359548f,
          -0.013350f, +0.315725f, -0.080315f,   -0.018692f, +0.305941f, -0.112452f,
          -0.025123f, +0.297729f, -0.151144f,   -0.032185f, +0.290565f, -0.193630f,
          -0.039419f, +0.283925f, -0.237151f,   -0.046367f, +0.277286f, -0.278946f,
          -0.052569f, +0.270122f, -0.316257f,   -0.057566f, +0.261910f, -0.346326f,
          -0.060902f, +0.252126f, -0.366390f,   -0.062115f, +0.240245f, -0.373691f,
          +0.000000f, +0.315725f, -0.081363f,   +0.000000f, +0.305941f, -0.113920f,
          +0.000000f, +0.297729f, -0.153116f,   +0.000000f, +0.290565f, -0.196156f,
          +0.000000f, +0.283925f, -0.240245f,   +0.000000f, +0.277286f, -0.282586f,
          +0.000000f, +0.270122f, -0.320384f,   +0.000000f, +0.261910f, -0.350845f,
          +0.000000f, +0.252126f, -0.371171f,   +0.000000f, +0.240245f, -0.378567f,
          +0.013350f, +0.315725f, -0.080315f,   +0.018692f, +0.305941f, -0.112452f,
          +0.025123f, +0.297729f, -0.151144f,   +0.032185f, +0.290565f, -0.193630f,
          +0.039419f, +0.283925f, -0.237151f,   +0.046367f, +0.277286f, -0.278946f,
          +0.052569f, +0.270122f, -0.316257f,   +0.057567f, +0.261910f, -0.346326f,
          +0.060902f, +0.252126f, -0.366390f,   +0.062115f, +0.240245f, -0.373691f,
          +0.025958f, +0.315725f, -0.077275f,   +0.036345f, +0.305941f, -0.108196f,
          +0.048850f, +0.297729f, -0.145423f,   +0.062582f, +0.290565f, -0.186302f,
          +0.076648f, +0.283925f, -0.228175f,   +0.090156f, +0.277286f, -0.268389f,
          +0.102215f, +0.270122f, -0.304288f,   +0.111933f, +0.261910f, -0.333218f,
          +0.118418f, +0.252126f, -0.352523f,   +0.120778f, +0.240245f, -0.359548f,
          +0.037668f, +0.315725f, -0.072400f,   +0.052740f, +0.305941f, -0.101370f,
          +0.070887f, +0.297729f, -0.136248f,   +0.090812f, +0.290565f, -0.174547f,
          +0.111224f, +0.283925f, -0.213780f,   +0.130826f, +0.277286f, -0.251456f,
          +0.148325f, +0.270122f, -0.285090f,   +0.162427f, +0.261910f, -0.312196f,
          +0.171837f, +0.252126f, -0.330283f,   +0.175261f, +0.240245f, -0.336864f,
          +0.048323f, +0.315725f, -0.065845f,   +0.067659f, +0.305941f, -0.092193f,
          +0.090939f, +0.297729f, -0.123914f,   +0.116501f, +0.290565f, -0.158745f,
          +0.142686f, +0.283925f, -0.194425f,   +0.167833f, +0.277286f, -0.228691f,
          +0.190283f, +0.270122f, -0.259280f,   +0.208374f, +0.261910f, -0.283931f,
          +0.220446f, +0.252126f, -0.300381f,   +0.224839f, +0.240245f, -0.306367f,
          +0.057768f, +0.315725f, -0.057768f,   +0.080883f, +0.305941f, -0.080883f,
          +0.108712f, +0.297729f, -0.108712f,   +0.139271f, +0.290565f, -0.139271f,
          +0.170574f, +0.283925f, -0.170574f,   +0.200636f, +0.277286f, -0.200636f,
          +0.227472f, +0.270122f, -0.227473f,   +0.249100f, +0.261910f, -0.249100f,
          +0.263531f, +0.252126f, -0.263532f,   +0.268783f, +0.240245f, -0.268783f,
          +0.065845f, +0.315725f, -0.048323f,   +0.092193f, +0.305941f, -0.067659f,
          +0.123914f, +0.297729f, -0.090939f,   +0.158745f, +0.290565f, -0.116501f,
          +0.194425f, +0.283925f, -0.142686f,   +0.228691f, +0.277286f, -0.167833f,
          +0.259280f, +0.270122f, -0.190283f,   +0.283931f, +0.261910f, -0.208374f,
          +0.300381f, +0.252126f, -0.220446f,   +0.306367f, +0.240245f, -0.224839f,
          +0.072400f, +0.315725f, -0.037668f,   +0.101370f, +0.305941f, -0.052740f,
          +0.136248f, +0.297729f, -0.070887f,   +0.174547f, +0.290565f, -0.090812f,
          +0.213780f, +0.283925f, -0.111224f,   +0.251456f, +0.277286f, -0.130826f,
          +0.285091f, +0.270122f, -0.148325f,   +0.312196f, +0.261910f, -0.162427f,
          +0.330283f, +0.252126f, -0.171837f,   +0.336864f, +0.240245f, -0.175261f,
          +0.077275f, +0.315725f, -0.025958f,   +0.108196f, +0.305941f, -0.036345f,
          +0.145423f, +0.297729f, -0.048850f,   +0.186302f, +0.290565f, -0.062582f,
          +0.228175f, +0.283925f, -0.076648f,   +0.268389f, +0.277286f, -0.090156f,
          +0.304289f, +0.270122f, -0.102215f,   +0.333218f, +0.261910f, -0.111933f,
          +0.352523f, +0.252126f, -0.118418f,   +0.359548f, +0.240245f, -0.120778f,
          +0.080315f, +0.315725f, -0.013350f,   +0.112452f, +0.305941f, -0.018692f,
          +0.151144f, +0.297729f, -0.025123f,   +0.193630f, +0.290565f, -0.032185f,
          +0.237150f, +0.283925f, -0.039419f,   +0.278946f, +0.277286f, -0.046367f,
          +0.316257f, +0.270122f, -0.052569f,   +0.346326f, +0.261910f, -0.057567f,
          +0.366390f, +0.252126f, -0.060902f,   +0.373691f, +0.240245f, -0.062115f,
    };
    static const float teapot_normals[] = {
          -0.929101f, -0.369037f, -0.024144f,   -0.913873f, -0.369281f, -0.168726f,
          -0.943499f, -0.297869f, -0.145200f,   -0.954789f, -0.297266f, +0.003303f,
          -0.985425f, -0.078929f, -0.150690f,   -0.996932f, -0.078152f, +0.004387f,
          -0.944422f, +0.296282f, -0.142418f,   -0.954744f, +0.297363f, +0.006170f,
          -0.596179f, +0.798237f, -0.085948f,   -0.601050f, +0.799174f, +0.007767f,
          -0.001582f, +0.999975f, +0.006869f,   +0.000934f, +0.999975f, +0.006962f,
          +0.395568f, +0.915930f, +0.067803f,   +0.402309f, +0.915488f, +0.005388f,
          +0.612070f, +0.784358f, +0.100764f,   +0.620975f, +0.783819f, +0.004283f,
          +0.742965f, +0.658369f, +0.120640f,   +0.753181f, +0.657804f, +0.003561f,
          +0.830872f, +0.540099f, +0.133960f,   +0.841949f, +0.539548f, +0.003051f,
          +0.884526f, +0.439859f, +0.155367f,   +0.898111f, +0.439490f, +0.015675f,
          -0.875661f, -0.369712f, -0.310694f,   -0.908704f, -0.298053f, -0.292269f,
          -0.949306f, -0.078738f, -0.304333f,   -0.909905f, +0.296974f, -0.289618f,
          -0.574271f, +0.798926f, -0.178691f,   -0.002223f, +0.999975f, +0.006645f,
          +0.379773f, +0.915997f, +0.129316f,   +0.588159f, +0.784642f, +0.195974f,
          +0.714334f, +0.658712f, +0.236274f,   +0.799082f, +0.540489f, +0.263323f,
          +0.848741f, +0.440280f, +0.292902f,   -0.815923f, -0.370067f, -0.444207f,
          -0.851427f, -0.298250f, -0.431415f,   -0.889689f, -0.078653f, -0.449742f,
          -0.852883f, +0.297660f, -0.428939f,   -0.538307f, +0.799540f, -0.266386f,
          -0.002884f, +0.999976f, +0.006337f,   +0.354587f, +0.916079f, +0.187263f,
          +0.549800f, +0.784884f, +0.285792f,   +0.668050f, +0.659087f, +0.345418f,
          +0.747591f, +0.540837f, +0.385492f,   +0.792061f, +0.440656f, +0.422447f,
          -0.736582f, -0.370292f, -0.565978f,   -0.773571f, -0.298368f, -0.559075f,
          -0.808532f, -0.078530f, -0.583189f,   -0.775308f, +0.298095f, -0.556810f,
          -0.489553f, +0.800026f, -0.346839f,   -0.003551f, +0.999976f, +0.005961f,
          +0.320918f, +0.916130f, +0.240245f,   +0.498301f, +0.785022f, +0.368017f,
          +0.605776f, +0.659302f, +0.445372f,   +0.678108f, +0.541073f, +0.497403f,
          +0.716348f, +0.440886f, +0.540800f,   -0.639653f, -0.370321f, -0.673577f,
          -0.677217f, -0.298362f, -0.672575f,   -0.708009f, -0.078446f, -0.701833f,
          -0.679204f, +0.298425f, -0.670540f,   -0.429342f, +0.800349f, -0.418458f,
          -0.004213f, +0.999976f, +0.005530f,   +0.279738f, +0.916086f, +0.287286f,
          +0.434983f, +0.785058f, +0.440992f,   +0.529134f, +0.659334f, +0.534131f,
          +0.592496f, +0.541115f, +0.596777f,   +0.623553f, +0.440936f, +0.645567f,
          -0.527126f, -0.370240f, -0.764892f,   -0.564423f, -0.298195f, -0.769745f,
          -0.590268f, -0.078244f, -0.803406f,   -0.566659f, +0.298573f, -0.767953f,
          -0.358971f, +0.800469f, -0.479988f,   -0.004864f, +0.999975f, +0.005053f,
          +0.231799f, +0.915976f, +0.327501f,   +0.361238f, +0.784889f, +0.503444f,
          +0.439726f, +0.659150f, +0.610051f,   +0.492535f, +0.540961f, +0.681741f,
          +0.515596f, +0.440804f, +0.734746f,   -0.401161f, -0.369941f, -0.837981f,
          -0.437356f, -0.297921f, -0.848506f,   -0.457600f, -0.078016f, -0.885729f,
          -0.439795f, +0.298676f, -0.846979f,   -0.279631f, +0.800505f, -0.530093f,
          -0.005487f, +0.999975f, +0.004541f,   +0.177966f, +0.915796f, +0.360064f,
          +0.278340f, +0.784576f, +0.554047f,   +0.339196f, +0.658774f, +0.671537f,
          +0.380132f, +0.540576f, +0.750518f,   +0.394528f, +0.440494f, +0.806420f,
          -0.264443f, -0.369547f, -0.890789f,   -0.298655f, -0.297495f, -0.906809f,
          -0.312767f, -0.077766f, -0.946641f,   -0.301274f, +0.298731f, -0.905535f,
          -0.192936f, +0.800450f, -0.567500f,   -0.006062f, +0.999974f, +0.003993f,
          +0.119330f, +0.915530f, +0.384143f,   +0.187969f, +0.784132f, +0.591442f,
          +0.229574f, +0.658268f, +0.716923f,   +0.257564f, +0.540045f, +0.801256f,
          +0.262897f, +0.440043f, +0.858631f,   -0.120558f, -0.369144f, -0.921520f,
          -0.151832f, -0.297009f, -0.942726f,   -0.159472f, -0.077507f, -0.984155f,
          -0.154598f, +0.298823f, -0.941703f,   -0.100971f, +0.800470f, -0.590806f,
          -0.006569f, +0.999973f, +0.003423f,   +0.057300f, +0.915238f, +0.398818f,
          +0.092358f, +0.783663f, +0.614282f,   +0.113627f, +0.657703f, +0.744658f,
          +0.127936f, +0.539485f, +0.832219f,   +0.124117f, +0.439569f, +0.889592f,
          +0.022853f, -0.372257f, -0.927848f,   -0.003205f, -0.298230f, -0.954489f,
          -0.004296f, -0.079380f, -0.996835f,   -0.006090f, +0.296162f, -0.955118f,
          -0.007712f, +0.798862f, -0.601465f,   -0.006935f, +0.999975f, +0.001199f,
          -0.005378f, +0.915373f, +0.402571f,   -0.004280f, +0.783727f, +0.621091f,
          -0.003560f, +0.657766f, +0.753214f,   -0.003051f, +0.539543f, +0.841953f,
          -0.015675f, +0.439490f, +0.898111f,   +0.156584f, -0.397182f, -0.904283f,
          +0.136021f, -0.320381f, -0.937472f,   +0.144170f, -0.106603f, -0.983794f,
          +0.139595f, +0.267304f, -0.953447f,   +0.086347f, +0.786874f, -0.611043f,
          -0.006919f, +0.999973f, -0.002364f,   -0.067965f, +0.914711f, +0.398353f,
          -0.100812f, +0.783066f, +0.613713f,   -0.120646f, +0.657691f, +0.743564f,
          -0.133959f, +0.539911f, +0.830994f,   -0.155367f, +0.439850f, +0.884530f,
          +0.291422f, -0.438828f, -0.850002f,   +0.276635f, -0.363765f, -0.889465f,
          +0.294769f, -0.160047f, -0.942070f,   +0.292020f, +0.210036f, -0.933065f,
          +0.189573f, +0.764341f, -0.616315f,   -0.006973f, +0.999972f, -0.002841f,
          -0.132006f, +0.911869f, +0.388676f,   -0.197434f, +0.780402f, +0.593290f,
          -0.236804f, +0.656429f, +0.716257f,   -0.263440f, +0.539794f, +0.799514f,
          -0.292907f, +0.440249f, +0.848756f,   +0.418292f, -0.476347f, -0.773385f,
          +0.409230f, -0.408487f, -0.815885f,   +0.437851f, -0.216251f, -0.872652f,
          +0.442602f, +0.145992f, -0.884754f,   +0.299071f, +0.736543f, -0.606681f,
          -0.006948f, +0.999970f, -0.003420f,   -0.194813f, +0.908723f, +0.369149f,
          -0.289998f, +0.777438f, +0.558114f,   -0.346988f, +0.655059f, +0.671191f,
          -0.385854f, +0.539566f, +0.748321f,   -0.422460f, +0.440596f, +0.792087f,
          +0.534923f, -0.497544f, -0.682867f,   +0.530546f, -0.438123f, -0.725651f,
          +0.568812f, -0.253962f, -0.782276f,   +0.582387f, +0.100726f, -0.806647f,
          +0.404978f, +0.715070f, -0.569797f,   -0.006740f, +0.999969f, -0.004049f,
          -0.253238f, +0.906456f, +0.337947f,   -0.375301f, +0.775319f, +0.507966f,
          -0.448124f, +0.654031f, +0.609449f,   -0.498049f, +0.539390f, +0.678974f,
          -0.540824f, +0.440808f, +0.716378f,   +0.641586f, -0.498044f, -0.583369f,
          +0.640897f, -0.445517f, -0.625112f,   +0.686783f, -0.263377f, -0.677467f,
          +0.705053f, +0.089482f, -0.703486f,   +0.494206f, +0.709619f, -0.502197f,
          -0.006306f, +0.999969f, -0.004664f,   -0.303975f, +0.905778f, +0.295239f,
          -0.450417f, +0.774710f, +0.443789f,   -0.537708f, +0.653700f, +0.532491f,
          -0.597625f, +0.539291f, +0.593304f,   -0.645598f, +0.440852f, +0.623581f,
          +0.737723f, -0.478839f, -0.475897f,   +0.740581f, -0.430237f, -0.516175f,
          +0.791248f, -0.243538f, -0.560906f,   +0.806128f, +0.114092f, -0.580638f,
          +0.556979f, +0.722219f, -0.410091f,   -0.005681f, +0.999970f, -0.005208f,
          -0.344501f, +0.906828f, +0.242861f,   -0.513127f, +0.775645f, +0.367526f,
          -0.613755f, +0.654077f, +0.442140f,   -0.682632f, +0.539296f, +0.493126f,
          -0.734781f, +0.440719f, +0.515620f,   +0.819981f, -0.445223f, -0.359733f,
          +0.827086f, -0.396797f, -0.398097f,   +0.878896f, -0.200830f, -0.432677f,
          +0.881254f, +0.164955f, -0.442923f,   +0.591488f, +0.746208f, -0.305476f,
          -0.004945f, +0.999972f, -0.005666f,   -0.373850f, +0.909095f, +0.183800f,
          -0.561973f, +0.777730f, +0.281642f,   -0.674597f, +0.654978f, +0.340474f,
          -0.751273f, +0.539298f, +0.380456f,   -0.806449f, +0.440430f, +0.394539f,
          +0.882497f, -0.407204f, -0.235338f,   +0.895008f, -0.354548f, -0.270658f,
          +0.944337f, -0.147990f, -0.293815f,   +0.928269f, +0.224438f, -0.296555f,
          +0.603809f, +0.771652f, -0.199920f,   -0.004190f, +0.999973f, -0.006065f,
          -0.392487f, +0.911748f, +0.121115f,   -0.596332f, +0.780183f, +0.188949f,
          -0.718824f, +0.656061f, +0.229946f,   -0.801746f, +0.539268f, +0.257668f,
          -0.858653f, +0.439997f, +0.262902f,   +0.919356f, -0.378856f, -0.106079f,
          +0.938318f, -0.317693f, -0.136497f,   +0.983583f, -0.102770f, -0.148331f,
          +0.950599f, +0.272514f, -0.148652f,   +0.604122f, +0.790590f, -0.100017f,
          -0.003483f, +0.999973f, -0.006447f,   -0.401959f, +0.913864f, +0.057283f,
          -0.616176f, +0.782183f, +0.092291f,   -0.745410f, +0.656857f, +0.113590f,
          -0.832420f, +0.539178f, +0.127926f,   -0.889599f, +0.439555f, +0.124115f,
          +0.929023f, -0.369005f, +0.027426f,   +0.953871f, -0.300216f, +0.001075f,
          +0.996650f, -0.081775f, -0.001052f,   +0.955923f, +0.293593f, -0.003912f,
          +0.602986f, +0.797723f, -0.006777f,   -0.000937f, +0.999976f, -0.006878f,
          -0.402778f, +0.915281f, -0.005528f,   -0.621271f, +0.783583f, -0.004393f,
          -0.753297f, +0.657670f, -0.003610f,   -0.841990f, +0.539485f, -0.003061f,
          -0.898111f, +0.439490f, -0.015675f,   +0.913873f, -0.369280f, +0.168726f,
          +0.943500f, -0.297869f, +0.145200f,   +0.985425f, -0.078929f, +0.150690f,
          +0.944422f, +0.296283f, +0.142418f,   +0.596180f, +0.798237f, +0.085949f,
          +0.001582f, +0.999975f, -0.006869f,   -0.395568f, +0.915930f, -0.067803f,
          -0.612070f, +0.784357f, -0.100765f,   -0.742964f, +0.658369f, -0.120640f,
          -0.830872f, +0.540098f, -0.133960f,   -0.884526f, +0.439859f, -0.155367f,
          +0.875661f, -0.369712f, +0.310694f,   +0.908707f, -0.298045f, +0.292269f,
          +0.949306f, -0.078749f, +0.304331f,   +0.909905f, +0.296975f, +0.289618f,
          +0.574271f, +0.798926f, +0.178692f,   +0.002223f, +0.999975f, -0.006645f,
          -0.379773f, +0.915997f, -0.129316f,   -0.588159f, +0.784642f, -0.195974f,
          -0.714334f, +0.658712f, -0.236274f,   -0.799082f, +0.540490f, -0.263323f,
          -0.848741f, +0.440280f, -0.292902f,   +0.815923f, -0.370067f, +0.444207f,
          +0.851431f, -0.298233f, +0.431418f,   +0.889689f, -0.078653f, +0.449742f,
          +0.852893f, +0.297626f, +0.428942f,   +0.538307f, +0.799540f, +0.266386f,
          +0.002884f, +0.999976f, -0.006337f,   -0.354587f, +0.916079f, -0.187263f,
          -0.549800f, +0.784884f, -0.285792f,   -0.668050f, +0.659087f, -0.345418f,
          -0.747591f, +0.540837f, -0.385492f,   -0.792061f, +0.440656f, -0.422447f,
          +0.736582f, -0.370292f, +0.565978f,   +0.773571f, -0.298368f, +0.559075f,
          +0.808531f, -0.078515f, +0.583192f,   +0.775306f, +0.298100f, +0.556810f,
          +0.489567f, +0.800014f, +0.346846f,   +0.003550f, +0.999976f, -0.005961f,
          -0.320925f, +0.916126f, -0.240249f,   -0.498301f, +0.785022f, -0.368017f,
          -0.605776f, +0.659302f, -0.445372f,   -0.678108f, +0.541073f, -0.497403f,
          -0.716348f, +0.440886f, -0.540800f,   +0.639653f, -0.370321f, +0.673577f,
          +0.677217f, -0.298362f, +0.672575f,   +0.708009f, -0.078446f, +0.701833f,
          +0.679192f, +0.298476f, +0.670530f,   +0.429342f, +0.800349f, +0.418458f,
          +0.004215f, +0.999976f, -0.005530f,   -0.279738f, +0.916086f, -0.287286f,
          -0.434996f, +0.785044f, -0.441003f,   -0.529134f, +0.659334f, -0.534131f,
          -0.592496f, +0.541115f, -0.596777f,   -0.623553f, +0.440936f, -0.645567f,
          +0.527126f, -0.370240f, +0.764892f,   +0.564423f, -0.298195f, +0.769745f,
          +0.590268f, -0.078244f, +0.803406f,   +0.566659f, +0.298573f, +0.767953f,
          +0.358957f, +0.800483f, +0.479976f,   +0.004863f, +0.999975f, -0.005053f,
          -0.231792f, +0.915980f, -0.327494f,   -0.361243f, +0.784881f, -0.503453f,
          -0.439726f, +0.659150f, -0.610051f,   -0.492535f, +0.540961f, -0.681741f,
          -0.515596f, +0.440804f, -0.734746f,   +0.401161f, -0.369941f, +0.837981f,
          +0.437356f, -0.297921f, +0.848506f,   +0.457600f, -0.078016f, +0.885729f,
          +0.439795f, +0.298676f, +0.846979f,   +0.279631f, +0.800505f, +0.530093f,
          +0.005487f, +0.999975f, -0.004541f,   -0.177966f, +0.915796f, -0.360064f,
          -0.278340f, +0.784576f, -0.554047f,   -0.339196f, +0.658774f, -0.671537f,
          -0.380132f, +0.540576f, -0.750518f,   -0.394528f, +0.440494f, -0.806420f,
          +0.264443f, -0.369547f, +0.890789f,   +0.298655f, -0.297495f, +0.906809f,
          +0.312770f, -0.077738f, +0.946642f,   +0.301282f, +0.298695f, +0.905544f,
          +0.192936f, +0.800450f, +0.567500f,   +0.006062f, +0.999974f, -0.003993f,
          -0.119330f, +0.915530f, -0.384143f,   -0.187969f, +0.784132f, -0.591442f,
          -0.229574f, +0.658268f, -0.716923f,   -0.257564f, +0.540045f, -0.801256f,
          -0.262897f, +0.440043f, -0.858631f,   +0.120558f, -0.369144f, +0.921520f,
          +0.151832f, -0.297009f, +0.942726f,   +0.159471f, -0.077452f, +0.984160f,
          +0.154598f, +0.298823f, +0.941703f,   +0.100982f, +0.800434f, +0.590853f,
          +0.006569f, +0.999973f, -0.003423f,   -0.057300f, +0.915238f, -0.398818f,
          -0.092358f, +0.783663f, -0.614282f,   -0.113627f, +0.657703f, -0.744658f,
          -0.127936f, +0.539485f, -0.832219f,   -0.124120f, +0.439574f, -0.889589f,
          -0.024144f, -0.369037f, +0.929101f,   +0.003303f, -0.297266f, +0.954789f,
          +0.004390f, -0.078124f, +0.996934f,   +0.006170f, +0.297363f, +0.954745f,
          +0.007765f, +0.799155f, +0.601074f,   +0.006962f, +0.999975f, -0.000934f,
          +0.005389f, +0.915488f, -0.402309f,   +0.004283f, +0.783819f, -0.620975f,
          +0.003561f, +0.657804f, -0.753181f,   +0.003051f, +0.539548f, -0.841949f,
          +0.015679f, +0.439500f, -0.898106f,   -0.168726f, -0.369281f, +0.913873f,
          -0.145200f, -0.297869f, +0.943500f,   -0.150693f, -0.078873f, +0.985429f,
          -0.142418f, +0.296283f, +0.944422f,   -0.085953f, +0.798201f, +0.596228f,
          +0.006869f, +0.999975f, +0.001582f,   +0.067803f, +0.915930f, -0.395568f,
          +0.100765f, +0.784358f, -0.612070f,   +0.120640f, +0.658369f, -0.742965f,
          +0.133960f, +0.540099f, -0.830872f,   +0.155367f, +0.439859f, -0.884526f,
          -0.310694f, -0.369712f, +0.875661f,   -0.292269f, -0.298045f, +0.908707f,
          -0.304331f, -0.078748f, +0.949306f,   -0.289620f, +0.297011f, +0.909892f,
          -0.178701f, +0.798907f, +0.574293f,   +0.006645f, +0.999975f, +0.002223f,
          +0.129310f, +0.916002f, -0.379762f,   +0.195974f, +0.784633f, -0.588171f,
          +0.236274f, +0.658712f, -0.714334f,   +0.263323f, +0.540489f, -0.799082f,
          +0.292902f, +0.440280f, -0.848741f,   -0.444207f, -0.370067f, +0.815923f,
          -0.431418f, -0.298233f, +0.851431f,   -0.449742f, -0.078653f, +0.889689f,
          -0.428942f, +0.297626f, +0.852893f,   -0.266386f, +0.799540f, +0.538307f,
          +0.006337f, +0.999976f, +0.002884f,   +0.187254f, +0.916088f, -0.354567f,
          +0.285792f, +0.784884f, -0.549800f,   +0.345425f, +0.659065f, -0.668068f,
          +0.385492f, +0.540837f, -0.747591f,   +0.422447f, +0.440656f, -0.792061f,
          -0.565978f, -0.370292f, +0.736582f,   -0.559075f, -0.298368f, +0.773571f,
          -0.583192f, -0.078515f, +0.808531f,   -0.556810f, +0.298100f, +0.775306f,
          -0.346846f, +0.800014f, +0.489567f,   +0.005960f, +0.999976f, +0.003551f,
          +0.240249f, +0.916126f, -0.320927f,   +0.368013f, +0.785031f, -0.498290f,
          +0.445379f, +0.659292f, -0.605783f,   +0.497403f, +0.541073f, -0.678108f,
          +0.540800f, +0.440886f, -0.716348f,   -0.673577f, -0.370321f, +0.639653f,
          -0.672575f, -0.298362f, +0.677217f,   -0.701833f, -0.078446f, +0.708009f,
          -0.670530f, +0.298476f, +0.679192f,   -0.418458f, +0.800349f, +0.429342f,
          +0.005531f, +0.999976f, +0.004214f,   +0.287286f, +0.916086f, -0.279738f,
          +0.441004f, +0.785044f, -0.434997f,   +0.534131f, +0.659334f, -0.529134f,
          +0.596777f, +0.541115f, -0.592496f,   +0.645567f, +0.440936f, -0.623553f,
          -0.764892f, -0.370240f, +0.527126f,   -0.769746f, -0.298185f, +0.564426f,
          -0.803403f, -0.078259f, +0.590270f,   -0.767953f, +0.298573f, +0.566659f,
          -0.479976f, +0.800483f, +0.358957f,   +0.005053f, +0.999975f, +0.004863f,
          +0.327495f, +0.915980f, -0.231794f,   +0.503451f, +0.784883f, -0.361242f,
          +0.610051f, +0.659150f, -0.439726f,   +0.681741f, +0.540961f, -0.492535f,
          +0.734746f, +0.440804f, -0.515596f,   -0.837981f, -0.369941f, +0.401161f,
          -0.848512f, -0.297904f, +0.437357f,   -0.885729f, -0.078016f, +0.457600f,
          -0.846987f, +0.298643f, +0.439801f,   -0.530093f, +0.800505f, +0.279631f,
          +0.004541f, +0.999975f, +0.005487f,   +0.360064f, +0.915796f, -0.177966f,
          +0.554047f, +0.784576f, -0.278340f,   +0.671537f, +0.658774f, -0.339196f,
          +0.750518f, +0.540576f, -0.380132f,   +0.806420f, +0.440494f, -0.394528f,
          -0.890789f, -0.369547f, +0.264443f,   -0.906809f, -0.297495f, +0.298655f,
          -0.946643f, -0.077755f, +0.312765f,   -0.905540f, +0.298717f, +0.301274f,
          -0.567499f, +0.800450f, +0.192936f,   +0.003993f, +0.999974f, +0.006062f,
          +0.384144f, +0.915529f, -0.119330f,   +0.591442f, +0.784133f, -0.187969f,
          +0.716923f, +0.658268f, -0.229574f,   +0.801256f, +0.540046f, -0.257564f,
          +0.858631f, +0.440043f, -0.262897f,   -0.921519f, -0.369145f, +0.120558f,
          -0.942726f, -0.297009f, +0.151833f,   -0.984155f, -0.077507f, +0.159473f,
          -0.941703f, +0.298822f, +0.154599f,   -0.590805f, +0.800471f, +0.100971f,
          +0.003423f, +0.999973f, +0.006569f,   +0.398818f, +0.915238f, -0.057300f,
          +0.614282f, +0.783663f, -0.092358f,   +0.744658f, +0.657703f, -0.113627f,
          +0.832219f, +0.539484f, -0.127936f,   +0.889592f, +0.439569f, -0.124117f,
          +0.893498f, +0.426092f, +0.141800f,   +0.904789f, +0.425859f, +0.001080f,
          +0.897753f, +0.416861f, +0.142361f,   +0.909083f, +0.416614f, +0.000970f,
          +0.904782f, +0.401025f, +0.143347f,   +0.916184f, +0.400756f, +0.000843f,
          +0.914493f, +0.377829f, +0.144736f,   +0.925987f, +0.377555f, +0.000711f,
          +0.926649f, +0.346207f, +0.146501f,   +0.938264f, +0.345921f, +0.000565f,
          +0.940800f, +0.304672f, +0.148562f,   +0.952554f, +0.304370f, +0.000402f,
          +0.956083f, +0.251339f, +0.150781f,   +0.967977f, +0.251040f, +0.000225f,
          +0.970961f, +0.183983f, +0.152923f,   +0.982987f, +0.183675f, +0.000026f,
          +0.982881f, +0.100258f, +0.154574f,   +0.994991f, +0.099962f, -0.000195f,
          +0.987662f, -0.024971f, +0.154595f,   +0.999678f, -0.025366f, -0.000912f,
          +0.859729f, +0.426524f, +0.280968f,   +0.863859f, +0.417270f, +0.282193f,
          +0.870658f, +0.401424f, +0.284279f,   +0.880044f, +0.378211f, +0.287192f,
          +0.891797f, +0.346551f, +0.290862f,   +0.905469f, +0.304985f, +0.295143f,
          +0.920244f, +0.251587f, +0.299760f,   +0.934632f, +0.184136f, +0.304233f,
          +0.946164f, +0.100305f, +0.307753f,   +0.950884f, -0.025139f, +0.308525f,
          +0.804729f, +0.426931f, +0.412482f,   +0.808625f, +0.417666f, +0.414344f,
          +0.815027f, +0.401800f, +0.417478f,   +0.823852f, +0.378576f, +0.421838f,
          +0.834908f, +0.346884f, +0.427318f,   +0.847766f, +0.305270f, +0.433710f,
          +0.861654f, +0.251824f, +0.440611f,   +0.875190f, +0.184287f, +0.447305f,
          +0.886045f, +0.100363f, +0.452606f,   +0.890583f, -0.025274f, +0.454117f,
          +0.730347f, +0.427196f, +0.533008f,   +0.733904f, +0.417928f, +0.535463f,
          +0.739747f, +0.402051f, +0.539565f,   +0.747795f, +0.378813f, +0.545256f,
          +0.757867f, +0.347106f, +0.552408f,   +0.769589f, +0.305454f, +0.560741f,
          +0.782246f, +0.251973f, +0.569737f,   +0.794587f, +0.184387f, +0.578476f,
          +0.804495f, +0.100386f, +0.585415f,   +0.808740f, -0.025392f, +0.587619f,
          +0.638549f, +0.427284f, +0.640065f,   +0.641686f, +0.418013f, +0.643043f,
          +0.646822f, +0.402128f, +0.648008f,   +0.653886f, +0.378885f, +0.654889f,
          +0.662727f, +0.347171f, +0.663524f,   +0.673017f, +0.305505f, +0.673583f,
          +0.684128f, +0.252004f, +0.684443f,   +0.694963f, +0.184397f, +0.694999f,
          +0.703673f, +0.100365f, +0.703400f,   +0.707518f, -0.025472f, +0.706236f,
          +0.531277f, +0.427184f, +0.731614f,   +0.533907f, +0.417915f, +0.735044f,
          +0.538210f, +0.402027f, +0.740746f,   +0.544122f, +0.378778f, +0.748638f,
          +0.551506f, +0.347068f, +0.758541f,   +0.560098f, +0.305414f, +0.770073f,
          +0.569385f, +0.251908f, +0.782523f,   +0.578439f, +0.184315f, +0.794629f,
          +0.585735f, +0.100291f, +0.804273f,   +0.589090f, -0.025544f, +0.807664f,
          +0.410561f, +0.426909f, +0.805723f,   +0.412626f, +0.417636f, +0.809518f,
          +0.415981f, +0.401754f, +0.815814f,   +0.420580f, +0.378512f, +0.824525f,
          +0.426329f, +0.346804f, +0.835446f,   +0.433007f, +0.305174f, +0.848159f,
          +0.440227f, +0.251690f, +0.861889f,   +0.447273f, +0.184129f, +0.875239f,
          +0.452960f, +0.100163f, +0.885886f,   +0.455738f, -0.025613f, +0.889745f,
          +0.278906f, +0.426494f, +0.860415f,   +0.280347f, +0.417224f, +0.864482f,
          +0.282671f, +0.401347f, +0.871216f,   +0.285843f, +0.378111f, +0.880526f,
          +0.289794f, +0.346424f, +0.892194f,   +0.294383f, +0.304815f, +0.905774f,
          +0.299343f, +0.251384f, +0.920435f,   +0.304198f, +0.183872f, +0.934695f,
          +0.308138f, +0.099981f, +0.946073f,   +0.310274f, -0.025702f, +0.950300f,
          +0.139651f, +0.426059f, +0.893852f,   +0.140436f, +0.416789f, +0.898089f,
          +0.141665f, +0.400916f, +0.905095f,   +0.143325f, +0.377690f, +0.914772f,
          +0.145386f, +0.346013f, +0.926897f,   +0.147771f, +0.304424f, +0.941004f,
          +0.150347f, +0.251038f, +0.956230f,   +0.152881f, +0.183591f, +0.971042f,
          +0.154968f, +0.099790f, +0.982867f,   +0.156414f, -0.025805f, +0.987354f,
          -0.001080f, +0.425859f, +0.904789f,   -0.000970f, +0.416614f, +0.909083f,
          -0.000843f, +0.400756f, +0.916184f,   -0.000711f, +0.377555f, +0.925987f,
          -0.000565f, +0.345921f, +0.938264f,   -0.000402f, +0.304370f, +0.952554f,
          -0.000225f, +0.251040f, +0.967977f,   -0.000026f, +0.183675f, +0.982987f,
          +0.000195f, +0.099962f, +0.994991f,   +0.000912f, -0.025366f, +0.999678f,
          -0.141800f, +0.426092f, +0.893498f,   -0.142361f, +0.416861f, +0.897753f,
          -0.143347f, +0.401025f, +0.904782f,   -0.144735f, +0.377830f, +0.914492f,
          -0.146500f, +0.346206f, +0.926649f,   -0.148562f, +0.304671f, +0.940800f,
          -0.150781f, +0.251339f, +0.956083f,   -0.152923f, +0.183983f, +0.970961f,
          -0.154574f, +0.100258f, +0.982881f,   -0.154595f, -0.024971f, +0.987662f,
          -0.280968f, +0.426524f, +0.859729f,   -0.282193f, +0.417270f, +0.863859f,
          -0.284279f, +0.401423f, +0.870658f,   -0.287193f, +0.378213f, +0.880043f,
          -0.290862f, +0.346551f, +0.891797f,   -0.295143f, +0.304983f, +0.905470f,
          -0.299760f, +0.251587f, +0.920244f,   -0.304233f, +0.184136f, +0.934632f,
          -0.307753f, +0.100305f, +0.946164f,   -0.308525f, -0.025139f, +0.950884f,
          -0.412482f, +0.426931f, +0.804729f,   -0.414344f, +0.417666f, +0.808625f,
          -0.417478f, +0.401800f, +0.815027f,   -0.421840f, +0.378574f, +0.823852f,
          -0.427323f, +0.346888f, +0.834904f,   -0.433711f, +0.305268f, +0.847766f,
          -0.440611f, +0.251824f, +0.861654f,   -0.447305f, +0.184287f, +0.875190f,
          -0.452606f, +0.100363f, +0.886045f,   -0.454117f, -0.025274f, +0.890583f,
          -0.533008f, +0.427196f, +0.730347f,   -0.535463f, +0.417928f, +0.733904f,
          -0.539565f, +0.402051f, +0.739747f,   -0.545256f, +0.378807f, +0.747798f,
          -0.552408f, +0.347106f, +0.757867f,   -0.560741f, +0.305460f, +0.769587f,
          -0.569737f, +0.251973f, +0.782246f,   -0.578475f, +0.184387f, +0.794586f,
          -0.585415f, +0.100386f, +0.804495f,   -0.587619f, -0.025392f, +0.808740f,
          -0.640065f, +0.427284f, +0.638549f,   -0.643043f, +0.418013f, +0.641686f,
          -0.648008f, +0.402128f, +0.646822f,   -0.654887f, +0.378888f, +0.653886f,
          -0.663521f, +0.347166f, +0.662733f,   -0.673582f, +0.305508f, +0.673018f,
          -0.684443f, +0.252004f, +0.684128f,   -0.694999f, +0.184397f, +0.694963f,
          -0.703400f, +0.100365f, +0.703673f,   -0.706236f, -0.025472f, +0.707518f,
          -0.731616f, +0.427182f, +0.531276f,   -0.735045f, +0.417917f, +0.533904f,
          -0.740746f, +0.402027f, +0.538210f,   -0.748637f, +0.378784f, +0.544119f,
          -0.758541f, +0.347068f, +0.551506f,   -0.770073f, +0.305408f, +0.560101f,
          -0.782523f, +0.251908f, +0.569385f,   -0.794629f, +0.184315f, +0.578439f,
          -0.804273f, +0.100291f, +0.585735f,   -0.807664f, -0.025544f, +0.589090f,
          -0.805723f, +0.426906f, +0.410563f,   -0.809518f, +0.417636f, +0.412626f,
          -0.815814f, +0.401757f, +0.415979f,   -0.824525f, +0.378512f, +0.420580f,
          -0.835447f, +0.346807f, +0.426325f,   -0.848161f, +0.305170f, +0.433006f,
          -0.861889f, +0.251690f, +0.440227f,   -0.875239f, +0.184129f, +0.447273f,
          -0.885886f, +0.100163f, +0.452960f,   -0.889745f, -0.025613f, +0.455738f,
          -0.860415f, +0.426494f, +0.278906f,   -0.864482f, +0.417223f, +0.280348f,
          -0.871215f, +0.401349f, +0.282671f,   -0.880526f, +0.378111f, +0.285843f,
          -0.892194f, +0.346424f, +0.289794f,   -0.905774f, +0.304815f, +0.294383f,
          -0.920435f, +0.251384f, +0.299343f,   -0.934695f, +0.183872f, +0.304198f,
          -0.946073f, +0.099981f, +0.308138f,   -0.950300f, -0.025702f, +0.310274f,
          -0.893852f, +0.426059f, +0.139651f,   -0.898090f, +0.416789f, +0.140436f,
          -0.905095f, +0.400916f, +0.141665f,   -0.914772f, +0.377690f, +0.143325f,
          -0.926897f, +0.346013f, +0.145386f,   -0.941004f, +0.304425f, +0.147771f,
          -0.956230f, +0.251037f, +0.150347f,   -0.971042f, +0.183591f, +0.152881f,
          -0.982867f, +0.099790f, +0.154968f,   -0.987354f, -0.025805f, +0.156414f,
          -0.904789f, +0.425859f, -0.001080f,   -0.909083f, +0.416614f, -0.000970f,
          -0.916184f, +0.400756f, -0.000843f,   -0.925987f, +0.377555f, -0.000711f,
          -0.938264f, +0.345921f, -0.000565f,   -0.952554f, +0.304370f, -0.000402f,
          -0.967977f, +0.251040f, -0.000225f,   -0.982987f, +0.183675f, -0.000026f,
          -0.994991f, +0.099962f, +0.000195f,   -0.999678f, -0.025366f, +0.000912f,
          -0.893498f, +0.426093f, -0.141800f,   -0.897753f, +0.416860f, -0.142361f,
          -0.904782f, +0.401025f, -0.143347f,   -0.914492f, +0.377830f, -0.144735f,
          -0.926649f, +0.346206f, -0.146500f,   -0.940800f, +0.304671f, -0.148562f,
          -0.956083f, +0.251339f, -0.150781f,   -0.970961f, +0.183983f, -0.152923f,
          -0.982881f, +0.100258f, -0.154574f,   -0.987662f, -0.024971f, -0.154595f,
          -0.859728f, +0.426526f, -0.280969f,   -0.863859f, +0.417270f, -0.282193f,
          -0.870659f, +0.401421f, -0.284279f,   -0.880043f, +0.378213f, -0.287193f,
          -0.891797f, +0.346551f, -0.290862f,   -0.905470f, +0.304983f, -0.295143f,
          -0.920244f, +0.251587f, -0.299760f,   -0.934632f, +0.184136f, -0.304233f,
          -0.946164f, +0.100305f, -0.307753f,   -0.950884f, -0.025139f, -0.308525f,
          -0.804729f, +0.426931f, -0.412482f,   -0.808623f, +0.417668f, -0.414346f,
          -0.815027f, +0.401799f, -0.417479f,   -0.823852f, +0.378576f, -0.421838f,
          -0.834906f, +0.346885f, -0.427320f,   -0.847766f, +0.305268f, -0.433711f,
          -0.861654f, +0.251824f, -0.440611f,   -0.875190f, +0.184287f, -0.447305f,
          -0.886045f, +0.100363f, -0.452606f,   -0.890583f, -0.025274f, -0.454117f,
          -0.730347f, +0.427196f, -0.533008f,   -0.733904f, +0.417928f, -0.535463f,
          -0.739747f, +0.402051f, -0.539565f,   -0.747795f, +0.378813f, -0.545256f,
          -0.757867f, +0.347106f, -0.552408f,   -0.769589f, +0.305454f, -0.560741f,
          -0.782246f, +0.251973f, -0.569737f,   -0.794587f, +0.184387f, -0.578476f,
          -0.804495f, +0.100386f, -0.585415f,   -0.808740f, -0.025392f, -0.587619f,
          -0.638549f, +0.427284f, -0.640065f,   -0.641686f, +0.418013f, -0.643043f,
          -0.646822f, +0.402128f, -0.648008f,   -0.653886f, +0.378888f, -0.654887f,
          -0.662730f, +0.347169f, -0.663523f,   -0.673017f, +0.305505f, -0.673583f,
          -0.684128f, +0.252004f, -0.684443f,   -0.694963f, +0.184397f, -0.694999f,
          -0.703673f, +0.100365f, -0.703400f,   -0.707518f, -0.025472f, -0.706236f,
          -0.531277f, +0.427184f, -0.731614f,   -0.533907f, +0.417915f, -0.735044f,
          -0.538210f, +0.402027f, -0.740746f,   -0.544119f, +0.378784f, -0.748637f,
          -0.551506f, +0.347068f, -0.758541f,   -0.560101f, +0.305408f, -0.770073f,
          -0.569385f, +0.251908f, -0.782523f,   -0.578439f, +0.184315f, -0.794629f,
          -0.585735f, +0.100291f, -0.804273f,   -0.589090f, -0.025544f, -0.807664f,
          -0.410561f, +0.426909f, -0.805723f,   -0.412626f, +0.417636f, -0.809518f,
          -0.415981f, +0.401754f, -0.815814f,   -0.420580f, +0.378512f, -0.824525f,
          -0.426325f, +0.346807f, -0.835447f,   -0.433006f, +0.305170f, -0.848161f,
          -0.440227f, +0.251690f, -0.861889f,   -0.447273f, +0.184129f, -0.875239f,
          -0.452960f, +0.100163f, -0.885886f,   -0.455738f, -0.025613f, -0.889745f,
          -0.278906f, +0.426494f, -0.860415f,   -0.280347f, +0.417224f, -0.864482f,
          -0.282671f, +0.401347f, -0.871216f,   -0.285843f, +0.378111f, -0.880526f,
          -0.289794f, +0.346424f, -0.892194f,   -0.294383f, +0.304815f, -0.905774f,
          -0.299343f, +0.251384f, -0.920435f,   -0.304198f, +0.183872f, -0.934695f,
          -0.308138f, +0.099981f, -0.946073f,   -0.310274f, -0.025702f, -0.950300f,
          -0.139656f, +0.426056f, -0.893853f,   -0.140436f, +0.416789f, -0.898089f,
          -0.141665f, +0.400916f, -0.905095f,   -0.143325f, +0.377690f, -0.914772f,
          -0.145386f, +0.346013f, -0.926897f,   -0.147771f, +0.304424f, -0.941004f,
          -0.150347f, +0.251038f, -0.956230f,   -0.152881f, +0.183591f, -0.971042f,
          -0.154968f, +0.099790f, -0.982867f,   -0.156414f, -0.025805f, -0.987354f,
          +0.001080f, +0.425859f, -0.904789f,   +0.000967f, +0.416608f, -0.909086f,
          +0.000843f, +0.400756f, -0.916184f,   +0.000711f, +0.377555f, -0.925987f,
          +0.000565f, +0.345921f, -0.938264f,   +0.000402f, +0.304370f, -0.952554f,
          +0.000225f, +0.251040f, -0.967977f,   +0.000026f, +0.183675f, -0.982987f,
          -0.000195f, +0.099962f, -0.994991f,   -0.000912f, -0.025366f, -0.999678f,
          +0.141804f, +0.426095f, -0.893496f,   +0.142364f, +0.416858f, -0.897754f,
          +0.143347f, +0.401025f, -0.904782f,   +0.144735f, +0.377830f, -0.914492f,
          +0.146500f, +0.346206f, -0.926649f,   +0.148562f, +0.304671f, -0.940800f,
          +0.150781f, +0.251339f, -0.956083f,   +0.152923f, +0.183983f, -0.970961f,
          +0.154574f, +0.100258f, -0.982881f,   +0.154595f, -0.024971f, -0.987662f,
          +0.280968f, +0.426524f, -0.859729f,   +0.282193f, +0.417270f, -0.863859f,
          +0.284279f, +0.401423f, -0.870658f,   +0.287193f, +0.378213f, -0.880043f,
          +0.290862f, +0.346551f, -0.891797f,   +0.295143f, +0.304983f, -0.905470f,
          +0.299760f, +0.251587f, -0.920244f,   +0.304233f, +0.184136f, -0.934632f,
          +0.307753f, +0.100305f, -0.946164f,   +0.308525f, -0.025139f, -0.950884f,
          +0.412482f, +0.426931f, -0.804729f,   +0.414344f, +0.417666f, -0.808625f,
          +0.417478f, +0.401800f, -0.815027f,   +0.421838f, +0.378576f, -0.823852f,
          +0.427320f, +0.346885f, -0.834906f,   +0.433711f, +0.305268f, -0.847766f,
          +0.440611f, +0.251824f, -0.861654f,   +0.447305f, +0.184287f, -0.875190f,
          +0.452606f, +0.100363f, -0.886045f,   +0.454117f, -0.025274f, -0.890583f,
          +0.533008f, +0.427196f, -0.730347f,   +0.535463f, +0.417928f, -0.733904f,
          +0.539565f, +0.402051f, -0.739747f,   +0.545256f, +0.378813f, -0.747795f,
          +0.552408f, +0.347106f, -0.757867f,   +0.560741f, +0.305454f, -0.769589f,
          +0.569737f, +0.251973f, -0.782246f,   +0.578475f, +0.184387f, -0.794586f,
          +0.585415f, +0.100386f, -0.804495f,   +0.587619f, -0.025392f, -0.808740f,
          +0.640065f, +0.427284f, -0.638549f,   +0.643043f, +0.418013f, -0.641686f,
          +0.648008f, +0.402128f, -0.646822f,   +0.654887f, +0.378888f, -0.653886f,
          +0.663523f, +0.347169f, -0.662730f,   +0.673583f, +0.305505f, -0.673017f,
          +0.684443f, +0.252004f, -0.684128f,   +0.694999f, +0.184397f, -0.694963f,
          +0.703400f, +0.100365f, -0.703673f,   +0.706236f, -0.025472f, -0.707518f,
          +0.731614f, +0.427184f, -0.531277f,   +0.735044f, +0.417915f, -0.533907f,
          +0.740746f, +0.402027f, -0.538210f,   +0.748637f, +0.378784f, -0.544119f,
          +0.758541f, +0.347068f, -0.551506f,   +0.770073f, +0.305408f, -0.560101f,
          +0.782523f, +0.251908f, -0.569385f,   +0.794629f, +0.184315f, -0.578439f,
          +0.804273f, +0.100291f, -0.585735f,   +0.807664f, -0.025544f, -0.589090f,
          +0.805723f, +0.426909f, -0.410561f,   +0.809518f, +0.417636f, -0.412626f,
          +0.815814f, +0.401754f, -0.415981f,   +0.824525f, +0.378512f, -0.420580f,
          +0.835447f, +0.346807f, -0.426325f,   +0.848161f, +0.305170f, -0.433006f,
          +0.861889f, +0.251690f, -0.440227f,   +0.875239f, +0.184129f, -0.447273f,
          +0.885886f, +0.100163f, -0.452960f,   +0.889745f, -0.025613f, -0.455738f,
          +0.860415f, +0.426494f, -0.278906f,   +0.864482f, +0.417224f, -0.280347f,
          +0.871216f, +0.401347f, -0.282671f,   +0.880526f, +0.378111f, -0.285843f,
          +0.892194f, +0.346424f, -0.289794f,   +0.905774f, +0.304815f, -0.294384f,
          +0.920435f, +0.251384f, -0.299343f,   +0.934695f, +0.183872f, -0.304198f,
          +0.946073f, +0.099981f, -0.308138f,   +0.950300f, -0.025702f, -0.310274f,
          +0.893852f, +0.426059f, -0.139652f,   +0.898090f, +0.416789f, -0.140437f,
          +0.905095f, +0.400916f, -0.141665f,   +0.914772f, +0.377690f, -0.143326f,
          +0.926897f, +0.346013f, -0.145387f,   +0.941005f, +0.304425f, -0.147771f,
          +0.956230f, +0.251037f, -0.150347f,   +0.971042f, +0.183591f, -0.152881f,
          +0.982867f, +0.099790f, -0.154968f,   +0.987354f, -0.025805f, -0.156414f,
          +0.966506f, -0.207912f, +0.150459f,   +0.978061f, -0.208314f, -0.001700f,
          +0.907749f, -0.395129f, +0.140940f,   +0.918527f, -0.395353f, -0.001966f,
          +0.832139f, -0.539381f, +0.128891f,   +0.842003f, -0.539469f, -0.002105f,
          +0.757718f, -0.641989f, +0.117107f,   +0.766723f, -0.641974f, -0.002163f,
          +0.695339f, -0.710633f, +0.107259f,   +0.703635f, -0.710558f, -0.002190f,
          +0.651230f, -0.752226f, +0.100277f,   +0.659037f, -0.752108f, -0.002226f,
          +0.632079f, -0.768789f, +0.097161f,   +0.639715f, -0.768609f, -0.002326f,
          +0.653392f, -0.750353f, +0.100244f,   +0.661391f, -0.750037f, -0.002594f,
          +0.764045f, -0.634487f, +0.116879f,   +0.773612f, -0.633650f, -0.003367f,
          +0.931136f, -0.334989f, +0.144110f,   +0.942561f, -0.334026f, -0.002465f,
          +0.930565f, -0.208329f, +0.301077f,   +0.873902f, -0.395689f, +0.282355f,
          +0.801018f, -0.539956f, +0.258491f,   +0.729316f, -0.642519f, +0.235090f,
          +0.669234f, -0.711115f, +0.215503f,   +0.626779f, -0.752653f, +0.201647f,
          +0.608375f, -0.769180f, +0.195554f,   +0.628982f, -0.750724f, +0.201980f,
          +0.735814f, -0.634754f, +0.235934f,   +0.896687f, -0.335097f, +0.289244f,
          +0.871619f, -0.208694f, +0.443540f,   +0.818478f, -0.396196f, +0.416080f,
          +0.750138f, -0.540494f, +0.380998f,   +0.682936f, -0.643020f, +0.346589f,
          +0.626646f, -0.711563f, +0.317794f,   +0.586890f, -0.753055f, +0.297436f,
          +0.569682f, -0.769553f, +0.288530f,   +0.589056f, -0.751086f, +0.298131f,
          +0.689385f, -0.635034f, +0.348539f,   +0.840055f, -0.335207f, +0.426548f,
          +0.791610f, -0.208969f, +0.574183f,   +0.743332f, -0.396542f, +0.538714f,
          +0.681238f, -0.540862f, +0.493339f,   +0.620193f, -0.643362f, +0.448827f,
          +0.569071f, -0.711874f, +0.411576f,   +0.532979f, -0.753331f, +0.385261f,
          +0.517388f, -0.769797f, +0.373794f,   +0.535053f, -0.751316f, +0.386318f,
          +0.626372f, -0.635215f, +0.451840f,   +0.763110f, -0.335282f, +0.552493f,
          +0.692666f, -0.209132f, +0.690273f,   +0.650461f, -0.396717f, +0.647700f,
          +0.596155f, -0.541030f, +0.593199f,   +0.542761f, -0.643512f, +0.539724f,
          +0.498052f, -0.711995f, +0.494982f,   +0.466495f, -0.753438f, +0.463372f,
          +0.452886f, -0.769891f, +0.449625f,   +0.468413f, -0.751379f, +0.464778f,
          +0.548491f, -0.635197f, +0.543767f,   +0.667917f, -0.335251f, +0.664450f,
          +0.576896f, -0.209182f, +0.789578f,   +0.541830f, -0.396723f, +0.740966f,
          +0.496674f, -0.540988f, +0.678710f,   +0.452259f, -0.643456f, +0.617597f,
          +0.415063f, -0.711936f, +0.566454f,   +0.388815f, -0.753367f, +0.530341f,
          +0.377517f, -0.769811f, +0.514657f,   +0.390513f, -0.751278f, +0.532054f,
          +0.457371f, -0.634980f, +0.622585f,   +0.556523f, -0.334986f, +0.760307f,
          +0.446525f, -0.209138f, +0.869987f,   +0.419514f, -0.396562f, +0.816546f,
          +0.384672f, -0.540769f, +0.748062f,   +0.350372f, -0.643212f, +0.680822f,
          +0.321635f, -0.711693f, +0.624535f,   +0.301361f, -0.753135f, +0.584782f,
          +0.292660f, -0.769575f, +0.567542f,   +0.302793f, -0.751001f, +0.586782f,
          +0.354714f, -0.634559f, +0.686667f,   +0.431045f, -0.334581f, +0.838007f,
          +0.304289f, -0.209046f, +0.929359f,   +0.286043f, -0.396276f, +0.872436f,
          +0.262432f, -0.540404f, +0.799433f,   +0.239152f, -0.642831f, +0.727719f,
          +0.219635f, -0.711323f, +0.667668f,   +0.205867f, -0.752775f, +0.625259f,
          +0.199994f, -0.769207f, +0.606896f,   +0.206994f, -0.750589f, +0.627511f,
          +0.242586f, -0.633959f, +0.734335f,   +0.294085f, -0.334058f, +0.895500f,
          +0.153818f, -0.208970f, +0.965749f,   +0.144799f, -0.395995f, +0.906764f,
          +0.133022f, -0.540029f, +0.831068f,   +0.121366f, -0.642434f, +0.756669f,
          +0.111582f, -0.710930f, +0.694354f,   +0.104688f, -0.752398f, +0.650336f,
          +0.101790f, -0.768817f, +0.631316f,   +0.105453f, -0.750133f, +0.652825f,
          +0.123749f, -0.633295f, +0.763953f,   +0.149115f, -0.333454f, +0.930899f,
          +0.001700f, -0.208314f, +0.978061f,   +0.001966f, -0.395353f, +0.918527f,
          +0.002105f, -0.539469f, +0.842003f,   +0.002163f, -0.641975f, +0.766723f,
          +0.002190f, -0.710558f, +0.703635f,   +0.002227f, -0.752108f, +0.659037f,
          +0.002326f, -0.768609f, +0.639715f,   +0.002594f, -0.750037f, +0.661391f,
          +0.003366f, -0.633650f, +0.773612f,   +0.002465f, -0.334024f, +0.942561f,
          -0.150460f, -0.207912f, +0.966506f,   -0.140940f, -0.395129f, +0.907750f,
          -0.128891f, -0.539381f, +0.832139f,   -0.117107f, -0.641989f, +0.757718f,
          -0.107258f, -0.710634f, +0.695338f,   -0.100277f, -0.752226f, +0.651230f,
          -0.097161f, -0.768789f, +0.632078f,   -0.100244f, -0.750353f, +0.653392f,
          -0.116880f, -0.634487f, +0.764045f,   -0.144110f, -0.334987f, +0.931137f,
          -0.301077f, -0.208329f, +0.930565f,   -0.282355f, -0.395689f, +0.873902f,
          -0.258491f, -0.539956f, +0.801018f,   -0.235090f, -0.642519f, +0.729316f,
          -0.215503f, -0.711115f, +0.669234f,   -0.201647f, -0.752653f, +0.626779f,
          -0.195554f, -0.769181f, +0.608375f,   -0.201979f, -0.750724f, +0.628982f,
          -0.235934f, -0.634755f, +0.735814f,   -0.289244f, -0.335095f, +0.896688f,
          -0.443540f, -0.208694f, +0.871619f,   -0.416080f, -0.396196f, +0.818478f,
          -0.380998f, -0.540494f, +0.750138f,   -0.346589f, -0.643020f, +0.682936f,
          -0.317794f, -0.711563f, +0.626646f,   -0.297436f, -0.753055f, +0.586890f,
          -0.288530f, -0.769553f, +0.569682f,   -0.298131f, -0.751086f, +0.589056f,
          -0.348538f, -0.635035f, +0.689385f,   -0.426547f, -0.335207f, +0.840056f,
          -0.574183f, -0.208969f, +0.791610f,   -0.538714f, -0.396542f, +0.743332f,
          -0.493339f, -0.540862f, +0.681238f,   -0.448827f, -0.643362f, +0.620193f,
          -0.411576f, -0.711874f, +0.569071f,   -0.385261f, -0.753331f, +0.532979f,
          -0.373794f, -0.769797f, +0.517388f,   -0.386318f, -0.751316f, +0.535053f,
          -0.451840f, -0.635215f, +0.626372f,   -0.552493f, -0.335282f, +0.763110f,
          -0.690273f, -0.209132f, +0.692666f,   -0.647700f, -0.396717f, +0.650461f,
          -0.593199f, -0.541030f, +0.596155f,   -0.539724f, -0.643512f, +0.542761f,
          -0.494982f, -0.711995f, +0.498052f,   -0.463372f, -0.753438f, +0.466495f,
          -0.449625f, -0.769891f, +0.452886f,   -0.464778f, -0.751379f, +0.468413f,
          -0.543767f, -0.635197f, +0.548491f,   -0.664450f, -0.335251f, +0.667917f,
          -0.789578f, -0.209182f, +0.576896f,   -0.740966f, -0.396723f, +0.541830f,
          -0.678710f, -0.540988f, +0.496674f,   -0.617597f, -0.643456f, +0.452259f,
          -0.566454f, -0.711936f, +0.415063f,   -0.530341f, -0.753367f, +0.388815f,
          -0.514657f, -0.769811f, +0.377517f,   -0.532054f, -0.751278f, +0.390513f,
          -0.622585f, -0.634980f, +0.457371f,   -0.760307f, -0.334986f, +0.556523f,
          -0.869987f, -0.209138f, +0.446525f,   -0.816546f, -0.396562f, +0.419514f,
          -0.748062f, -0.540769f, +0.384672f,   -0.680822f, -0.643212f, +0.350372f,
          -0.624535f, -0.711694f, +0.321635f,   -0.584782f, -0.753135f, +0.301361f,
          -0.567542f, -0.769575f, +0.292660f,   -0.586783f, -0.751001f, +0.302794f,
          -0.686667f, -0.634560f, +0.354715f,   -0.838007f, -0.334581f, +0.431045f,
          -0.929359f, -0.209046f, +0.304289f,   -0.872436f, -0.396276f, +0.286043f,
          -0.799433f, -0.540404f, +0.262432f,   -0.727719f, -0.642831f, +0.239152f,
          -0.667668f, -0.711323f, +0.219635f,   -0.625259f, -0.752775f, +0.205868f,
          -0.606895f, -0.769208f, +0.199994f,   -0.627513f, -0.750587f, +0.206994f,
          -0.734334f, -0.633960f, +0.242586f,   -0.895499f, -0.334059f, +0.294085f,
          -0.965749f, -0.208970f, +0.153818f,   -0.906764f, -0.395995f, +0.144799f,
          -0.831068f, -0.540029f, +0.133022f,   -0.756670f, -0.642434f, +0.121367f,
          -0.694354f, -0.710930f, +0.111582f,   -0.650337f, -0.752397f, +0.104688f,
          -0.631315f, -0.768817f, +0.101790f,   -0.652826f, -0.750132f, +0.105453f,
          -0.763953f, -0.633295f, +0.123750f,   -0.930898f, -0.333456f, +0.149114f,
          -0.978061f, -0.208314f, +0.001700f,   -0.918527f, -0.395353f, +0.001966f,
          -0.842003f, -0.539469f, +0.002105f,   -0.766724f, -0.641974f, +0.002163f,
          -0.703635f, -0.710559f, +0.002190f,   -0.659037f, -0.752107f, +0.002227f,
          -0.639715f, -0.768609f, +0.002326f,   -0.661391f, -0.750037f, +0.002594f,
          -0.773612f, -0.633650f, +0.003367f,   -0.942560f, -0.334026f, +0.002465f,
          -0.966507f, -0.207911f, -0.150459f,   -0.907749f, -0.395130f, -0.140939f,
          -0.832139f, -0.539381f, -0.128891f,   -0.757719f, -0.641988f, -0.117107f,
          -0.695338f, -0.710634f, -0.107258f,   -0.651230f, -0.752226f, -0.100277f,
          -0.632078f, -0.768789f, -0.097161f,   -0.653392f, -0.750353f, -0.100244f,
          -0.764045f, -0.634487f, -0.116879f,   -0.931137f, -0.334988f, -0.144110f,
          -0.930566f, -0.208325f, -0.301077f,   -0.873902f, -0.395689f, -0.282355f,
          -0.801017f, -0.539958f, -0.258489f,   -0.729317f, -0.642519f, -0.235090f,
          -0.669234f, -0.711115f, -0.215503f,   -0.626779f, -0.752653f, -0.201648f,
          -0.608375f, -0.769181f, -0.195554f,   -0.628981f, -0.750724f, -0.201979f,
          -0.735814f, -0.634755f, -0.235934f,   -0.896688f, -0.335095f, -0.289244f,
          -0.871619f, -0.208694f, -0.443540f,   -0.818478f, -0.396194f, -0.416082f,
          -0.750136f, -0.540496f, -0.380999f,   -0.682937f, -0.643019f, -0.346589f,
          -0.626646f, -0.711563f, -0.317794f,   -0.586890f, -0.753055f, -0.297436f,
          -0.569682f, -0.769553f, -0.288530f,   -0.589056f, -0.751086f, -0.298131f,
          -0.689385f, -0.635035f, -0.348538f,   -0.840056f, -0.335207f, -0.426548f,
          -0.791610f, -0.208969f, -0.574183f,   -0.743332f, -0.396542f, -0.538714f,
          -0.681238f, -0.540862f, -0.493339f,   -0.620194f, -0.643361f, -0.448827f,
          -0.569071f, -0.711874f, -0.411576f,   -0.532979f, -0.753331f, -0.385261f,
          -0.517388f, -0.769797f, -0.373794f,   -0.535053f, -0.751316f, -0.386318f,
          -0.626372f, -0.635215f, -0.451840f,   -0.763110f, -0.335282f, -0.552494f,
          -0.692666f, -0.209132f, -0.690273f,   -0.650461f, -0.396717f, -0.647700f,
          -0.596155f, -0.541030f, -0.593199f,   -0.542762f, -0.643511f, -0.539725f,
          -0.498052f, -0.711995f, -0.494982f,   -0.466495f, -0.753438f, -0.463372f,
          -0.452886f, -0.769891f, -0.449625f,   -0.468413f, -0.751379f, -0.464778f,
          -0.548491f, -0.635197f, -0.543767f,   -0.667917f, -0.335251f, -0.664450f,
          -0.576896f, -0.209182f, -0.789578f,   -0.541830f, -0.396723f, -0.740966f,
          -0.496674f, -0.540988f, -0.678710f,   -0.452259f, -0.643455f, -0.617597f,
          -0.415063f, -0.711936f, -0.566454f,   -0.388815f, -0.753367f, -0.530341f,
          -0.377517f, -0.769811f, -0.514657f,   -0.390513f, -0.751278f, -0.532054f,
          -0.457371f, -0.634980f, -0.622585f,   -0.556523f, -0.334986f, -0.760307f,
          -0.446525f, -0.209138f, -0.869987f,   -0.419514f, -0.396562f, -0.816546f,
          -0.384672f, -0.540769f, -0.748062f,   -0.350372f, -0.643211f, -0.680822f,
          -0.321635f, -0.711693f, -0.624535f,   -0.301361f, -0.753135f, -0.584782f,
          -0.292660f, -0.769575f, -0.567542f,   -0.302793f, -0.751001f, -0.586782f,
          -0.354715f, -0.634559f, -0.686667f,   -0.431044f, -0.334581f, -0.838007f,
          -0.304289f, -0.209046f, -0.929359f,   -0.286043f, -0.396276f, -0.872436f,
          -0.262432f, -0.540404f, -0.799433f,   -0.239152f, -0.642830f, -0.727719f,
          -0.219635f, -0.711323f, -0.667668f,   -0.205867f, -0.752775f, -0.625259f,
          -0.199994f, -0.769207f, -0.606896f,   -0.206994f, -0.750588f, -0.627511f,
          -0.242586f, -0.633959f, -0.734335f,   -0.294084f, -0.334058f, -0.895500f,
          -0.153818f, -0.208970f, -0.965749f,   -0.144799f, -0.395995f, -0.906764f,
          -0.133022f, -0.540029f, -0.831068f,   -0.121366f, -0.642433f, -0.756670f,
          -0.111582f, -0.710930f, -0.694354f,   -0.104688f, -0.752398f, -0.650336f,
          -0.101790f, -0.768817f, -0.631316f,   -0.105453f, -0.750133f, -0.652825f,
          -0.123750f, -0.633295f, -0.763953f,   -0.149114f, -0.333454f, -0.930899f,
          -0.001700f, -0.208314f, -0.978061f,   -0.001966f, -0.395353f, -0.918527f,
          -0.002105f, -0.539469f, -0.842003f,   -0.002163f, -0.641974f, -0.766724f,
          -0.002190f, -0.710558f, -0.703635f,   -0.002226f, -0.752108f, -0.659037f,
          -0.002326f, -0.768609f, -0.639715f,   -0.002594f, -0.750037f, -0.661391f,
          -0.003367f, -0.633650f, -0.773612f,   -0.002465f, -0.334024f, -0.942561f,
          +0.150460f, -0.207912f, -0.966506f,   +0.140940f, -0.395129f, -0.907750f,
          +0.128891f, -0.539381f, -0.832139f,   +0.117107f, -0.641988f, -0.757719f,
          +0.107258f, -0.710634f, -0.695338f,   +0.100277f, -0.752226f, -0.651230f,
          +0.097161f, -0.768789f, -0.632078f,   +0.100244f, -0.750353f, -0.653392f,
          +0.116879f, -0.634487f, -0.764045f,   +0.144110f, -0.334987f, -0.931137f,
          +0.301077f, -0.208329f, -0.930565f,   +0.282355f, -0.395689f, -0.873902f,
          +0.258491f, -0.539956f, -0.801018f,   +0.235090f, -0.642519f, -0.729317f,
          +0.215503f, -0.711115f, -0.669234f,   +0.201648f, -0.752653f, -0.626779f,
          +0.195554f, -0.769181f, -0.608375f,   +0.201979f, -0.750724f, -0.628982f,
          +0.235934f, -0.634755f, -0.735814f,   +0.289244f, -0.335095f, -0.896688f,
          +0.443540f, -0.208694f, -0.871619f,   +0.416080f, -0.396196f, -0.818478f,
          +0.380998f, -0.540494f, -0.750138f,   +0.346589f, -0.643019f, -0.682937f,
          +0.317794f, -0.711563f, -0.626646f,   +0.297436f, -0.753055f, -0.586890f,
          +0.288530f, -0.769553f, -0.569682f,   +0.298131f, -0.751086f, -0.589056f,
          +0.348538f, -0.635035f, -0.689385f,   +0.426548f, -0.335207f, -0.840056f,
          +0.574183f, -0.208969f, -0.791610f,   +0.538714f, -0.396542f, -0.743332f,
          +0.493339f, -0.540862f, -0.681238f,   +0.448827f, -0.643361f, -0.620194f,
          +0.411576f, -0.711874f, -0.569071f,   +0.385261f, -0.753331f, -0.532979f,
          +0.373794f, -0.769797f, -0.517388f,   +0.386318f, -0.751316f, -0.535053f,
          +0.451840f, -0.635215f, -0.626372f,   +0.552494f, -0.335282f, -0.763110f,
          +0.690273f, -0.209132f, -0.692666f,   +0.647700f, -0.396717f, -0.650461f,
          +0.593199f, -0.541030f, -0.596155f,   +0.539725f, -0.643511f, -0.542762f,
          +0.494982f, -0.711995f, -0.498052f,   +0.463372f, -0.753438f, -0.466495f,
          +0.449625f, -0.769891f, -0.452886f,   +0.464778f, -0.751379f, -0.468413f,
          +0.543767f, -0.635197f, -0.548491f,   +0.664450f, -0.335251f, -0.667917f,
          +0.789578f, -0.209182f, -0.576896f,   +0.740966f, -0.396723f, -0.541830f,
          +0.678710f, -0.540988f, -0.496674f,   +0.617597f, -0.643455f, -0.452259f,
          +0.566454f, -0.711936f, -0.415063f,   +0.530341f, -0.753367f, -0.388815f,
          +0.514657f, -0.769811f, -0.377517f,   +0.532054f, -0.751278f, -0.390513f,
          +0.622585f, -0.634980f, -0.457371f,   +0.760307f, -0.334986f, -0.556523f,
          +0.869987f, -0.209138f, -0.446525f,   +0.816546f, -0.396562f, -0.419514f,
          +0.748062f, -0.540769f, -0.384672f,   +0.680822f, -0.643211f, -0.350372f,
          +0.624535f, -0.711693f, -0.321635f,   +0.584782f, -0.753135f, -0.301361f,
          +0.567542f, -0.769575f, -0.292660f,   +0.586782f, -0.751001f, -0.302793f,
          +0.686667f, -0.634559f, -0.354715f,   +0.838007f, -0.334581f, -0.431044f,
          +0.929359f, -0.209046f, -0.304289f,   +0.872436f, -0.396276f, -0.286043f,
          +0.799433f, -0.540404f, -0.262432f,   +0.727719f, -0.642830f, -0.239152f,
          +0.667668f, -0.711323f, -0.219635f,   +0.625259f, -0.752775f, -0.205867f,
          +0.606896f, -0.769207f, -0.199994f,   +0.627511f, -0.750588f, -0.206994f,
          +0.734335f, -0.633959f, -0.242586f,   +0.895500f, -0.334058f, -0.294084f,
          +0.965749f, -0.208970f, -0.153818f,   +0.906764f, -0.395995f, -0.144799f,
          +0.831068f, -0.540029f, -0.133022f,   +0.756670f, -0.642433f, -0.121367f,
          +0.694354f, -0.710930f, -0.111582f,   +0.650336f, -0.752398f, -0.104688f,
          +0.631316f, -0.768817f, -0.101790f,   +0.652825f, -0.750133f, -0.105453f,
          +0.763953f, -0.633295f, -0.123750f,   +0.930898f, -0.333456f, -0.149115f,
          +0.907329f, -0.395543f, +0.142475f,   +0.918058f, -0.396447f, -0.000397f,
          +0.637612f, -0.763873f, +0.099749f,   +0.644805f, -0.764347f, -0.000644f,
          +0.379927f, -0.923124f, +0.059139f,   +0.384180f, -0.923258f, -0.000673f,
          +0.226375f, -0.973411f, +0.035000f,   +0.228899f, -0.973450f, -0.000633f,
          +0.138857f, -0.990084f, +0.021268f,   +0.140384f, -0.990097f, -0.000585f,
          +0.086103f, -0.996201f, +0.013005f,   +0.087019f, -0.996206f, -0.000542f,
          +0.052274f, -0.998603f, +0.007717f,   +0.052797f, -0.998605f, -0.000504f,
          +0.029518f, -0.999556f, +0.004167f,   +0.029775f, -0.999557f, -0.000472f,
          +0.014117f, -0.999899f, +0.001768f,   +0.014201f, -0.999899f, -0.000448f,
          -0.000000f, -1.000000f, +0.000000f,   +0.873155f, -0.396326f, +0.283773f,
          +0.613198f, -0.764478f, +0.198902f,   +0.365284f, -0.923364f, +0.118179f,
          +0.217667f, -0.973497f, +0.070176f,   +0.133544f, -0.990116f, +0.042847f,
          +0.082837f, -0.996214f, +0.026389f,   +0.050318f, -0.998608f, +0.015844f,
          +0.028441f, -0.999557f, +0.008758f,   +0.013637f, -0.999899f, +0.003966f,
          +0.817450f, -0.397013f, +0.417320f,   +0.573746f, -0.765023f, +0.292498f,
          +0.341710f, -0.923580f, +0.173878f,   +0.203638f, -0.973575f, +0.103360f,
          +0.124966f, -0.990146f, +0.063208f,   +0.077543f, -0.996225f, +0.039021f,
          +0.047129f, -0.998612f, +0.023521f,   +0.026667f, -0.999559f, +0.013100f,
          +0.012822f, -0.999899f, +0.006052f,   +0.742073f, -0.397519f, +0.539728f,
          +0.520642f, -0.765425f, +0.378227f,   +0.310066f, -0.923732f, +0.224897f,
          +0.184808f, -0.973630f, +0.133759f,   +0.113442f, -0.990166f, +0.081865f,
          +0.070421f, -0.996233f, +0.050600f,   +0.042829f, -0.998615f, +0.030560f,
          +0.024265f, -0.999560f, +0.017085f,   +0.011703f, -0.999900f, +0.007971f,
          +0.649021f, -0.397826f, +0.648465f,   +0.455315f, -0.765634f, +0.454415f,
          +0.271184f, -0.923812f, +0.270240f,   +0.161676f, -0.973657f, +0.160789f,
          +0.099279f, -0.990176f, +0.098460f,   +0.061663f, -0.996237f, +0.060904f,
          +0.037534f, -0.998616f, +0.036829f,   +0.021299f, -0.999560f, +0.020638f,
          +0.010314f, -0.999900f, +0.009687f,   +0.540253f, -0.397972f, +0.741447f,
          +0.379081f, -0.765684f, +0.519640f,   +0.225854f, -0.923818f, +0.309111f,
          +0.134705f, -0.973658f, +0.183970f,   +0.082762f, -0.990176f, +0.112701f,
          +0.051442f, -0.996237f, +0.069752f,   +0.031351f, -0.998617f, +0.042215f,
          +0.017830f, -0.999560f, +0.023695f,   +0.008681f, -0.999900f, +0.011170f,
          +0.417850f, -0.397904f, +0.816746f,   +0.293338f, -0.765587f, +0.572563f,
          +0.174877f, -0.923763f, +0.340706f,   +0.104372f, -0.973634f, +0.202838f,
          +0.064181f, -0.990167f, +0.124299f,   +0.039941f, -0.996234f, +0.076965f,
          +0.024387f, -0.998615f, +0.046613f,   +0.013918f, -0.999560f, +0.026197f,
          +0.006835f, -0.999900f, +0.012391f,   +0.284335f, -0.397814f, +0.872294f,
          +0.199800f, -0.765386f, +0.611772f,   +0.119247f, -0.923661f, +0.364184f,
          +0.071261f, -0.973593f, +0.216882f,   +0.043891f, -0.990151f, +0.132948f,
          +0.027377f, -0.996227f, +0.082351f,   +0.016777f, -0.998613f, +0.049905f,
          +0.009639f, -0.999559f, +0.028078f,   +0.004810f, -0.999900f, +0.013318f,
          +0.143117f, -0.397761f, +0.906258f,   +0.100785f, -0.765209f, +0.635844f,
          +0.060318f, -0.923562f, +0.378675f,   +0.036170f, -0.973552f, +0.225583f,
          +0.022380f, -0.990134f, +0.138323f,   +0.014051f, -0.996221f, +0.085711f,
          +0.008699f, -0.998611f, +0.051969f,   +0.005092f, -0.999559f, +0.029269f,
          +0.002651f, -0.999900f, +0.013920f,   +0.000397f, -0.396447f, +0.918058f,
          +0.000644f, -0.764347f, +0.644805f,   +0.000674f, -0.923258f, +0.384179f,
          +0.000633f, -0.973450f, +0.228900f,   +0.000585f, -0.990097f, +0.140384f,
          +0.000542f, -0.996206f, +0.087020f,   +0.000504f, -0.998605f, +0.052797f,
          +0.000472f, -0.999557f, +0.029775f,   +0.000448f, -0.999899f, +0.014201f,
          -0.142475f, -0.395543f, +0.907329f,   -0.099749f, -0.763873f, +0.637612f,
          -0.059139f, -0.923124f, +0.379927f,   -0.035000f, -0.973411f, +0.226375f,
          -0.021268f, -0.990084f, +0.138856f,   -0.013005f, -0.996201f, +0.086103f,
          -0.007717f, -0.998603f, +0.052274f,   -0.004167f, -0.999556f, +0.029518f,
          -0.001768f, -0.999899f, +0.014117f,   -0.283773f, -0.396326f, +0.873155f,
          -0.198902f, -0.764478f, +0.613198f,   -0.118179f, -0.923364f, +0.365284f,
          -0.070176f, -0.973497f, +0.217667f,   -0.042847f, -0.990116f, +0.133544f,
          -0.026389f, -0.996214f, +0.082837f,   -0.015844f, -0.998608f, +0.050318f,
          -0.008758f, -0.999557f, +0.028441f,   -0.003966f, -0.999899f, +0.013637f,
          -0.417320f, -0.397013f, +0.817450f,   -0.292498f, -0.765023f, +0.573746f,
          -0.173878f, -0.923580f, +0.341710f,   -0.103360f, -0.973575f, +0.203638f,
          -0.063208f, -0.990146f, +0.124966f,   -0.039021f, -0.996225f, +0.077543f,
          -0.023520f, -0.998612f, +0.047129f,   -0.013099f, -0.999559f, +0.026667f,
          -0.006051f, -0.999899f, +0.012822f,   -0.539728f, -0.397519f, +0.742073f,
          -0.378227f, -0.765425f, +0.520642f,   -0.224897f, -0.923732f, +0.310066f,
          -0.133759f, -0.973630f, +0.184808f,   -0.081865f, -0.990166f, +0.113442f,
          -0.050600f, -0.996233f, +0.070421f,   -0.030560f, -0.998615f, +0.042829f,
          -0.017084f, -0.999560f, +0.024265f,   -0.007971f, -0.999900f, +0.011704f,
          -0.648465f, -0.397826f, +0.649021f,   -0.454415f, -0.765634f, +0.455315f,
          -0.270240f, -0.923812f, +0.271184f,   -0.160789f, -0.973657f, +0.161676f,
          -0.098460f, -0.990176f, +0.099279f,   -0.060904f, -0.996237f, +0.061663f,
          -0.036829f, -0.998616f, +0.037534f,   -0.020638f, -0.999560f, +0.021299f,
          -0.009687f, -0.999900f, +0.010314f,   -0.741447f, -0.397972f, +0.540253f,
          -0.519640f, -0.765684f, +0.379081f,   -0.309111f, -0.923818f, +0.225854f,
          -0.183970f, -0.973658f, +0.134705f,   -0.112701f, -0.990176f, +0.082762f,
          -0.069752f, -0.996237f, +0.051442f,   -0.042215f, -0.998617f, +0.031351f,
          -0.023695f, -0.999560f, +0.017830f,   -0.011170f, -0.999900f, +0.008681f,
          -0.816746f, -0.397904f, +0.417850f,   -0.572563f, -0.765587f, +0.293338f,
          -0.340706f, -0.923763f, +0.174877f,   -0.202838f, -0.973634f, +0.104372f,
          -0.124300f, -0.990167f, +0.064181f,   -0.076965f, -0.996234f, +0.039941f,
          -0.046613f, -0.998615f, +0.024388f,   -0.026197f, -0.999560f, +0.013919f,
          -0.012391f, -0.999900f, +0.006836f,   -0.872294f, -0.397814f, +0.284335f,
          -0.611772f, -0.765386f, +0.199800f,   -0.364184f, -0.923661f, +0.119247f,
          -0.216882f, -0.973593f, +0.071261f,   -0.132949f, -0.990151f, +0.043892f,
          -0.082351f, -0.996227f, +0.027377f,   -0.049905f, -0.998613f, +0.016777f,
          -0.028078f, -0.999559f, +0.009639f,   -0.013318f, -0.999900f, +0.004810f,
          -0.906258f, -0.397761f, +0.143117f,   -0.635844f, -0.765209f, +0.100785f,
          -0.378675f, -0.923562f, +0.060318f,   -0.225583f, -0.973552f, +0.036170f,
          -0.138323f, -0.990134f, +0.022380f,   -0.085711f, -0.996221f, +0.014051f,
          -0.051969f, -0.998611f, +0.008699f,   -0.029269f, -0.999559f, +0.005092f,
          -0.013920f, -0.999900f, +0.002651f,   -0.918058f, -0.396447f, +0.000397f,
          -0.644806f, -0.764346f, +0.000644f,   -0.384178f, -0.923259f, +0.000674f,
          -0.228901f, -0.973449f, +0.000634f,   -0.140383f, -0.990097f, +0.000586f,
          -0.087020f, -0.996206f, +0.000542f,   -0.052797f, -0.998605f, +0.000504f,
          -0.029775f, -0.999557f, +0.000472f,   -0.014201f, -0.999899f, +0.000448f,
          -0.907329f, -0.395544f, -0.142475f,   -0.637615f, -0.763870f, -0.099749f,
          -0.379926f, -0.923125f, -0.059138f,   -0.226375f, -0.973411f, -0.035000f,
          -0.138856f, -0.990084f, -0.021267f,   -0.086102f, -0.996201f, -0.013005f,
          -0.052274f, -0.998603f, -0.007717f,   -0.029518f, -0.999556f, -0.004167f,
          -0.014117f, -0.999899f, -0.001768f,   -0.873154f, -0.396327f, -0.283772f,
          -0.613199f, -0.764477f, -0.198903f,   -0.365285f, -0.923364f, -0.118179f,
          -0.217667f, -0.973497f, -0.070176f,   -0.133544f, -0.990116f, -0.042847f,
          -0.082836f, -0.996214f, -0.026389f,   -0.050318f, -0.998608f, -0.015844f,
          -0.028441f, -0.999557f, -0.008758f,   -0.013637f, -0.999899f, -0.003966f,
          -0.817451f, -0.397013f, -0.417320f,   -0.573746f, -0.765023f, -0.292498f,
          -0.341710f, -0.923580f, -0.173878f,   -0.203638f, -0.973575f, -0.103360f,
          -0.124966f, -0.990146f, -0.063208f,   -0.077542f, -0.996225f, -0.039020f,
          -0.047129f, -0.998612f, -0.023520f,   -0.026667f, -0.999559f, -0.013099f,
          -0.012822f, -0.999899f, -0.006051f,   -0.742074f, -0.397519f, -0.539727f,
          -0.520642f, -0.765425f, -0.378228f,   -0.310066f, -0.923732f, -0.224896f,
          -0.184808f, -0.973630f, -0.133759f,   -0.113443f, -0.990166f, -0.081865f,
          -0.070420f, -0.996233f, -0.050600f,   -0.042829f, -0.998615f, -0.030560f,
          -0.024265f, -0.999560f, -0.017085f,   -0.011703f, -0.999900f, -0.007971f,
          -0.649021f, -0.397826f, -0.648465f,   -0.455315f, -0.765634f, -0.454415f,
          -0.271184f, -0.923812f, -0.270240f,   -0.161675f, -0.973657f, -0.160789f,
          -0.099280f, -0.990176f, -0.098460f,   -0.061662f, -0.996237f, -0.060903f,
          -0.037534f, -0.998616f, -0.036828f,   -0.021299f, -0.999560f, -0.020638f,
          -0.010314f, -0.999900f, -0.009687f,   -0.540254f, -0.397972f, -0.741447f,
          -0.379081f, -0.765684f, -0.519640f,   -0.225854f, -0.923818f, -0.309111f,
          -0.134705f, -0.973658f, -0.183970f,   -0.082762f, -0.990176f, -0.112701f,
          -0.051442f, -0.996237f, -0.069751f,   -0.031351f, -0.998617f, -0.042215f,
          -0.017830f, -0.999560f, -0.023695f,   -0.008681f, -0.999900f, -0.011170f,
          -0.417850f, -0.397904f, -0.816746f,   -0.293338f, -0.765587f, -0.572564f,
          -0.174877f, -0.923762f, -0.340706f,   -0.104372f, -0.973634f, -0.202838f,
          -0.064181f, -0.990167f, -0.124300f,   -0.039941f, -0.996234f, -0.076964f,
          -0.024387f, -0.998615f, -0.046613f,   -0.013919f, -0.999560f, -0.026197f,
          -0.006836f, -0.999900f, -0.012391f,   -0.284336f, -0.397814f, -0.872294f,
          -0.199800f, -0.765386f, -0.611772f,   -0.119247f, -0.923661f, -0.364184f,
          -0.071261f, -0.973593f, -0.216882f,   -0.043892f, -0.990151f, -0.132949f,
          -0.027377f, -0.996227f, -0.082350f,   -0.016777f, -0.998613f, -0.049905f,
          -0.009639f, -0.999559f, -0.028078f,   -0.004810f, -0.999900f, -0.013318f,
          -0.143117f, -0.397761f, -0.906258f,   -0.100784f, -0.765209f, -0.635845f,
          -0.060318f, -0.923562f, -0.378675f,   -0.036170f, -0.973552f, -0.225583f,
          -0.022380f, -0.990134f, -0.138323f,   -0.014051f, -0.996221f, -0.085710f,
          -0.008699f, -0.998611f, -0.051969f,   -0.005092f, -0.999559f, -0.029269f,
          -0.002651f, -0.999900f, -0.013920f,   -0.000398f, -0.396447f, -0.918058f,
          -0.000643f, -0.764347f, -0.644805f,   -0.000674f, -0.923258f, -0.384179f,
          -0.000633f, -0.973450f, -0.228900f,   -0.000585f, -0.990097f, -0.140384f,
          -0.000542f, -0.996207f, -0.087019f,   -0.000504f, -0.998605f, -0.052797f,
          -0.000472f, -0.999557f, -0.029775f,   -0.000448f, -0.999899f, -0.014201f,
          +0.142475f, -0.395543f, -0.907330f,   +0.099749f, -0.763873f, -0.637612f,
          +0.059139f, -0.923124f, -0.379927f,   +0.035000f, -0.973411f, -0.226374f,
          +0.021268f, -0.990084f, -0.138857f,   +0.013005f, -0.996201f, -0.086102f,
          +0.007717f, -0.998603f, -0.052274f,   +0.004167f, -0.999556f, -0.029518f,
          +0.001768f, -0.999899f, -0.014117f,   +0.283773f, -0.396326f, -0.873155f,
          +0.198903f, -0.764478f, -0.613198f,   +0.118179f, -0.923364f, -0.365285f,
          +0.070176f, -0.973497f, -0.217667f,   +0.042847f, -0.990116f, -0.133544f,
          +0.026389f, -0.996214f, -0.082836f,   +0.015844f, -0.998608f, -0.050318f,
          +0.008758f, -0.999557f, -0.028441f,   +0.003966f, -0.999899f, -0.013637f,
          +0.417320f, -0.397013f, -0.817450f,   +0.292498f, -0.765023f, -0.573746f,
          +0.173878f, -0.923580f, -0.341710f,   +0.103360f, -0.973575f, -0.203638f,
          +0.063208f, -0.990146f, -0.124966f,   +0.039020f, -0.996225f, -0.077542f,
          +0.023520f, -0.998612f, -0.047129f,   +0.013099f, -0.999559f, -0.026667f,
          +0.006051f, -0.999899f, -0.012822f,   +0.539727f, -0.397519f, -0.742074f,
          +0.378228f, -0.765425f, -0.520642f,   +0.224896f, -0.923732f, -0.310066f,
          +0.133759f, -0.973630f, -0.184808f,   +0.081865f, -0.990166f, -0.113443f,
          +0.050600f, -0.996233f, -0.070420f,   +0.030560f, -0.998615f, -0.042829f,
          +0.017085f, -0.999560f, -0.024265f,   +0.007971f, -0.999900f, -0.011703f,
          +0.648465f, -0.397826f, -0.649021f,   +0.454415f, -0.765634f, -0.455315f,
          +0.270240f, -0.923812f, -0.271184f,   +0.160789f, -0.973657f, -0.161675f,
          +0.098460f, -0.990176f, -0.099280f,   +0.060903f, -0.996237f, -0.061662f,
          +0.036828f, -0.998616f, -0.037534f,   +0.020638f, -0.999560f, -0.021299f,
          +0.009687f, -0.999900f, -0.010314f,   +0.741447f, -0.397972f, -0.540254f,
          +0.519640f, -0.765684f, -0.379081f,   +0.309111f, -0.923818f, -0.225854f,
          +0.183970f, -0.973658f, -0.134705f,   +0.112701f, -0.990176f, -0.082762f,
          +0.069751f, -0.996237f, -0.051442f,   +0.042215f, -0.998617f, -0.031351f,
          +0.023695f, -0.999560f, -0.017830f,   +0.011170f, -0.999900f, -0.008681f,
          +0.816746f, -0.397904f, -0.417850f,   +0.572564f, -0.765587f, -0.293338f,
          +0.340706f, -0.923762f, -0.174877f,   +0.202838f, -0.973634f, -0.104372f,
          +0.124300f, -0.990167f, -0.064181f,   +0.076964f, -0.996234f, -0.039941f,
          +0.046613f, -0.998615f, -0.024387f,   +0.026197f, -0.999560f, -0.013918f,
          +0.012391f, -0.999900f, -0.006836f,   +0.872295f, -0.397813f, -0.284335f,
          +0.611772f, -0.765386f, -0.199800f,   +0.364184f, -0.923661f, -0.119247f,
          +0.216882f, -0.973593f, -0.071261f,   +0.132949f, -0.990151f, -0.043892f,
          +0.082350f, -0.996227f, -0.027377f,   +0.049905f, -0.998613f, -0.016777f,
          +0.028078f, -0.999559f, -0.009639f,   +0.013318f, -0.999900f, -0.004810f,
          +0.906259f, -0.397760f, -0.143117f,   +0.635846f, -0.765208f, -0.100784f,
          +0.378676f, -0.923562f, -0.060317f,   +0.225582f, -0.973552f, -0.036169f,
          +0.138324f, -0.990134f, -0.022380f,   +0.085710f, -0.996221f, -0.014051f,
          +0.051969f, -0.998611f, -0.008699f,   +0.029269f, -0.999559f, -0.005092f,
          +0.013920f, -0.999900f, -0.002651f,   +0.001123f, -0.999659f, +0.026083f,
          +0.001132f, -0.981286f, +0.192555f,   +0.004633f, -0.987241f, +0.159169f,
          +0.004724f, -0.999988f, -0.001343f,   +0.016827f, -0.987233f, +0.158392f,
          +0.017033f, -0.999854f, -0.001587f,   +0.040993f, -0.986638f, +0.157687f,
          +0.041327f, -0.999144f, -0.001907f,   +0.082537f, -0.984040f, +0.157647f,
          +0.083040f, -0.996544f, -0.002276f,   +0.150483f, -0.975682f, +0.159371f,
          +0.151313f, -0.988483f, -0.002637f,   +0.259801f, -0.951479f, +0.164897f,
          +0.261499f, -0.965199f, -0.002920f,   +0.431553f, -0.884541f, +0.177056f,
          +0.435905f, -0.899988f, -0.002873f,   +0.671832f, -0.714162f, +0.196505f,
          +0.683374f, -0.730066f, -0.001936f,   +0.897892f, -0.384795f, +0.213829f,
          +0.919675f, -0.392681f, +0.000464f,   +0.975015f, -0.045047f, +0.217525f,
          +0.999090f, -0.042549f, +0.002833f,   +0.001098f, -0.914501f, +0.404581f,
          +0.004531f, -0.931561f, +0.363556f,   +0.016504f, -0.931904f, +0.362329f,
          +0.040257f, -0.931543f, +0.361395f,   +0.081116f, -0.928572f, +0.362180f,
          +0.147711f, -0.918393f, +0.367064f,   +0.253553f, -0.889622f, +0.379847f,
          +0.415147f, -0.814606f, +0.405056f,   +0.629200f, -0.640119f, +0.440858f,
          +0.817589f, -0.335881f, +0.467688f,   +0.881311f, -0.038889f, +0.470934f,
          +0.000951f, -0.752310f, +0.658809f,   +0.003969f, -0.791140f, +0.611622f,
          +0.014604f, -0.792040f, +0.610294f,   +0.035808f, -0.791866f, +0.609644f,
          +0.072231f, -0.787925f, +0.611520f,   +0.131003f, -0.774414f, +0.618967f,
          +0.221904f, -0.738908f, +0.636218f,   +0.353079f, -0.656821f, +0.666274f,
          +0.512105f, -0.494029f, +0.702626f,   +0.641255f, -0.250936f, +0.725137f,
          +0.686651f, -0.029565f, +0.726385f,   +0.000557f, -0.417763f, +0.908556f,
          +0.002413f, -0.491630f, +0.870801f,   +0.009150f, -0.492952f, +0.870008f,
          +0.022761f, -0.492944f, +0.869763f,   +0.046109f, -0.488757f, +0.871201f,
          +0.083085f, -0.475091f, +0.876005f,   +0.137793f, -0.442725f, +0.886007f,
          +0.210885f, -0.378383f, +0.901307f,   +0.291447f, -0.272132f, +0.917062f,
          +0.353389f, -0.136079f, +0.925526f,   +0.378011f, -0.018495f, +0.925616f,
          -0.000135f, +0.092788f, +0.995686f,   -0.000449f, -0.000214f, +1.000000f,
          -0.001078f, -0.001027f, +0.999999f,   -0.001973f, -0.002010f, +0.999996f,
          -0.003226f, -0.003212f, +0.999990f,   -0.004929f, -0.004665f, +0.999977f,
          -0.007117f, -0.006323f, +0.999955f,   -0.009624f, -0.007987f, +0.999922f,
          -0.011932f, -0.009226f, +0.999886f,   -0.013295f, -0.009564f, +0.999866f,
          -0.010439f, -0.007580f, +0.999917f,   -0.000821f, +0.556769f, +0.830667f,
          -0.003495f, +0.491023f, +0.871139f,   -0.012120f, +0.490889f, +0.871138f,
          -0.028898f, +0.488503f, +0.872083f,   -0.057123f, +0.480274f, +0.875256f,
          -0.100879f, +0.460015f, +0.882162f,   -0.163506f, +0.418004f, +0.893610f,
          -0.242752f, +0.342887f, +0.907469f,   -0.323545f, +0.231639f, +0.917421f,
          -0.380455f, +0.103337f, +0.919008f,   -0.396385f, +0.001330f, +0.918083f,
          -0.001250f, +0.823097f, +0.567900f,   -0.005450f, +0.790401f, +0.612566f,
          -0.019333f, +0.790283f, +0.612437f,   -0.046696f, +0.787786f, +0.614177f,
          -0.093403f, +0.778668f, +0.620445f,   -0.167539f, +0.754596f, +0.634442f,
          -0.277527f, +0.699500f, +0.658543f,   -0.422357f, +0.588417f, +0.689478f,
          -0.571792f, +0.405657f, +0.713089f,   -0.671763f, +0.184662f, +0.717380f,
          -0.699209f, +0.007363f, +0.714880f,   -0.001472f, +0.944898f, +0.327361f,
          -0.006412f, +0.930936f, +0.365127f,   -0.022978f, +0.930694f, +0.365077f,
          -0.055854f, +0.928662f, +0.366697f,   -0.112482f, +0.921281f, +0.372277f,
          -0.203873f, +0.900191f, +0.384827f,   -0.343117f, +0.846561f, +0.406947f,
          -0.532294f, +0.725493f, +0.436260f,   -0.729618f, +0.506891f, +0.459041f,
          -0.855759f, +0.231863f, +0.462510f,   -0.887943f, +0.010878f, +0.459826f,
          -0.001558f, +0.991432f, +0.130615f,   -0.006796f, +0.986870f, +0.161375f,
          -0.024492f, +0.986581f, +0.161425f,   -0.059779f, +0.984918f, +0.162368f,
          -0.120855f, +0.978802f, +0.165352f,   -0.220236f, +0.960186f, +0.171867f,
          -0.373597f, +0.909298f, +0.183309f,   -0.585092f, +0.786279f, +0.198574f,
          -0.806495f, +0.552642f, +0.210124f,   -0.944139f, +0.253197f, +0.210931f,
          -0.977704f, +0.012860f, +0.209594f,   -0.001571f, +0.999667f, -0.025755f,
          -0.006652f, +0.999977f, +0.001482f,   -0.024323f, +0.999703f, +0.001676f,
          -0.059765f, +0.998211f, +0.001852f,   -0.121316f, +0.992612f, +0.002002f,
          -0.221803f, +0.975089f, +0.002080f,   -0.377582f, +0.925974f, +0.001938f,
          -0.593875f, +0.804556f, +0.001301f,   -0.822060f, +0.569401f, -0.000014f,
          -0.964461f, +0.264221f, -0.001786f,   -0.999845f, +0.017527f, -0.001395f,
          -0.001536f, +0.981519f, -0.191359f,   -0.006269f, +0.987430f, -0.157930f,
          -0.023265f, +0.987239f, -0.157539f,   -0.057546f, +0.985761f, -0.157998f,
          -0.117155f, +0.980058f, -0.160505f,   -0.214412f, +0.962409f, -0.166720f,
          -0.365019f, +0.913751f, -0.178382f,   -0.574374f, +0.795012f, -0.195067f,
          -0.797253f, +0.565933f, -0.210018f,   -0.939481f, +0.266569f, -0.215214f,
          -0.976825f, +0.022066f, -0.212900f,   -0.001405f, +0.915454f, -0.402421f,
          -0.005729f, +0.932575f, -0.360932f,   -0.021369f, +0.932588f, -0.360311f,
          -0.052978f, +0.930959f, -0.361261f,   -0.107782f, +0.924285f, -0.366171f,
          -0.196613f, +0.904634f, -0.378128f,   -0.332642f, +0.853915f, -0.400223f,
          -0.519315f, +0.737780f, -0.431268f,   -0.717862f, +0.523902f, -0.458477f,
          -0.848405f, +0.248578f, -0.467351f,   -0.885709f, +0.022628f, -0.463689f,
          -0.001123f, +0.754831f, -0.655918f,   -0.004624f, +0.794127f, -0.607735f,
          -0.017415f, +0.794564f, -0.606931f,   -0.043314f, +0.792859f, -0.607864f,
          -0.088006f, +0.785014f, -0.613194f,   -0.159428f, +0.763008f, -0.626420f,
          -0.266178f, +0.711286f, -0.650555f,   -0.408457f, +0.604952f, -0.683517f,
          -0.558260f, +0.425958f, -0.711973f,   -0.661230f, +0.204085f, -0.721889f,
          -0.694868f, +0.021295f, -0.718822f,   -0.000610f, +0.422849f, -0.906200f,
          -0.002600f, +0.498214f, -0.867050f,   -0.010088f, +0.499095f, -0.866489f,
          -0.025379f, +0.498098f, -0.866749f,   -0.051669f, +0.491735f, -0.869210f,
          -0.092980f, +0.473981f, -0.875612f,   -0.152812f, +0.435152f, -0.887294f,
          -0.229625f, +0.363360f, -0.902907f,   -0.309473f, +0.253894f, -0.916387f,
          -0.367213f, +0.124207f, -0.921807f,   -0.389605f, +0.016511f, -0.920834f,
          +0.000114f, -0.086697f, -0.996235f,   +0.000343f, +0.009278f, -0.999957f,
          +0.000697f, +0.009741f, -0.999952f,   +0.001045f, +0.010345f, -0.999946f,
          +0.001371f, +0.011078f, -0.999938f,   +0.001643f, +0.011883f, -0.999928f,
          +0.001796f, +0.012610f, -0.999919f,   +0.001735f, +0.012952f, -0.999915f,
          +0.001380f, +0.012517f, -0.999921f,   +0.000767f, +0.011094f, -0.999938f,
          -0.001916f, +0.007439f, -0.999970f,   +0.000712f, -0.552957f, -0.833210f,
          +0.002958f, -0.484597f, -0.874733f,   +0.010372f, -0.485012f, -0.874446f,
          +0.024856f, -0.483955f, -0.874740f,   +0.049370f, -0.478486f, -0.876706f,
          +0.087936f, -0.463217f, -0.881871f,   +0.144760f, -0.428807f, -0.891723f,
          +0.220348f, -0.362019f, -0.905753f,   +0.303071f, -0.253605f, -0.918603f,
          +0.365846f, -0.117088f, -0.923281f,   +0.386416f, -0.004940f, -0.922312f,
          +0.001002f, -0.821522f, -0.570176f,   +0.004291f, -0.787589f, -0.616186f,
          +0.015332f, -0.788028f, -0.615449f,   +0.037089f, -0.787305f, -0.615448f,
          +0.074314f, -0.782658f, -0.618000f,   +0.134277f, -0.768144f, -0.626039f,
          +0.226969f, -0.731078f, -0.643436f,   +0.360579f, -0.646337f, -0.672481f,
          +0.521663f, -0.479815f, -0.705440f,   +0.650848f, -0.234299f, -0.722150f,
          +0.693154f, -0.017991f, -0.720565f,   +0.001103f, -0.944445f, -0.328667f,
          +0.004735f, -0.930038f, -0.367432f,   +0.016977f, -0.930152f, -0.366782f,
          +0.041125f, -0.929496f, -0.366532f,   +0.082557f, -0.926138f, -0.368038f,
          +0.150081f, -0.915377f, -0.373579f,   +0.257429f, -0.885492f, -0.386827f,
          +0.421256f, -0.808163f, -0.411603f,   +0.637305f, -0.629429f, -0.444592f,
          +0.824766f, -0.321308f, -0.465321f,   +0.885287f, -0.028939f, -0.464144f,
          +0.001122f, -0.991382f, -0.130998f,   +0.004828f, -0.986729f, -0.162303f,
          +0.017283f, -0.986624f, -0.162095f,   +0.041824f, -0.985899f, -0.162029f,
          +0.083935f, -0.983096f, -0.162719f,   +0.152793f, -0.974346f, -0.165240f,
          +0.263617f, -0.949285f, -0.171357f,   +0.437613f, -0.880266f, -0.183379f,
          +0.679831f, -0.705381f, -0.200668f,   +0.903899f, -0.371275f, -0.212416f,
          +0.976811f, -0.035876f, -0.211075f,   +0.964773f, +0.149485f, +0.216490f,
          +0.987954f, +0.154728f, +0.002612f,   +0.939087f, +0.268508f, +0.214520f,
          +0.961457f, +0.274952f, +0.001583f,   +0.904856f, +0.370597f, +0.209508f,
          +0.925917f, +0.377727f, +0.000472f,   +0.863061f, +0.463421f, +0.200914f,
          +0.882168f, +0.470935f, -0.000530f,   +0.812900f, +0.550753f, +0.189381f,
          +0.829543f, +0.558442f, -0.001278f,   +0.753015f, +0.633893f, +0.176489f,
          +0.766940f, +0.641716f, -0.001657f,   +0.682195f, +0.712376f, +0.164715f,
          +0.693377f, +0.720574f, -0.001568f,   +0.599922f, +0.784437f, +0.157329f,
          +0.608448f, +0.793594f, -0.000884f,   +0.506724f, +0.847479f, +0.158146f,
          +0.512698f, +0.858569f, +0.000462f,   +0.458908f, +0.879078f, +0.128936f,
          +0.462942f, +0.885997f, -0.026350f,   +0.872629f, +0.135338f, +0.469257f,
          +0.849372f, +0.246248f, +0.466829f,   +0.819583f, +0.342870f, +0.459046f,
          +0.784638f, +0.432101f, +0.444558f,   +0.743352f, +0.517354f, +0.423996f,
          +0.693424f, +0.599353f, +0.399923f,   +0.632623f, +0.676581f, +0.376863f,
          +0.559664f, +0.745803f, +0.361324f,   +0.474801f, +0.802627f, +0.361045f,
          +0.435518f, +0.839317f, +0.325379f,   +0.681108f, +0.104800f, +0.724644f,
          +0.663018f, +0.194952f, +0.722773f,   +0.641615f, +0.275454f, +0.715860f,
          +0.618802f, +0.352032f, +0.702252f,   +0.593278f, +0.427810f, +0.681909f,
          +0.561888f, +0.503097f, +0.656640f,   +0.520761f, +0.575096f, +0.630930f,
          +0.466784f, +0.638034f, +0.612393f,   +0.398966f, +0.684005f, +0.610707f,
          +0.376439f, +0.732287f, +0.567494f,   +0.376991f, +0.055764f, +0.924537f,
          +0.367419f, +0.107940f, +0.923771f,   +0.356813f, +0.155757f, +0.921099f,
          +0.347028f, +0.202661f, +0.915697f,   +0.337512f, +0.250998f, +0.907241f,
          +0.326030f, +0.301268f, +0.896070f,   +0.309042f, +0.351164f, +0.883842f,
          +0.282579f, +0.394643f, +0.874303f,   +0.244121f, +0.422466f, +0.872885f,
          +0.251053f, +0.493614f, +0.832657f,   -0.006290f, -0.006158f, +0.999961f,
          -0.004409f, -0.006649f, +0.999968f,   -0.002978f, -0.007112f, +0.999970f,
          -0.001911f, -0.007612f, +0.999969f,   -0.001127f, -0.008181f, +0.999966f,
          -0.000567f, -0.008797f, +0.999961f,   -0.000209f, -0.009373f, +0.999956f,
          -0.000044f, -0.009743f, +0.999953f,   -0.000065f, -0.009689f, +0.999953f,
          +0.038916f, +0.076884f, +0.996280f,   -0.387684f, -0.067569f, +0.919313f,
          -0.373150f, -0.124176f, +0.919423f,   -0.356836f, -0.176791f, +0.917286f,
          -0.341132f, -0.227383f, +0.912100f,   -0.326071f, -0.277498f, +0.903699f,
          -0.310204f, -0.327182f, +0.892595f,   -0.291162f, -0.374149f, +0.880476f,
          -0.266349f, -0.413366f, +0.870739f,   -0.234098f, -0.437726f, +0.868098f,
          -0.187367f, -0.377398f, +0.906898f,   -0.687550f, -0.115429f, +0.716904f,
          -0.661987f, -0.218679f, +0.716905f,   -0.628651f, -0.313678f, +0.711621f,
          -0.591802f, -0.400707f, +0.699431f,   -0.553126f, -0.480367f, +0.680661f,
          -0.512474f, -0.552280f, +0.657539f,   -0.469069f, -0.614687f, +0.634141f,
          -0.422241f, -0.664691f, +0.616359f,   -0.371881f, -0.698798f, +0.611053f,
          -0.332340f, -0.674826f, +0.658908f,   -0.875015f, -0.144783f, +0.461937f,
          -0.841992f, -0.278726f, +0.461911f,   -0.794273f, -0.400879f, +0.456537f,
          -0.737731f, -0.508255f, +0.444331f,   -0.676873f, -0.600108f, +0.426279f,
          -0.614681f, -0.676805f, +0.405096f,   -0.552930f, -0.739082f, +0.384742f,
          -0.492714f, -0.787733f, +0.369743f,   -0.434663f, -0.823393f, +0.364818f,
          -0.401543f, -0.820855f, +0.406154f,   -0.964544f, -0.158095f, +0.211330f,
          -0.927803f, -0.307469f, +0.211291f,   -0.871848f, -0.443199f, +0.208463f,
          -0.803618f, -0.559827f, +0.201967f,   -0.729937f, -0.655853f, +0.192480f,
          -0.656037f, -0.732564f, +0.181563f,   -0.585247f, -0.792582f, +0.171171f,
          -0.519253f, -0.838841f, +0.163470f,   -0.458721f, -0.873962f, +0.160514f,
          -0.429469f, -0.882027f, +0.193869f,   -0.987462f, -0.157858f, -0.000193f,
          -0.950257f, -0.311466f, -0.000220f,   -0.892424f, -0.451198f, +0.000047f,
          -0.821138f, -0.570730f, +0.000409f,   -0.743984f, -0.668197f, +0.000696f,
          -0.666910f, -0.745138f, +0.000801f,   -0.593685f, -0.804697f, +0.000640f,
          -0.526187f, -0.850369f, +0.000154f,   -0.465043f, -0.885288f, -0.000682f,
          -0.436861f, -0.899141f, +0.026417f,   -0.965669f, -0.150384f, -0.211820f,
          -0.930040f, -0.300138f, -0.211996f,   -0.875008f, -0.436744f, -0.208842f,
          -0.807302f, -0.554574f, -0.201771f,   -0.733736f, -0.651809f, -0.191771f,
          -0.659631f, -0.729559f, -0.180642f,   -0.588411f, -0.790372f, -0.170543f,
          -0.521905f, -0.837143f, -0.163727f,   -0.460811f, -0.872528f, -0.162321f,
          -0.433349f, -0.891526f, -0.131874f,   -0.876180f, -0.134825f, -0.462743f,
          -0.844460f, -0.269191f, -0.463058f,   -0.797883f, -0.392406f, -0.457603f,
          -0.742079f, -0.501221f, -0.445080f,   -0.681486f, -0.594606f, -0.426639f,
          -0.619101f, -0.672640f, -0.405301f,   -0.556844f, -0.735927f, -0.385144f,
          -0.495911f, -0.785178f, -0.370900f,   -0.437029f, -0.821001f, -0.367373f,
          -0.413630f, -0.848382f, -0.330391f,   -0.688197f, -0.103923f, -0.718043f,
          -0.663883f, -0.207605f, -0.718443f,   -0.631675f, -0.303580f, -0.713321f,
          -0.595689f, -0.391883f, -0.701129f,   -0.557488f, -0.472935f, -0.682305f,
          -0.516943f, -0.546176f, -0.659137f,   -0.473256f, -0.609626f, -0.635912f,
          -0.425805f, -0.660264f, -0.618661f,   -0.374572f, -0.694461f, -0.614345f,
          -0.361229f, -0.736939f, -0.571344f,   -0.386935f, -0.055716f, -0.920422f,
          -0.373477f, -0.112742f, -0.920763f,   -0.358130f, -0.165966f, -0.918803f,
          -0.343222f, -0.217247f, -0.913785f,   -0.328800f, -0.268046f, -0.905562f,
          -0.313394f, -0.318395f, -0.894656f,   -0.294570f, -0.365959f, -0.882781f,
          -0.269675f, -0.405600f, -0.873364f,   -0.237040f, -0.430181f, -0.871066f,
          -0.245416f, -0.497095f, -0.832267f,   -0.003648f, +0.004494f, -0.999983f,
          -0.002807f, +0.003608f, -0.999990f,   -0.002283f, +0.002869f, -0.999993f,
          -0.002033f, +0.002256f, -0.999995f,   -0.002020f, +0.001736f, -0.999996f,
          -0.002223f, +0.001267f, -0.999997f,   -0.002609f, +0.000829f, -0.999996f,
          -0.003121f, +0.000416f, -0.999995f,   -0.003643f, +0.000045f, -0.999993f,
          -0.041213f, -0.083172f, -0.995683f,   +0.380928f, +0.064012f, -0.922386f,
          +0.370121f, +0.115733f, -0.921747f,   +0.358338f, +0.163338f, -0.919193f,
          +0.347396f, +0.210213f, -0.913853f,   +0.336721f, +0.258629f, -0.905389f,
          +0.324069f, +0.309055f, -0.894128f,   +0.305939f, +0.359107f, -0.881728f,
          +0.278461f, +0.402666f, -0.871963f,   +0.239272f, +0.430433f, -0.870331f,
          +0.188116f, +0.372521f, -0.908758f,   +0.684874f, +0.110435f, -0.720245f,
          +0.665471f, +0.200048f, -0.719117f,   +0.642670f, +0.280279f, -0.713035f,
          +0.618470f, +0.356698f, -0.700186f,   +0.591619f, +0.432403f, -0.680452f,
          +0.558991f, +0.507679f, -0.655585f,   +0.516808f, +0.579771f, -0.629901f,
          +0.462003f, +0.642953f, -0.610872f,   +0.393611f, +0.689327f, -0.608193f,
          +0.342409f, +0.670896f, -0.657766f,   +0.875067f, +0.139233f, -0.463543f,
          +0.850683f, +0.249544f, -0.462673f,   +0.819716f, +0.345881f, -0.456544f,
          +0.783608f, +0.434954f, -0.443593f,   +0.741244f, +0.520144f, -0.424273f,
          +0.690392f, +0.602192f, -0.400904f,   +0.628806f, +0.679579f, -0.377857f,
          +0.555209f, +0.749069f, -0.361439f,   +0.469850f, +0.806339f, -0.359247f,
          +0.419647f, +0.813586f, -0.402461f,   +0.965489f, +0.153064f, -0.210720f,
          +0.939109f, +0.271421f, -0.210725f,   +0.904173f, +0.373197f, -0.207835f,
          +0.861673f, +0.465885f, -0.201175f,   +0.810847f, +0.553173f, -0.191118f,
          +0.750356f, +0.636331f, -0.179022f,   +0.678973f, +0.714891f, -0.167110f,
          +0.596181f, +0.787047f, -0.158509f,   +0.502529f, +0.850201f, -0.156915f,
          +0.452997f, +0.870782f, -0.191135f,   -0.083062f, +0.995730f, +0.040279f,
          -0.085278f, +0.949363f, +0.302388f,   -0.165598f, +0.956594f, +0.239801f,
          -0.174404f, +0.984635f, -0.008763f,   -0.385780f, +0.893874f, +0.228394f,
          -0.406222f, +0.913740f, -0.007937f,   -0.620594f, +0.754065f, +0.215055f,
          -0.648319f, +0.761341f, -0.006534f,   -0.796357f, +0.570495f, +0.200877f,
          -0.824130f, +0.566382f, -0.004627f,   -0.886079f, +0.420038f, +0.196039f,
          -0.912149f, +0.409849f, -0.002680f,   -0.914364f, +0.347107f, +0.208458f,
          -0.942298f, +0.334773f, -0.000903f,   -0.902195f, +0.359013f, +0.239069f,
          -0.937349f, +0.348390f, +0.000777f,   -0.847840f, +0.450259f, +0.280062f,
          -0.893451f, +0.449155f, +0.002380f,   -0.735265f, +0.603404f, +0.308690f,
          -0.783096f, +0.621891f, +0.003527f,   -0.641041f, +0.713878f, +0.281860f,
          -0.674097f, +0.738041f, -0.029804f,   -0.080374f, +0.808473f, +0.583019f,
          -0.143910f, +0.843155f, +0.518054f,   -0.324902f, +0.803363f, +0.499046f,
          -0.528728f, +0.701539f, +0.477797f,   -0.694374f, +0.558000f, +0.454403f,
          -0.783660f, +0.433211f, +0.445203f,   -0.804372f, +0.370636f, +0.464344f,
          -0.772949f, +0.377478f, +0.509961f,   -0.697953f, +0.441435f, +0.563912f,
          -0.589411f, +0.544975f, +0.596319f,   -0.520023f, +0.629910f, +0.576878f,
          -0.058962f, +0.572195f, +0.817995f,   -0.097583f, +0.632304f, +0.768550f,
          -0.210186f, +0.623209f, +0.753281f,   -0.349723f, +0.576027f, +0.738842f,
          -0.479814f, +0.496845f, +0.723134f,   -0.559581f, +0.417700f, +0.715818f,
          -0.576109f, +0.371390f, +0.728126f,   -0.541150f, +0.366509f, +0.756854f,
          -0.474992f, +0.393304f, +0.787207f,   -0.398243f, +0.440766f, +0.804443f,
          -0.359156f, +0.496040f, +0.790538f,   -0.017871f, +0.271458f, +0.962284f,
          -0.026034f, +0.343987f, +0.938613f,   -0.049161f, +0.359592f, +0.931814f,
          -0.091505f, +0.364043f, +0.926876f,   -0.147960f, +0.353830f, +0.923532f,
          -0.201340f, +0.332977f, +0.921189f,   -0.233297f, +0.312576f, +0.920798f,
          -0.238494f, +0.300887f, +0.923357f,   -0.223457f, +0.297229f, +0.928290f,
          -0.197871f, +0.293927f, +0.935122f,   -0.190863f, +0.320816f, +0.927711f,
          +0.039710f, -0.054886f, +0.997703f,   +0.063563f, +0.017670f, +0.997821f,
          +0.133835f, +0.049766f, +0.989753f,   +0.195440f, +0.096891f, +0.975918f,
          +0.223688f, +0.145356f, +0.963761f,   +0.205024f, +0.179201f, +0.962212f,
          +0.150044f, +0.190392f, +0.970174f,   +0.085632f, +0.181112f, +0.979727f,
          +0.034742f, +0.154504f, +0.987381f,   +0.006038f, +0.100040f, +0.994965f,
          -0.018448f, +0.095650f, +0.995244f,   +0.106537f, -0.373777f, +0.921380f,
          +0.159910f, -0.306368f, +0.938386f,   +0.308762f, -0.255404f, +0.916207f,
          +0.447777f, -0.164672f, +0.878851f,   +0.531055f, -0.065240f, +0.844822f,
          +0.540189f, +0.008046f, +0.841505f,   +0.485235f, +0.038615f, +0.873531f,
          +0.391467f, +0.025576f, +0.919837f,   +0.292378f, -0.029163f, +0.955858f,
          +0.213952f, -0.141716f, +0.966510f,   +0.158572f, -0.183298f, +0.970184f,
          +0.171529f, -0.649885f, +0.740424f,   +0.248698f, -0.588513f, +0.769287f,
          +0.449869f, -0.509712f, +0.733356f,   +0.629518f, -0.373015f, +0.681591f,
          +0.734578f, -0.230986f, +0.637997f,   +0.761310f, -0.134834f, +0.634214f,
          +0.727439f, -0.103078f, +0.678386f,   +0.645451f, -0.136365f, +0.751530f,
          +0.531092f, -0.233087f, +0.814624f,   +0.407405f, -0.403022f, +0.819509f,
          +0.320167f, -0.478171f, +0.817830f,   +0.221566f, -0.846218f, +0.484587f,
          +0.315131f, -0.792256f, +0.522516f,   +0.542614f, -0.684105f, +0.487413f,
          +0.737369f, -0.510441f, +0.442422f,   +0.847900f, -0.339911f, +0.406850f,
          +0.884673f, -0.233086f, +0.403762f,   +0.873207f, -0.209037f, +0.440242f,
          +0.819045f, -0.269787f, +0.506340f,   +0.712324f, -0.412414f, +0.567898f,
          +0.548296f, -0.619116f, +0.562198f,   +0.431148f, -0.709662f, +0.557217f,
          +0.248485f, -0.946287f, +0.206871f,   +0.351491f, -0.902098f, +0.250348f,
          +0.588105f, -0.775258f, +0.230452f,   +0.786716f, -0.581803f, +0.206358f,
          +0.898183f, -0.397550f, +0.187674f,   +0.939814f, -0.286989f, +0.185438f,
          +0.941208f, -0.269789f, +0.203325f,   +0.906150f, -0.349889f, +0.237632f,
          +0.808589f, -0.522379f, +0.270748f,   +0.617913f, -0.740573f, +0.264074f,
          +0.482005f, -0.833661f, +0.269594f,   +0.253696f, -0.966476f, -0.039537f,
          +0.357365f, -0.933931f, +0.007928f,   +0.595358f, -0.803432f, +0.006827f,
          +0.795772f, -0.605574f, +0.005165f,   +0.908926f, -0.416944f, +0.003332f,
          +0.952709f, -0.303878f, +0.001752f,   +0.957945f, -0.286950f, +0.000543f,
          +0.928730f, -0.370756f, -0.000257f,   +0.835265f, -0.549848f, -0.000438f,
          +0.638966f, -0.769235f, +0.000307f,   +0.494965f, -0.868595f, +0.023483f,
          +0.241078f, -0.924458f, -0.295397f,   +0.343988f, -0.909469f, -0.233536f,
          +0.580459f, -0.785150f, -0.215885f,   +0.781802f, -0.592150f, -0.195304f,
          +0.896419f, -0.404973f, -0.180082f,   +0.939840f, -0.290074f, -0.180439f,
          +0.942307f, -0.268429f, -0.200007f,   +0.908936f, -0.344161f, -0.235347f,
          +0.814833f, -0.513325f, -0.269340f,   +0.628352f, -0.731903f, -0.263614f,
          +0.488347f, -0.845946f, -0.214226f,   +0.205584f, -0.793387f, -0.572951f,
          +0.307985f, -0.805248f, -0.506676f,   +0.535601f, -0.699341f, -0.473343f,
          +0.733480f, -0.525174f, -0.431507f,   +0.847540f, -0.350368f, -0.398646f,
          +0.886426f, -0.237905f, -0.397052f,   +0.876418f, -0.208428f, -0.434108f,
          +0.824367f, -0.264269f, -0.500580f,   +0.720894f, -0.403068f, -0.563780f,
          +0.560209f, -0.608534f, -0.562007f,   +0.447649f, -0.743958f, -0.496122f,
          +0.149352f, -0.567082f, -0.810008f,   +0.241720f, -0.606347f, -0.757572f,
          +0.443662f, -0.530119f, -0.722591f,   +0.627113f, -0.392378f, -0.672881f,
          +0.736566f, -0.245122f, -0.630386f,   +0.766382f, -0.142309f, -0.626423f,
          +0.734966f, -0.104473f, -0.670007f,   +0.655424f, -0.132776f, -0.743498f,
          +0.542869f, -0.225555f, -0.808961f,   +0.419872f, -0.391729f, -0.818691f,
          +0.350935f, -0.535505f, -0.768166f,   +0.083254f, -0.271927f, -0.958710f,
          +0.152588f, -0.326843f, -0.932679f,   +0.303045f, -0.279349f, -0.911113f,
          +0.446870f, -0.187881f, -0.874647f,   +0.536188f, -0.083141f, -0.839994f,
          +0.550388f, -0.002665f, -0.834905f,   +0.499096f, +0.034514f, -0.865859f,
          +0.407108f, +0.026483f, -0.912996f,   +0.307322f, -0.024473f, -0.951291f,
          +0.226693f, -0.131920f, -0.964991f,   +0.198731f, -0.247728f, -0.948228f,
          +0.020269f, +0.053359f, -0.998370f,   +0.055569f, -0.003457f, -0.998449f,
          +0.128138f, +0.024669f, -0.991450f,   +0.195475f, +0.071913f, -0.978069f,
          +0.231632f, +0.125074f, -0.964730f,   +0.220217f, +0.166273f, -0.961175f,
          +0.169354f, +0.184565f, -0.968119f,   +0.105033f, +0.180318f, -0.977984f,
          +0.051304f, +0.157134f, -0.986244f,   +0.018859f, +0.107474f, -0.994029f,
          +0.022944f, +0.042006f, -0.998854f,   -0.030252f, +0.375055f, -0.926509f,
          -0.034683f, +0.323654f, -0.945540f,   -0.055154f, +0.335809f, -0.940314f,
          -0.091676f, +0.340414f, -0.935796f,   -0.140161f, +0.334652f, -0.931860f,
          -0.186357f, +0.321058f, -0.928544f,   -0.214622f, +0.307515f, -0.927023f,
          -0.220017f, +0.300286f, -0.928128f,   -0.207772f, +0.299414f, -0.931226f,
          -0.185763f, +0.300039f, -0.935665f,   -0.150867f, +0.282998f, -0.947181f,
          -0.063200f, +0.658662f, -0.749780f,   -0.106657f, +0.614663f, -0.781545f,
          -0.216828f, +0.603342f, -0.767440f,   -0.351534f, +0.556736f, -0.752641f,
          -0.475776f, +0.481832f, -0.735850f,   -0.550945f, +0.409018f, -0.727436f,
          -0.564755f, +0.368359f, -0.738487f,   -0.528700f, +0.367261f, -0.765242f,
          -0.463127f, +0.396520f, -0.792644f,   -0.388076f, +0.447210f, -0.805854f,
          -0.319592f, +0.471236f, -0.822069f,   -0.078352f, +0.865587f, -0.494590f,
          -0.152986f, +0.830620f, -0.535411f,   -0.332455f, +0.789275f, -0.516254f,
          -0.532856f, +0.687640f, -0.493168f,   -0.694569f, +0.547210f, -0.467050f,
          -0.781226f, +0.427013f, -0.455352f,   -0.800442f, +0.368713f, -0.472592f,
          -0.767778f, +0.379097f, -0.516530f,   -0.691648f, +0.446062f, -0.568025f,
          -0.582196f, +0.552704f, -0.596294f,   -0.482879f, +0.615214f, -0.623169f,
          -0.081876f, +0.973740f, -0.212429f,   -0.173942f, +0.950270f, -0.258323f,
          -0.394037f, +0.885672f, -0.245602f,   -0.626691f, +0.744713f, -0.229481f,
          -0.799229f, +0.562547f, -0.211600f,   -0.886732f, +0.415200f, -0.203261f,
          -0.913937f, +0.345587f, -0.212813f,   -0.900999f, +0.360753f, -0.240953f,
          -0.845313f, +0.455418f, -0.279357f,   -0.730184f, +0.611518f, -0.304758f,
          -0.616543f, +0.709213f, -0.341894f,   -0.492737f, +0.829935f, +0.261571f,
          -0.521993f, +0.852936f, -0.004774f,   -0.415684f, +0.881391f, +0.224404f,
          -0.440118f, +0.897913f, -0.006931f,   -0.326898f, +0.928776f, +0.174682f,
          -0.345359f, +0.938427f, -0.009050f,   -0.209561f, +0.972545f, +0.101192f,
          -0.218727f, +0.975718f, -0.011556f,   -0.016746f, +0.999469f, -0.027953f,
          -0.005501f, +0.999870f, -0.015167f,   +0.377637f, +0.877761f, -0.294832f,
          +0.467694f, +0.883646f, -0.020780f,   +0.811035f, +0.148655f, -0.565795f,
          +0.998397f, +0.053920f, -0.017198f,   +0.746311f, -0.448662f, -0.491652f,
          +0.841995f, -0.539443f, -0.006695f,   +0.631010f, -0.668912f, -0.392916f,
          +0.688332f, -0.725391f, -0.002708f,   +0.609404f, -0.738026f, -0.289732f,
          +0.636625f, -0.768534f, +0.063752f,   -0.404344f, +0.751959f, +0.520637f,
          -0.342032f, +0.825526f, +0.448911f,   -0.270293f, +0.896094f, +0.352075f,
          -0.176454f, +0.961589f, +0.210264f,   -0.029810f, +0.999242f, -0.025030f,
          +0.223861f, +0.873498f, -0.432306f,   +0.481602f, +0.321862f, -0.815147f,
          +0.518983f, -0.225036f, -0.824631f,   +0.478956f, -0.505298f, -0.717826f,
          +0.491291f, -0.612945f, -0.618815f,   -0.286098f, +0.624237f, +0.726963f,
          -0.245364f, +0.735607f, +0.631410f,   -0.197483f, +0.844598f, +0.497650f,
          -0.134325f, +0.944031f, +0.301268f,   -0.039714f, +0.999199f, -0.004816f,
          +0.102079f, +0.886430f, -0.451465f,   +0.240826f, +0.476404f, -0.845602f,
          +0.298166f, +0.016485f, -0.954372f,   +0.304341f, -0.289824f, -0.907402f,
          +0.334124f, -0.426337f, -0.840594f,   -0.153688f, +0.444351f, +0.882572f,
          -0.137364f, +0.608456f, +0.781609f,   -0.116789f, +0.772937f, +0.623642f,
          -0.087577f, +0.919962f, +0.382100f,   -0.042956f, +0.998760f, +0.025147f,
          +0.019713f, +0.905622f, -0.423628f,   +0.082424f, +0.599033f, -0.796471f,
          +0.122561f, +0.234535f, -0.964351f,   +0.141146f, -0.055050f, -0.988457f,
          +0.176017f, -0.206160f, -0.962557f,   -0.004396f, +0.189743f, +0.981824f,
          -0.010485f, +0.417290f, +0.908713f,   -0.019913f, +0.663368f, +0.748029f,
          -0.030922f, +0.885021f, +0.464523f,   -0.038918f, +0.997526f, +0.058547f,
          -0.038452f, +0.924745f, -0.378639f,   -0.029393f, +0.699488f, -0.714039f,
          -0.017283f, +0.430580f, -0.902387f,   -0.007903f, +0.192514f, -0.981262f,
          +0.023577f, +0.044107f, -0.998749f,   +0.160802f, -0.151903f, +0.975227f,
          +0.149302f, +0.116651f, +0.981887f,   +0.115736f, +0.471957f, +0.873992f,
          +0.050821f, +0.827899f, +0.558570f,   -0.026623f, +0.995462f, +0.091358f,
          -0.083796f, +0.942345f, -0.323981f,   -0.115587f, +0.785704f, -0.607708f,
          -0.131853f, +0.606907f, -0.783759f,   -0.142284f, +0.441588f, -0.885864f,
          -0.121741f, +0.316947f, -0.940598f,   +0.307300f, -0.511213f, +0.802638f,
          +0.325450f, -0.292562f, +0.899161f,   +0.315210f, +0.120714f, +0.941313f,
          +0.190909f, +0.720717f, +0.666424f,   -0.003665f, +0.993020f, +0.117893f,
          -0.122073f, +0.958490f, -0.257670f,   -0.184971f, +0.859626f, -0.476266f,
          -0.223739f, +0.756514f, -0.614513f,   -0.253310f, +0.662834f, -0.704617f,
          -0.247808f, +0.577756f, -0.777682f,   +0.391620f, -0.762190f, +0.515462f,
          +0.444553f, -0.651471f, +0.614783f,   +0.522301f, -0.369970f, +0.768325f,
          +0.453936f, +0.491214f, +0.743405f,   +0.032079f, +0.991595f, +0.125340f,
          -0.154152f, +0.972195f, -0.176277f,   -0.237973f, +0.916987f, -0.320161f,
          -0.289708f, +0.865833f, -0.407926f,   -0.329497f, +0.821559f, -0.465266f,
          -0.336404f, +0.775220f, -0.534665f,   +0.421003f, -0.877145f, +0.231028f,
          +0.483621f, -0.829573f, +0.279141f,   +0.606601f, -0.702498f, +0.372199f,
          +0.825282f, +0.070625f, +0.560287f,   +0.074735f, +0.992835f, +0.093239f,
          -0.176331f, +0.980937f, -0.081665f,   -0.270773f, +0.950823f, -0.150392f,
          -0.326682f, +0.925836f, -0.190016f,   -0.368404f, +0.904523f, -0.214747f,
          -0.381827f, +0.883980f, -0.269793f,   +0.425155f, -0.905115f, +0.003163f,
          +0.487150f, -0.873297f, +0.006166f,   +0.611576f, -0.791084f, +0.012703f,
          +0.977309f, -0.208263f, +0.038641f,   +0.107131f, +0.993985f, +0.022712f,
          -0.181143f, +0.983400f, +0.010532f,   -0.279746f, +0.960053f, +0.006394f,
          -0.336455f, +0.941690f, +0.004312f,   -0.377932f, +0.925828f, +0.003109f,
          -0.394655f, +0.918042f, -0.038032f,   +0.420310f, -0.879733f, -0.222284f,
          +0.482683f, -0.835447f, -0.262765f,   +0.605343f, -0.720079f, -0.339185f,
          +0.865952f, -0.013603f, -0.499942f,   +0.098237f, +0.993380f, -0.059537f,
          -0.170509f, +0.980074f, +0.101890f,   -0.268242f, +0.949354f, +0.163627f,
          -0.325275f, +0.924361f, +0.199381f,   -0.367491f, +0.903119f, +0.222097f,
          -0.388221f, +0.903219f, +0.182972f,   +0.392860f, -0.768973f, -0.504324f,
          +0.447801f, -0.666498f, -0.596032f,   +0.533903f, -0.407833f, -0.740689f,
          +0.503545f, +0.443167f, -0.741651f,   +0.052467f, +0.992791f, -0.107764f,
          -0.147280f, +0.969917f, +0.193829f,   -0.234596f, +0.913132f, +0.333399f,
          -0.287645f, +0.861573f, +0.418272f,   -0.327999f, +0.817074f, +0.474137f,
          -0.354951f, +0.823932f, +0.441753f,   +0.311710f, -0.524022f, -0.792615f,
          +0.334440f, -0.316210f, -0.887784f,   +0.334222f, +0.084141f, -0.938731f,
          +0.218356f, +0.701542f, -0.678351f,   +0.012126f, +0.993602f, -0.112285f,
          -0.114337f, +0.955982f, +0.270233f,   -0.180344f, +0.854217f, +0.487636f,
          -0.220421f, +0.749735f, +0.623949f,   -0.250326f, +0.654832f, +0.713114f,
          -0.281692f, +0.657454f, +0.698859f,   +0.167701f, -0.167664f, -0.971476f,
          +0.159944f, +0.095035f, -0.982541f,   +0.130461f, +0.450380f, -0.883254f,
          +0.066795f, +0.818192f, -0.571051f,   -0.014189f, +0.995528f, -0.093396f,
          -0.075513f, +0.940893f, +0.330180f,   -0.109857f, +0.780637f, +0.615254f,
          -0.127317f, +0.599525f, +0.790164f,   -0.137899f, +0.432128f, +0.891207f,
          -0.167170f, +0.412263f, +0.895597f,   +0.002691f, +0.176201f, -0.984351f,
          -0.001471f, +0.402177f, -0.915561f,   -0.009413f, +0.650595f, -0.759366f,
          -0.019844f, +0.878591f, -0.477163f,   -0.028579f, +0.997405f, -0.066086f,
          -0.029946f, +0.925242f, +0.378194f,   -0.022888f, +0.697198f, +0.716513f,
          -0.012110f, +0.425628f, +0.904817f,   -0.002981f, +0.185109f, +0.982714f,
          -0.026551f, +0.134046f, +0.990619f,   -0.147433f, +0.434416f, -0.888564f,
          -0.129928f, +0.598121f, -0.790803f,   -0.108468f, +0.764224f, -0.635764f,
          -0.078628f, +0.914993f, -0.395734f,   -0.033735f, +0.998731f, -0.037403f,
          +0.028393f, +0.908873f, +0.416104f,   +0.089531f, +0.601542f, +0.793808f,
          +0.128068f, +0.234419f, +0.963663f,   +0.146073f, -0.058220f, +0.987559f,
          +0.123761f, -0.131508f, +0.983559f,   -0.280652f, +0.617434f, -0.734853f,
          -0.238872f, +0.728360f, -0.642209f,   -0.190151f, +0.838170f, -0.511189f,
          -0.126242f, +0.939939f, -0.317140f,   -0.030895f, +0.999446f, -0.012405f,
          +0.111322f, +0.892687f, +0.436712f,   +0.249251f, +0.484654f, +0.838442f,
          +0.304655f, +0.021070f, +0.952230f,   +0.309563f, -0.289355f, +0.905784f,
          +0.278297f, -0.365718f, +0.888145f,   -0.399667f, +0.748007f, -0.529860f,
          -0.336299f, +0.820921f, -0.461510f,   -0.263641f, +0.891713f, -0.367888f,
          -0.168793f, +0.958605f, -0.229316f,   -0.020845f, +0.999779f, +0.002599f,
          +0.235063f, +0.881469f, +0.409582f,   +0.494243f, +0.332398f, +0.803266f,
          +0.527688f, -0.220183f, +0.820405f,   +0.484557f, -0.504028f, +0.714955f,
          +0.434703f, -0.564503f, +0.701691f,   -0.489074f, +0.828944f, -0.271399f,
          -0.411203f, +0.879840f, -0.238314f,   -0.321524f, +0.927125f, -0.192514f,
          -0.202667f, +0.971474f, -0.123140f,   -0.007166f, +0.999974f, +0.000785f,
          +0.394201f, +0.880610f, +0.262928f,   +0.827744f, +0.146495f, +0.541644f,
          +0.752949f, -0.450124f, +0.480059f,   +0.634012f, -0.669582f, +0.386896f,
          +0.571230f, -0.710640f, +0.410715f,   +0.124933f, +0.991892f, +0.023282f,
          +0.127452f, +0.991838f, +0.003656f,   +0.000000f, +1.000000f, -0.000000f,
          +0.356899f, +0.932201f, +0.060203f,   +0.363140f, +0.931724f, +0.004327f,
          +0.851327f, +0.506358f, +0.137271f,   +0.863762f, +0.503882f, +0.004132f,
          +0.939241f, -0.310008f, +0.147379f,   +0.950358f, -0.311158f, +0.000565f,
          +0.789162f, -0.602062f, +0.121428f,   +0.798419f, -0.602099f, -0.001906f,
          +0.737621f, -0.665881f, +0.111885f,   +0.746299f, -0.665602f, -0.003428f,
          +0.770906f, -0.626261f, +0.116192f,   +0.780058f, -0.625691f, -0.004447f,
          +0.894189f, -0.426493f, +0.136124f,   +0.905079f, -0.425225f, -0.004097f,
          +0.967684f, +0.201170f, +0.152048f,   +0.979091f, +0.203421f, -0.000122f,
          +0.601451f, +0.792937f, +0.097510f,   +0.608520f, +0.793534f, +0.002747f,
          +0.119607f, +0.991907f, +0.042609f,   +0.342788f, +0.932280f, +0.115540f,
          +0.819196f, +0.506239f, +0.269519f,   +0.904062f, -0.310964f, +0.293213f,
          +0.759674f, -0.602833f, +0.243902f,   +0.710274f, -0.666526f, +0.226394f,
          +0.742544f, -0.626855f, +0.235969f,   +0.861397f, -0.426876f, +0.275266f,
          +0.931433f, +0.201983f, +0.302715f,   +0.577775f, +0.793548f, +0.190941f,
          +0.111325f, +0.991921f, +0.060824f,   +0.320150f, +0.932371f, +0.167894f,
          +0.766627f, +0.506218f, +0.395001f,   +0.846442f, -0.311839f, +0.431617f,
          +0.711355f, -0.603583f, +0.360085f,   +0.665315f, -0.667172f, +0.335019f,
          +0.695760f, -0.627469f, +0.349573f,   +0.807222f, -0.427338f, +0.407153f,
          +0.872175f, +0.202604f, +0.445267f,   +0.539983f, +0.794081f, +0.279023f,
          +0.100378f, +0.991928f, +0.077480f,   +0.289787f, +0.932414f, +0.215936f,
          +0.695277f, +0.506105f, +0.510340f,   +0.768134f, -0.312501f, +0.558850f,
          +0.645788f, -0.604117f, +0.466904f,   +0.604267f, -0.667624f, +0.434901f,
          +0.632149f, -0.627891f, +0.454027f,   +0.733467f, -0.427633f, +0.528352f,
          +0.791804f, +0.203089f, +0.576021f,   +0.489389f, +0.794463f, +0.359620f,
          +0.087051f, +0.991924f, +0.092240f,   +0.252516f, +0.932379f, +0.258660f,
          +0.607041f, +0.505793f, +0.612922f,   +0.671202f, -0.312905f, +0.671996f,
          +0.564727f, -0.604350f, +0.562000f,   +0.528776f, -0.667791f, +0.523881f,
          +0.553439f, -0.628016f, +0.547084f,   +0.642125f, -0.427630f, +0.636246f,
          +0.692432f, +0.203523f, +0.692182f,   +0.427306f, +0.794684f, +0.431146f,
          +0.071622f, +0.991909f, +0.104822f,   +0.209138f, +0.932253f, +0.295239f,
          +0.503908f, +0.505215f, +0.700596f,   +0.557832f, -0.313056f, +0.768648f,
          +0.469964f, -0.604263f, +0.643428f,   +0.440499f, -0.667645f, +0.600175f,
          +0.461345f, -0.627807f, +0.626912f,   +0.535188f, -0.427274f, +0.728705f,
          +0.576196f, +0.203978f, +0.791449f,   +0.354994f, +0.794756f, +0.492283f,
          +0.054386f, +0.991881f, +0.114952f,   +0.160497f, +0.932035f, +0.324888f,
          +0.388045f, +0.504344f, +0.771400f,   +0.430433f, -0.313008f, +0.846613f,
          +0.363432f, -0.603890f, +0.709390f,   +0.341201f, -0.667213f, +0.662125f,
          +0.357685f, -0.627285f, +0.691791f,   +0.414762f, -0.426572f, +0.803747f,
          +0.445393f, +0.204489f, +0.871670f,   +0.273772f, +0.794705f, +0.541750f,
          +0.035705f, +0.991845f, +0.122345f,   +0.107615f, +0.931744f, +0.346802f,
          +0.262085f, +0.503212f, +0.823462f,   +0.291935f, -0.312862f, +0.903820f,
          +0.247477f, -0.603326f, +0.758125f,   +0.233015f, -0.666582f, +0.708077f,
          +0.244659f, -0.626542f, +0.739991f,   +0.283400f, -0.425614f, +0.859382f,
          +0.302815f, +0.205082f, +0.930722f,   +0.185276f, +0.794579f, +0.578202f,
          +0.016067f, +0.991809f, +0.126718f,   +0.051848f, +0.931436f, +0.360193f,
          +0.129401f, +0.501956f, +0.855158f,   +0.146113f, -0.312795f, +0.938515f,
          +0.125166f, -0.602775f, +0.788033f,   +0.118745f, -0.665941f, +0.736493f,
          +0.125166f, -0.625773f, +0.769897f,   +0.144488f, -0.424606f, +0.893774f,
          +0.152182f, +0.205773f, +0.966695f,   +0.091723f, +0.794484f, +0.600318f,
          -0.003656f, +0.991838f, +0.127452f,   -0.004328f, +0.931724f, +0.363140f,
          -0.004131f, +0.503883f, +0.863762f,   -0.000565f, -0.311158f, +0.950358f,
          +0.001906f, -0.602100f, +0.798419f,   +0.003429f, -0.665601f, +0.746300f,
          +0.004446f, -0.625692f, +0.780058f,   +0.004097f, -0.425225f, +0.905078f,
          +0.000124f, +0.203422f, +0.979091f,   -0.002747f, +0.793534f, +0.608520f,
          -0.023281f, +0.991892f, +0.124933f,   -0.060203f, +0.932201f, +0.356900f,
          -0.137274f, +0.506362f, +0.851325f,   -0.147380f, -0.310008f, +0.939241f,
          -0.121428f, -0.602062f, +0.789162f,   -0.111887f, -0.665880f, +0.737621f,
          -0.116194f, -0.626262f, +0.770905f,   -0.136127f, -0.426494f, +0.894188f,
          -0.152056f, +0.201173f, +0.967682f,   -0.097510f, +0.792937f, +0.601451f,
          -0.042609f, +0.991907f, +0.119607f,   -0.115538f, +0.932280f, +0.342789f,
          -0.269516f, +0.506239f, +0.819197f,   -0.293213f, -0.310961f, +0.904063f,
          -0.243902f, -0.602834f, +0.759673f,   -0.226395f, -0.666526f, +0.710273f,
          -0.235972f, -0.626854f, +0.742544f,   -0.275263f, -0.426878f, +0.861397f,
          -0.302717f, +0.201986f, +0.931431f,   -0.190942f, +0.793549f, +0.577773f,
          -0.060825f, +0.991921f, +0.111325f,   -0.167894f, +0.932371f, +0.320150f,
          -0.395000f, +0.506218f, +0.766628f,   -0.431616f, -0.311838f, +0.846443f,
          -0.360084f, -0.603584f, +0.711355f,   -0.335018f, -0.667172f, +0.665315f,
          -0.349573f, -0.627469f, +0.695760f,   -0.407157f, -0.427337f, +0.807221f,
          -0.445266f, +0.202602f, +0.872176f,   -0.279023f, +0.794082f, +0.539982f,
          -0.077480f, +0.991928f, +0.100378f,   -0.215935f, +0.932414f, +0.289786f,
          -0.510340f, +0.506105f, +0.695278f,   -0.558851f, -0.312501f, +0.768133f,
          -0.466904f, -0.604118f, +0.645788f,   -0.434901f, -0.667625f, +0.604267f,
          -0.454023f, -0.627892f, +0.632151f,   -0.528352f, -0.427633f, +0.733468f,
          -0.576022f, +0.203090f, +0.791804f,   -0.359619f, +0.794464f, +0.489389f,
          -0.092240f, +0.991924f, +0.087052f,   -0.258660f, +0.932379f, +0.252515f,
          -0.612923f, +0.505792f, +0.607041f,   -0.671996f, -0.312905f, +0.671202f,
          -0.562000f, -0.604350f, +0.564727f,   -0.523881f, -0.667791f, +0.528776f,
          -0.547084f, -0.628016f, +0.553440f,   -0.636240f, -0.427631f, +0.642130f,
          -0.692176f, +0.203523f, +0.692438f,   -0.431145f, +0.794684f, +0.427307f,
          -0.104822f, +0.991909f, +0.071622f,   -0.295239f, +0.932253f, +0.209138f,
          -0.700596f, +0.505214f, +0.503909f,   -0.768649f, -0.313057f, +0.557830f,
          -0.643429f, -0.604263f, +0.469963f,   -0.600174f, -0.667645f, +0.440500f,
          -0.626912f, -0.627807f, +0.461345f,   -0.728707f, -0.427272f, +0.535187f,
          -0.791447f, +0.203974f, +0.576200f,   -0.492283f, +0.794755f, +0.354995f,
          -0.114952f, +0.991881f, +0.054386f,   -0.324888f, +0.932035f, +0.160497f,
          -0.771399f, +0.504344f, +0.388048f,   -0.846612f, -0.313010f, +0.430434f,
          -0.709391f, -0.603889f, +0.363432f,   -0.662126f, -0.667212f, +0.341200f,
          -0.691789f, -0.627286f, +0.357687f,   -0.803746f, -0.426572f, +0.414764f,
          -0.871672f, +0.204489f, +0.445390f,   -0.541752f, +0.794703f, +0.273773f,
          -0.122345f, +0.991845f, +0.035705f,   -0.346802f, +0.931744f, +0.107616f,
          -0.823462f, +0.503211f, +0.262085f,   -0.903819f, -0.312865f, +0.291937f,
          -0.758123f, -0.603327f, +0.247480f,   -0.708078f, -0.666582f, +0.233013f,
          -0.739994f, -0.626541f, +0.244655f,   -0.859381f, -0.425614f, +0.283401f,
          -0.930723f, +0.205082f, +0.302813f,   -0.578203f, +0.794578f, +0.185275f,
          -0.126717f, +0.991809f, +0.016067f,   -0.360193f, +0.931436f, +0.051848f,
          -0.855159f, +0.501955f, +0.129400f,   -0.938515f, -0.312796f, +0.146112f,
          -0.788033f, -0.602775f, +0.125166f,   -0.736492f, -0.665942f, +0.118746f,
          -0.769898f, -0.625772f, +0.125164f,   -0.893775f, -0.424605f, +0.144484f,
          -0.966696f, +0.205770f, +0.152177f,   -0.600320f, +0.794483f, +0.091722f,
          -0.127452f, +0.991838f, -0.003656f,   -0.363140f, +0.931724f, -0.004327f,
          -0.863763f, +0.503882f, -0.004132f,   -0.950358f, -0.311158f, -0.000565f,
          -0.798419f, -0.602099f, +0.001906f,   -0.746299f, -0.665602f, +0.003428f,
          -0.780058f, -0.625691f, +0.004447f,   -0.905078f, -0.425225f, +0.004097f,
          -0.979091f, +0.203421f, +0.000122f,   -0.608520f, +0.793534f, -0.002747f,
          -0.124933f, +0.991892f, -0.023282f,   -0.356899f, +0.932201f, -0.060203f,
          -0.851327f, +0.506358f, -0.137271f,   -0.939241f, -0.310008f, -0.147379f,
          -0.789162f, -0.602062f, -0.121428f,   -0.737621f, -0.665881f, -0.111885f,
          -0.770906f, -0.626261f, -0.116192f,   -0.894189f, -0.426493f, -0.136124f,
          -0.967684f, +0.201169f, -0.152049f,   -0.601450f, +0.792937f, -0.097510f,
          -0.119607f, +0.991906f, -0.042609f,   -0.342788f, +0.932280f, -0.115540f,
          -0.819195f, +0.506239f, -0.269520f,   -0.904061f, -0.310965f, -0.293213f,
          -0.759674f, -0.602832f, -0.243903f,   -0.710274f, -0.666526f, -0.226394f,
          -0.742544f, -0.626855f, -0.235969f,   -0.861398f, -0.426876f, -0.275265f,
          -0.931433f, +0.201981f, -0.302714f,   -0.577774f, +0.793548f, -0.190941f,
          -0.111326f, +0.991921f, -0.060824f,   -0.320150f, +0.932371f, -0.167895f,
          -0.766627f, +0.506218f, -0.395001f,   -0.846442f, -0.311839f, -0.431617f,
          -0.711355f, -0.603583f, -0.360085f,   -0.665316f, -0.667171f, -0.335019f,
          -0.695760f, -0.627469f, -0.349573f,   -0.807222f, -0.427339f, -0.407154f,
          -0.872175f, +0.202602f, -0.445267f,   -0.539984f, +0.794080f, -0.279023f,
          -0.100378f, +0.991928f, -0.077480f,   -0.289787f, +0.932414f, -0.215936f,
          -0.695277f, +0.506106f, -0.510340f,   -0.768133f, -0.312501f, -0.558851f,
          -0.645788f, -0.604118f, -0.466905f,   -0.604267f, -0.667623f, -0.434903f,
          -0.632148f, -0.627891f, -0.454028f,   -0.733466f, -0.427634f, -0.528353f,
          -0.791805f, +0.203087f, -0.576022f,   -0.489389f, +0.794463f, -0.359620f,
          -0.087051f, +0.991924f, -0.092240f,   -0.252516f, +0.932379f, -0.258660f,
          -0.607043f, +0.505791f, -0.612922f,   -0.671201f, -0.312904f, -0.671997f,
          -0.564727f, -0.604350f, -0.562000f,   -0.528777f, -0.667790f, -0.523881f,
          -0.553439f, -0.628016f, -0.547085f,   -0.642125f, -0.427630f, -0.636245f,
          -0.692433f, +0.203522f, -0.692181f,   -0.427307f, +0.794683f, -0.431146f,
          -0.071622f, +0.991909f, -0.104822f,   -0.209139f, +0.932253f, -0.295239f,
          -0.503910f, +0.505214f, -0.700595f,   -0.557833f, -0.313057f, -0.768646f,
          -0.469966f, -0.604262f, -0.643428f,   -0.440501f, -0.667645f, -0.600175f,
          -0.461345f, -0.627808f, -0.626912f,   -0.535188f, -0.427274f, -0.728705f,
          -0.576198f, +0.203975f, -0.791448f,   -0.354995f, +0.794755f, -0.492283f,
          -0.054385f, +0.991881f, -0.114953f,   -0.160497f, +0.932035f, -0.324888f,
          -0.388047f, +0.504343f, -0.771400f,   -0.430433f, -0.313010f, -0.846612f,
          -0.363432f, -0.603889f, -0.709391f,   -0.341199f, -0.667213f, -0.662126f,
          -0.357683f, -0.627286f, -0.691792f,   -0.414762f, -0.426572f, -0.803747f,
          -0.445391f, +0.204489f, -0.871672f,   -0.273772f, +0.794705f, -0.541750f,
          -0.035705f, +0.991845f, -0.122345f,   -0.107615f, +0.931744f, -0.346802f,
          -0.262082f, +0.503212f, -0.823462f,   -0.291936f, -0.312864f, -0.903819f,
          -0.247478f, -0.603327f, -0.758124f,   -0.233012f, -0.666582f, -0.708078f,
          -0.244655f, -0.626541f, -0.739994f,   -0.283402f, -0.425613f, -0.859382f,
          -0.302818f, +0.205079f, -0.930722f,   -0.185275f, +0.794579f, -0.578201f,
          -0.016067f, +0.991809f, -0.126717f,   -0.051848f, +0.931436f, -0.360193f,
          -0.129400f, +0.501955f, -0.855159f,   -0.146110f, -0.312795f, -0.938515f,
          -0.125165f, -0.602775f, -0.788033f,   -0.118746f, -0.665942f, -0.736492f,
          -0.125164f, -0.625772f, -0.769898f,   -0.144483f, -0.424604f, -0.893776f,
          -0.152178f, +0.205770f, -0.966696f,   -0.091722f, +0.794484f, -0.600319f,
          +0.003656f, +0.991838f, -0.127452f,   +0.004327f, +0.931724f, -0.363140f,
          +0.004132f, +0.503882f, -0.863763f,   +0.000565f, -0.311158f, -0.950358f,
          -0.001906f, -0.602100f, -0.798419f,   -0.003428f, -0.665602f, -0.746299f,
          -0.004447f, -0.625692f, -0.780058f,   -0.004099f, -0.425224f, -0.905079f,
          -0.000122f, +0.203420f, -0.979092f,   +0.002747f, +0.793534f, -0.608520f,
          +0.023282f, +0.991892f, -0.124933f,   +0.060203f, +0.932201f, -0.356900f,
          +0.137270f, +0.506360f, -0.851326f,   +0.147378f, -0.310009f, -0.939241f,
          +0.121428f, -0.602062f, -0.789162f,   +0.111885f, -0.665881f, -0.737621f,
          +0.116191f, -0.626261f, -0.770907f,   +0.136123f, -0.426492f, -0.894190f,
          +0.152049f, +0.201170f, -0.967684f,   +0.097509f, +0.792938f, -0.601450f,
          +0.042609f, +0.991907f, -0.119607f,   +0.115538f, +0.932281f, -0.342789f,
          +0.269518f, +0.506240f, -0.819196f,   +0.293212f, -0.310963f, -0.904062f,
          +0.243900f, -0.602834f, -0.759673f,   +0.226394f, -0.666526f, -0.710274f,
          +0.235971f, -0.626854f, -0.742544f,   +0.275266f, -0.426876f, -0.861397f,
          +0.302718f, +0.201982f, -0.931432f,   +0.190943f, +0.793548f, -0.577773f,
          +0.060825f, +0.991921f, -0.111325f,   +0.167894f, +0.932371f, -0.320150f,
          +0.395000f, +0.506216f, -0.766629f,   +0.431619f, -0.311838f, -0.846441f,
          +0.360085f, -0.603583f, -0.711355f,   +0.335015f, -0.667172f, -0.665316f,
          +0.349570f, -0.627469f, -0.695761f,   +0.407157f, -0.427337f, -0.807221f,
          +0.445268f, +0.202600f, -0.872175f,   +0.279024f, +0.794081f, -0.539983f,
          +0.077480f, +0.991928f, -0.100378f,   +0.215937f, +0.932414f, -0.289785f,
          +0.510342f, +0.506105f, -0.695276f,   +0.558851f, -0.312504f, -0.768133f,
          +0.466907f, -0.604116f, -0.645787f,   +0.434904f, -0.667624f, -0.604266f,
          +0.454024f, -0.627892f, -0.632150f,   +0.528351f, -0.427633f, -0.733467f,
          +0.576020f, +0.203088f, -0.791805f,   +0.359620f, +0.794463f, -0.489389f,
          +0.092240f, +0.991924f, -0.087051f,   +0.258660f, +0.932379f, -0.252516f,
          +0.612923f, +0.505794f, -0.607040f,   +0.671995f, -0.312907f, -0.671202f,
          +0.562000f, -0.604350f, -0.564728f,   +0.523882f, -0.667790f, -0.528777f,
          +0.547084f, -0.628016f, -0.553440f,   +0.636241f, -0.427631f, -0.642129f,
          +0.692180f, +0.203522f, -0.692434f,   +0.431146f, +0.794683f, -0.427307f,
          +0.104822f, +0.991909f, -0.071622f,   +0.295239f, +0.932253f, -0.209139f,
          +0.700595f, +0.505214f, -0.503909f,   +0.768648f, -0.313056f, -0.557832f,
          +0.643428f, -0.604263f, -0.469964f,   +0.600174f, -0.667645f, -0.440501f,
          +0.626913f, -0.627807f, -0.461345f,   +0.728704f, -0.427274f, -0.535190f,
          +0.791446f, +0.203974f, -0.576201f,   +0.492284f, +0.794755f, -0.354995f,
          +0.114953f, +0.991881f, -0.054386f,   +0.324888f, +0.932035f, -0.160497f,
          +0.771399f, +0.504344f, -0.388048f,   +0.846612f, -0.313010f, -0.430433f,
          +0.709391f, -0.603889f, -0.363432f,   +0.662126f, -0.667212f, -0.341201f,
          +0.691790f, -0.627286f, -0.357686f,   +0.803748f, -0.426571f, -0.414760f,
          +0.871672f, +0.204487f, -0.445390f,   +0.541751f, +0.794704f, -0.273773f,
          +0.122345f, +0.991845f, -0.035705f,   +0.346802f, +0.931744f, -0.107617f,
          +0.823462f, +0.503212f, -0.262085f,   +0.903819f, -0.312864f, -0.291937f,
          +0.758123f, -0.603327f, -0.247480f,   +0.708078f, -0.666581f, -0.233013f,
          +0.739993f, -0.626541f, -0.244656f,   +0.859381f, -0.425615f, -0.283401f,
          +0.930724f, +0.205080f, -0.302812f,   +0.578203f, +0.794578f, -0.185275f,
          +0.126717f, +0.991809f, -0.016067f,   +0.360193f, +0.931436f, -0.051848f,
          +0.855159f, +0.501954f, -0.129400f,   +0.938515f, -0.312796f, -0.146112f,
          +0.788033f, -0.602775f, -0.125166f,   +0.736492f, -0.665942f, -0.118746f,
          +0.769898f, -0.625772f, -0.125164f,   +0.893775f, -0.424606f, -0.144484f,
          +0.966696f, +0.205770f, -0.152178f,   +0.600319f, +0.794484f, -0.091722f,
          +0.348933f, +0.935400f, +0.057200f,   +0.353373f, +0.935480f, +0.002199f,
          +0.235950f, +0.970999f, +0.038596f,   +0.239028f, +0.971012f, +0.001406f,
          +0.179591f, +0.983306f, +0.029254f,   +0.181971f, +0.983303f, +0.000952f,
          +0.153830f, +0.987781f, +0.024992f,   +0.155908f, +0.987771f, +0.000753f,
          +0.149962f, +0.988392f, +0.024347f,   +0.152038f, +0.988374f, +0.000717f,
          +0.168092f, +0.985393f, +0.027289f,   +0.170481f, +0.985361f, +0.000801f,
          +0.217888f, +0.975334f, +0.035342f,   +0.221063f, +0.975259f, +0.001007f,
          +0.331140f, +0.942060f, +0.053573f,   +0.336076f, +0.941834f, +0.001397f,
          +0.609278f, +0.786860f, +0.098142f,   +0.618474f, +0.785802f, +0.002154f,
          +0.841730f, +0.528519f, +0.110265f,   +0.848732f, +0.528369f, -0.021916f,
          +0.335041f, +0.935597f, +0.111380f,   +0.226574f, +0.971084f, +0.075237f,
          +0.172484f, +0.983353f, +0.057151f,   +0.147766f, +0.987813f, +0.048892f,
          +0.144070f, +0.988420f, +0.047651f,   +0.161509f, +0.985424f, +0.053416f,
          +0.209392f, +0.975379f, +0.069219f,   +0.318317f, +0.942142f, +0.105085f,
          +0.586043f, +0.786953f, +0.193024f,   +0.813498f, +0.528989f, +0.241644f,
          +0.312968f, +0.935774f, +0.162415f,   +0.211663f, +0.971161f, +0.109750f,
          +0.161161f, +0.983395f, +0.083431f,   +0.138085f, +0.987842f, +0.071414f,
          +0.134646f, +0.988445f, +0.069616f,   +0.150959f, +0.985454f, +0.078049f,
          +0.195745f, +0.975423f, +0.101168f,   +0.297647f, +0.942225f, +0.153684f,
          +0.548300f, +0.787075f, +0.282630f,   +0.765121f, +0.529447f, +0.366437f,
          +0.283512f, +0.935896f, +0.209091f,   +0.191758f, +0.971214f, +0.141323f,
          +0.146033f, +0.983424f, +0.107477f,   +0.125142f, +0.987862f, +0.092023f,
          +0.122038f, +0.988462f, +0.089719f,   +0.136836f, +0.985473f, +0.100597f,
          +0.177452f, +0.975449f, +0.130419f,   +0.269897f, +0.942271f, +0.198197f,
          +0.497392f, +0.787123f, +0.364743f,   +0.698282f, +0.529779f, +0.481390f,
          +0.247441f, +0.935955f, +0.250520f,   +0.167385f, +0.971237f, +0.169353f,
          +0.127500f, +0.983436f, +0.128832f,   +0.109278f, +0.987869f, +0.110331f,
          +0.106577f, +0.988467f, +0.107581f,   +0.119511f, +0.985477f, +0.120632f,
          +0.155004f, +0.975453f, +0.156414f,   +0.235805f, +0.942266f, +0.237761f,
          +0.434714f, +0.787029f, +0.437732f,   +0.614827f, +0.529956f, +0.584068f,
          +0.205474f, +0.935952f, +0.285961f,   +0.139024f, +0.971231f, +0.193346f,
          +0.105930f, +0.983430f, +0.147118f,   +0.090811f, +0.987863f, +0.126013f,
          +0.088577f, +0.988460f, +0.122883f,   +0.099333f, +0.985466f, +0.137801f,
          +0.128850f, +0.975431f, +0.178695f,   +0.196059f, +0.942207f, +0.271674f,
          +0.361547f, +0.786772f, +0.500273f,   +0.516619f, +0.529874f, +0.672561f,
          +0.158357f, +0.935894f, +0.314684f,   +0.107181f, +0.971198f, +0.212807f,
          +0.081705f, +0.983409f, +0.161962f,   +0.070066f, +0.987846f, +0.138750f,
          +0.068352f, +0.988442f, +0.135317f,   +0.076659f, +0.985441f, +0.151757f,
          +0.099451f, +0.975385f, +0.196810f,   +0.151369f, +0.942090f, +0.299256f,
          +0.279217f, +0.786349f, +0.551083f,   +0.405470f, +0.529586f, +0.745072f,
          +0.107008f, +0.935798f, +0.335905f,   +0.072466f, +0.971146f, +0.227209f,
          +0.055291f, +0.983375f, +0.172962f,   +0.047441f, +0.987819f, +0.148198f,
          +0.046290f, +0.988414f, +0.144548f,   +0.051921f, +0.985403f, +0.162124f,
          +0.067377f, +0.975317f, +0.210278f,   +0.102601f, +0.941923f, +0.319771f,
          +0.189368f, +0.785768f, +0.588820f,   +0.283595f, +0.529170f, +0.799720f,
          +0.052690f, +0.935705f, +0.348827f,   +0.035731f, +0.971094f, +0.236008f,
          +0.027327f, +0.983341f, +0.179703f,   +0.023483f, +0.987791f, +0.154002f,
          +0.022925f, +0.988385f, +0.150229f,   +0.025718f, +0.985364f, +0.168514f,
          +0.033393f, +0.975244f, +0.218594f,   +0.050924f, +0.941742f, +0.332459f,
          +0.094194f, +0.785113f, +0.612147f,   +0.153967f, +0.528686f, +0.834736f,
          -0.002199f, +0.935480f, +0.353373f,   -0.001406f, +0.971012f, +0.239028f,
          -0.000952f, +0.983303f, +0.181971f,   -0.000753f, +0.987771f, +0.155908f,
          -0.000717f, +0.988374f, +0.152038f,   -0.000801f, +0.985361f, +0.170481f,
          -0.001007f, +0.975259f, +0.221063f,   -0.001397f, +0.941834f, +0.336076f,
          -0.002154f, +0.785802f, +0.618474f,   +0.021916f, +0.528369f, +0.848732f,
          -0.057200f, +0.935400f, +0.348933f,   -0.038596f, +0.970999f, +0.235950f,
          -0.029254f, +0.983306f, +0.179591f,   -0.024992f, +0.987781f, +0.153830f,
          -0.024347f, +0.988392f, +0.149962f,   -0.027289f, +0.985394f, +0.168092f,
          -0.035342f, +0.975334f, +0.217888f,   -0.053573f, +0.942060f, +0.331140f,
          -0.098142f, +0.786860f, +0.609278f,   -0.110265f, +0.528519f, +0.841730f,
          -0.111380f, +0.935597f, +0.335041f,   -0.075237f, +0.971084f, +0.226574f,
          -0.057151f, +0.983353f, +0.172483f,   -0.048892f, +0.987813f, +0.147766f,
          -0.047651f, +0.988420f, +0.144070f,   -0.053416f, +0.985424f, +0.161509f,
          -0.069219f, +0.975379f, +0.209392f,   -0.105087f, +0.942141f, +0.318319f,
          -0.193026f, +0.786957f, +0.586037f,   -0.241644f, +0.528989f, +0.813498f,
          -0.162414f, +0.935774f, +0.312968f,   -0.109750f, +0.971161f, +0.211664f,
          -0.083431f, +0.983395f, +0.161161f,   -0.071414f, +0.987842f, +0.138085f,
          -0.069616f, +0.988445f, +0.134646f,   -0.078049f, +0.985454f, +0.150959f,
          -0.101168f, +0.975423f, +0.195745f,   -0.153685f, +0.942223f, +0.297651f,
          -0.282630f, +0.787075f, +0.548300f,   -0.366432f, +0.529483f, +0.765099f,
          -0.209092f, +0.935896f, +0.283511f,   -0.141323f, +0.971214f, +0.191758f,
          -0.107477f, +0.983424f, +0.146033f,   -0.092023f, +0.987862f, +0.125142f,
          -0.089719f, +0.988462f, +0.122038f,   -0.100597f, +0.985473f, +0.136836f,
          -0.130419f, +0.975449f, +0.177452f,   -0.198197f, +0.942271f, +0.269897f,
          -0.364743f, +0.787120f, +0.497398f,   -0.481381f, +0.529796f, +0.698276f,
          -0.250520f, +0.935955f, +0.247441f,   -0.169353f, +0.971237f, +0.167384f,
          -0.128832f, +0.983436f, +0.127500f,   -0.110331f, +0.987869f, +0.109278f,
          -0.107581f, +0.988467f, +0.106577f,   -0.120632f, +0.985477f, +0.119511f,
          -0.156414f, +0.975453f, +0.155004f,   -0.237761f, +0.942266f, +0.235805f,
          -0.437732f, +0.787029f, +0.434714f,   -0.584068f, +0.529956f, +0.614827f,
          -0.285961f, +0.935952f, +0.205474f,   -0.193346f, +0.971231f, +0.139024f,
          -0.147118f, +0.983430f, +0.105929f,   -0.126012f, +0.987863f, +0.090812f,
          -0.122883f, +0.988460f, +0.088578f,   -0.137801f, +0.985466f, +0.099333f,
          -0.178695f, +0.975431f, +0.128852f,   -0.271674f, +0.942207f, +0.196059f,
          -0.500273f, +0.786772f, +0.361547f,   -0.672561f, +0.529874f, +0.516619f,
          -0.314684f, +0.935894f, +0.158357f,   -0.212807f, +0.971198f, +0.107182f,
          -0.161962f, +0.983409f, +0.081706f,   -0.138749f, +0.987846f, +0.070065f,
          -0.135317f, +0.988442f, +0.068352f,   -0.151756f, +0.985441f, +0.076659f,
          -0.196811f, +0.975384f, +0.099453f,   -0.299259f, +0.942088f, +0.151372f,
          -0.551083f, +0.786349f, +0.279217f,   -0.745072f, +0.529586f, +0.405470f,
          -0.335905f, +0.935798f, +0.107008f,   -0.227209f, +0.971146f, +0.072466f,
          -0.172962f, +0.983375f, +0.055291f,   -0.148198f, +0.987819f, +0.047441f,
          -0.144548f, +0.988415f, +0.046288f,   -0.162124f, +0.985404f, +0.051920f,
          -0.210278f, +0.975317f, +0.067375f,   -0.319777f, +0.941921f, +0.102603f,
          -0.588820f, +0.785768f, +0.189368f,   -0.799720f, +0.529170f, +0.283595f,
          -0.348827f, +0.935705f, +0.052690f,   -0.236008f, +0.971094f, +0.035731f,
          -0.179703f, +0.983341f, +0.027327f,   -0.154002f, +0.987791f, +0.023483f,
          -0.150229f, +0.988385f, +0.022926f,   -0.168514f, +0.985364f, +0.025718f,
          -0.218593f, +0.975245f, +0.033391f,   -0.332461f, +0.941741f, +0.050923f,
          -0.612147f, +0.785113f, +0.094194f,   -0.834736f, +0.528686f, +0.153967f,
          -0.353373f, +0.935480f, -0.002199f,   -0.239028f, +0.971012f, -0.001406f,
          -0.181971f, +0.983303f, -0.000952f,   -0.155908f, +0.987771f, -0.000753f,
          -0.152038f, +0.988375f, -0.000717f,   -0.170480f, +0.985361f, -0.000801f,
          -0.221063f, +0.975259f, -0.001007f,   -0.336076f, +0.941834f, -0.001397f,
          -0.618474f, +0.785802f, -0.002154f,   -0.848732f, +0.528369f, +0.021916f,
          -0.348933f, +0.935400f, -0.057200f,   -0.235950f, +0.970999f, -0.038596f,
          -0.179591f, +0.983306f, -0.029254f,   -0.153830f, +0.987781f, -0.024992f,
          -0.149962f, +0.988392f, -0.024347f,   -0.168092f, +0.985394f, -0.027289f,
          -0.217888f, +0.975334f, -0.035342f,   -0.331140f, +0.942060f, -0.053573f,
          -0.609278f, +0.786860f, -0.098142f,   -0.841730f, +0.528519f, -0.110265f,
          -0.335041f, +0.935597f, -0.111380f,   -0.226574f, +0.971084f, -0.075237f,
          -0.172483f, +0.983353f, -0.057151f,   -0.147766f, +0.987813f, -0.048892f,
          -0.144070f, +0.988420f, -0.047651f,   -0.161509f, +0.985424f, -0.053416f,
          -0.209392f, +0.975379f, -0.069219f,   -0.318317f, +0.942142f, -0.105085f,
          -0.586043f, +0.786953f, -0.193024f,   -0.813498f, +0.528989f, -0.241644f,
          -0.312967f, +0.935774f, -0.162415f,   -0.211664f, +0.971161f, -0.109750f,
          -0.161161f, +0.983395f, -0.083431f,   -0.138084f, +0.987842f, -0.071414f,
          -0.134646f, +0.988445f, -0.069616f,   -0.150959f, +0.985454f, -0.078049f,
          -0.195745f, +0.975423f, -0.101168f,   -0.297647f, +0.942225f, -0.153684f,
          -0.548300f, +0.787075f, -0.282630f,   -0.765121f, +0.529447f, -0.366437f,
          -0.283511f, +0.935896f, -0.209092f,   -0.191757f, +0.971214f, -0.141324f,
          -0.146033f, +0.983424f, -0.107477f,   -0.125142f, +0.987862f, -0.092022f,
          -0.122039f, +0.988462f, -0.089719f,   -0.136835f, +0.985473f, -0.100596f,
          -0.177454f, +0.975449f, -0.130419f,   -0.269896f, +0.942271f, -0.198196f,
          -0.497397f, +0.787121f, -0.364743f,   -0.698282f, +0.529779f, -0.481390f,
          -0.247441f, +0.935955f, -0.250520f,   -0.167385f, +0.971237f, -0.169353f,
          -0.127500f, +0.983436f, -0.128832f,   -0.109277f, +0.987869f, -0.110331f,
          -0.106577f, +0.988467f, -0.107581f,   -0.119511f, +0.985477f, -0.120632f,
          -0.155004f, +0.975453f, -0.156414f,   -0.235804f, +0.942267f, -0.237761f,
          -0.434714f, +0.787029f, -0.437732f,   -0.614842f, +0.529927f, -0.584077f,
          -0.205476f, +0.935952f, -0.285960f,   -0.139025f, +0.971231f, -0.193343f,
          -0.105932f, +0.983430f, -0.147118f,   -0.090812f, +0.987863f, -0.126012f,
          -0.088577f, +0.988460f, -0.122883f,   -0.099333f, +0.985466f, -0.137801f,
          -0.128850f, +0.975431f, -0.178695f,   -0.196060f, +0.942206f, -0.271676f,
          -0.361542f, +0.786775f, -0.500272f,   -0.516621f, +0.529859f, -0.672571f,
          -0.158357f, +0.935894f, -0.314683f,   -0.107179f, +0.971198f, -0.212806f,
          -0.081706f, +0.983409f, -0.161962f,   -0.070066f, +0.987846f, -0.138750f,
          -0.068352f, +0.988442f, -0.135317f,   -0.076659f, +0.985441f, -0.151756f,
          -0.099453f, +0.975384f, -0.196811f,   -0.151372f, +0.942088f, -0.299259f,
          -0.279217f, +0.786349f, -0.551083f,   -0.405470f, +0.529586f, -0.745072f,
          -0.107007f, +0.935798f, -0.335904f,   -0.072466f, +0.971146f, -0.227209f,
          -0.055288f, +0.983375f, -0.172963f,   -0.047441f, +0.987819f, -0.148198f,
          -0.046290f, +0.988414f, -0.144548f,   -0.051920f, +0.985404f, -0.162124f,
          -0.067375f, +0.975317f, -0.210278f,   -0.102603f, +0.941921f, -0.319777f,
          -0.189368f, +0.785768f, -0.588820f,   -0.283595f, +0.529170f, -0.799720f,
          -0.052689f, +0.935705f, -0.348826f,   -0.035730f, +0.971094f, -0.236008f,
          -0.027327f, +0.983341f, -0.179703f,   -0.023482f, +0.987792f, -0.154001f,
          -0.022925f, +0.988385f, -0.150229f,   -0.025719f, +0.985364f, -0.168515f,
          -0.033391f, +0.975245f, -0.218593f,   -0.050923f, +0.941741f, -0.332461f,
          -0.094194f, +0.785113f, -0.612147f,   -0.153967f, +0.528686f, -0.834736f,
          +0.002199f, +0.935480f, -0.353373f,   +0.001407f, +0.971012f, -0.239028f,
          +0.000952f, +0.983303f, -0.181972f,   +0.000752f, +0.987772f, -0.155907f,
          +0.000717f, +0.988374f, -0.152038f,   +0.000802f, +0.985361f, -0.170481f,
          +0.001007f, +0.975259f, -0.221063f,   +0.001397f, +0.941834f, -0.336076f,
          +0.002154f, +0.785802f, -0.618474f,   -0.021916f, +0.528369f, -0.848732f,
          +0.057199f, +0.935400f, -0.348933f,   +0.038595f, +0.970998f, -0.235950f,
          +0.029254f, +0.983306f, -0.179591f,   +0.024993f, +0.987781f, -0.153829f,
          +0.024347f, +0.988392f, -0.149962f,   +0.027289f, +0.985393f, -0.168094f,
          +0.035342f, +0.975334f, -0.217888f,   +0.053573f, +0.942060f, -0.331140f,
          +0.098142f, +0.786860f, -0.609278f,   +0.110265f, +0.528519f, -0.841730f,
          +0.111381f, +0.935597f, -0.335040f,   +0.075237f, +0.971084f, -0.226574f,
          +0.057151f, +0.983353f, -0.172484f,   +0.048891f, +0.987813f, -0.147766f,
          +0.047651f, +0.988420f, -0.144070f,   +0.053417f, +0.985424f, -0.161510f,
          +0.069219f, +0.975379f, -0.209392f,   +0.105085f, +0.942142f, -0.318317f,
          +0.193024f, +0.786953f, -0.586043f,   +0.241644f, +0.528989f, -0.813498f,
          +0.162415f, +0.935774f, -0.312967f,   +0.109751f, +0.971161f, -0.211663f,
          +0.083431f, +0.983395f, -0.161161f,   +0.071414f, +0.987843f, -0.138083f,
          +0.069616f, +0.988445f, -0.134646f,   +0.078049f, +0.985454f, -0.150960f,
          +0.101168f, +0.975423f, -0.195745f,   +0.153684f, +0.942225f, -0.297647f,
          +0.282630f, +0.787075f, -0.548300f,   +0.366437f, +0.529447f, -0.765121f,
          +0.209092f, +0.935896f, -0.283511f,   +0.141324f, +0.971214f, -0.191757f,
          +0.107477f, +0.983424f, -0.146033f,   +0.092022f, +0.987862f, -0.125142f,
          +0.089720f, +0.988462f, -0.122038f,   +0.100597f, +0.985473f, -0.136836f,
          +0.130418f, +0.975449f, -0.177454f,   +0.198195f, +0.942271f, -0.269896f,
          +0.364743f, +0.787120f, -0.497398f,   +0.481390f, +0.529779f, -0.698282f,
          +0.250520f, +0.935955f, -0.247441f,   +0.169353f, +0.971237f, -0.167385f,
          +0.128832f, +0.983436f, -0.127500f,   +0.110331f, +0.987869f, -0.109277f,
          +0.107581f, +0.988467f, -0.106577f,   +0.120632f, +0.985477f, -0.119511f,
          +0.156414f, +0.975453f, -0.155004f,   +0.237761f, +0.942266f, -0.235805f,
          +0.437732f, +0.787029f, -0.434714f,   +0.584077f, +0.529927f, -0.614842f,
          +0.285960f, +0.935952f, -0.205475f,   +0.193346f, +0.971231f, -0.139024f,
          +0.147118f, +0.983430f, -0.105930f,   +0.126012f, +0.987863f, -0.090812f,
          +0.122883f, +0.988460f, -0.088577f,   +0.137801f, +0.985466f, -0.099333f,
          +0.178695f, +0.975431f, -0.128850f,   +0.271676f, +0.942206f, -0.196059f,
          +0.500272f, +0.786775f, -0.361543f,   +0.672570f, +0.529861f, -0.516621f,
          +0.314683f, +0.935894f, -0.158357f,   +0.212807f, +0.971198f, -0.107181f,
          +0.161962f, +0.983409f, -0.081706f,   +0.138749f, +0.987846f, -0.070065f,
          +0.135317f, +0.988442f, -0.068352f,   +0.151756f, +0.985441f, -0.076659f,
          +0.196811f, +0.975384f, -0.099453f,   +0.299259f, +0.942088f, -0.151372f,
          +0.551083f, +0.786349f, -0.279217f,   +0.745072f, +0.529586f, -0.405470f,
          +0.335905f, +0.935798f, -0.107008f,   +0.227209f, +0.971146f, -0.072466f,
          +0.172962f, +0.983375f, -0.055291f,   +0.148198f, +0.987819f, -0.047441f,
          +0.144548f, +0.988414f, -0.046289f,   +0.162124f, +0.985404f, -0.051920f,
          +0.210278f, +0.975317f, -0.067375f,   +0.319777f, +0.941921f, -0.102603f,
          +0.588820f, +0.785768f, -0.189368f,   +0.799720f, +0.529170f, -0.283596f,
          +0.348827f, +0.935705f, -0.052690f,   +0.236008f, +0.971094f, -0.035731f,
          +0.179703f, +0.983341f, -0.027327f,   +0.154002f, +0.987791f, -0.023483f,
          +0.150229f, +0.988385f, -0.022926f,   +0.168514f, +0.985364f, -0.025718f,
          +0.218593f, +0.975245f, -0.033391f,   +0.332461f, +0.941741f, -0.050923f,
          +0.612147f, +0.785113f, -0.094194f,   +0.834736f, +0.528686f, -0.153967f,
    };
    static const float teapot_texcoords[] = {
          0.684148f, 0.500000f,   0.681346f, 0.442629f,   0.679187f, 0.443206f,
          0.681961f, 0.500000f,   0.678340f, 0.443432f,   0.681102f, 0.500000f,
          0.678620f, 0.443358f,   0.681386f, 0.500000f,   0.679843f, 0.443031f,
          0.682625f, 0.500000f,   0.681825f, 0.442501f,   0.684633f, 0.500000f,
          0.684383f, 0.441818f,   0.687224f, 0.500000f,   0.687331f, 0.441030f,
          0.690211f, 0.500000f,   0.690487f, 0.440187f,   0.693408f, 0.500000f,
          0.693666f, 0.439338f,   0.696628f, 0.500000f,   0.696683f, 0.438531f,
          0.699685f, 0.500000f,   0.673219f, 0.388447f,   0.671142f, 0.389569f,
          0.670327f, 0.390009f,   0.670596f, 0.389864f,   0.671773f, 0.389228f,
          0.673680f, 0.388198f,   0.676141f, 0.386870f,   0.678978f, 0.385338f,
          0.682014f, 0.383699f,   0.685073f, 0.382047f,   0.687976f, 0.380479f,
          0.660185f, 0.338126f,   0.658239f, 0.339753f,   0.657475f, 0.340392f,
          0.657728f, 0.340181f,   0.658830f, 0.339259f,   0.660617f, 0.337765f,
          0.662923f, 0.335836f,   0.665581f, 0.333614f,   0.668425f, 0.331235f,
          0.671291f, 0.328838f,   0.674011f, 0.326563f,   0.642661f, 0.292336f,
          0.640892f, 0.294423f,   0.640197f, 0.295243f,   0.640426f, 0.294972f,
          0.641429f, 0.293789f,   0.643054f, 0.291872f,   0.645151f, 0.289399f,
          0.647568f, 0.286547f,   0.650156f, 0.283495f,   0.652762f, 0.280421f,
          0.655236f, 0.277502f,   0.621066f, 0.251748f,   0.619513f, 0.254244f,
          0.618903f, 0.255224f,   0.619105f, 0.254900f,   0.619984f, 0.253486f,
          0.621410f, 0.251194f,   0.623250f, 0.248237f,   0.625371f, 0.244828f,
          0.627640f, 0.241180f,   0.629927f, 0.237505f,   0.632097f, 0.234016f,
          0.595815f, 0.217035f,   0.594516f, 0.219880f,   0.594007f, 0.220996f,
          0.594175f, 0.220628f,   0.594911f, 0.219016f,   0.596104f, 0.216403f,
          0.597642f, 0.213033f,   0.599416f, 0.209147f,   0.601315f, 0.204989f,
          0.603228f, 0.200800f,   0.605043f, 0.196823f,   0.567328f, 0.188867f,
          0.566316f, 0.191995f,   0.565918f, 0.193223f,   0.566050f, 0.192817f,
          0.566623f, 0.191045f,   0.567553f, 0.188173f,   0.568753f, 0.184467f,
          0.570135f, 0.180194f,   0.571615f, 0.175622f,   0.573106f, 0.171016f,
          0.574521f, 0.166643f,   0.536022f, 0.167916f,   0.535324f, 0.171255f,
          0.535051f, 0.172565f,   0.535141f, 0.172132f,   0.535536f, 0.170241f,
          0.536177f, 0.167175f,   0.537004f, 0.163219f,   0.537957f, 0.158659f,
          0.538977f, 0.153779f,   0.540004f, 0.148863f,   0.540979f, 0.144196f,
          0.502315f, 0.154853f,   0.501956f, 0.158323f,   0.501815f, 0.159685f,
          0.501861f, 0.159235f,   0.502065f, 0.157269f,   0.502394f, 0.154083f,
          0.502819f, 0.149972f,   0.503309f, 0.145232f,   0.503834f, 0.140160f,
          0.504362f, 0.135050f,   0.504864f, 0.130200f,   0.466623f, 0.150350f,
          0.466623f, 0.153865f,   0.466623f, 0.155245f,   0.466623f, 0.154789f,
          0.466623f, 0.152797f,   0.466623f, 0.149569f,   0.466623f, 0.145405f,
          0.466623f, 0.140603f,   0.466623f, 0.135465f,   0.466623f, 0.130288f,
          0.466623f, 0.125375f,   0.427534f, 0.154853f,   0.428813f, 0.158323f,
          0.429692f, 0.159685f,   0.430219f, 0.159235f,   0.430448f, 0.157269f,
          0.430427f, 0.154083f,   0.430210f, 0.149972f,   0.429845f, 0.145232f,
          0.429385f, 0.140160f,   0.428881f, 0.135050f,   0.428382f, 0.130200f,
          0.391854f, 0.167916f,   0.394007f, 0.171255f,   0.395446f, 0.172565f,
          0.396264f, 0.172132f,   0.396550f, 0.170241f,   0.396398f, 0.167175f,
          0.395899f, 0.163220f,   0.395145f, 0.158659f,   0.394227f, 0.153779f,
          0.393237f, 0.148863f,   0.392267f, 0.144196f,   0.359751f, 0.188867f,
          0.362435f, 0.191995f,   0.364171f, 0.193223f,   0.365081f, 0.192817f,
          0.365291f, 0.191045f,   0.364922f, 0.188173f,   0.364099f, 0.184467f,
          0.362944f, 0.180194f,   0.361582f, 0.175622f,   0.360134f, 0.171016f,
          0.358725f, 0.166643f,   0.331390f, 0.217035f,   0.334326f, 0.219880f,
          0.336147f, 0.220996f,   0.336999f, 0.220628f,   0.337031f, 0.219016f,
          0.336388f, 0.216403f,   0.335217f, 0.213033f,   0.333667f, 0.209147f,
          0.331883f, 0.204989f,   0.330013f, 0.200800f,   0.328203f, 0.196823f,
          0.306937f, 0.251748f,   0.309911f, 0.254244f,   0.311658f, 0.255224f,
          0.312343f, 0.254900f,   0.312129f, 0.253486f,   0.311181f, 0.251194f,
          0.309661f, 0.248237f,   0.307734f, 0.244828f,   0.305564f, 0.241180f,
          0.303315f, 0.237505f,   0.301149f, 0.234016f,   0.286558f, 0.292336f,
          0.289419f, 0.294424f,   0.290988f, 0.295243f,   0.291439f, 0.294972f,
          0.290947f, 0.293789f,   0.289689f, 0.291872f,   0.287838f, 0.289399f,
          0.285569f, 0.286547f,   0.283059f, 0.283495f,   0.280481f, 0.280421f,
          0.278011f, 0.277503f,   0.270418f, 0.338126f,   0.273081f, 0.339753f,
          0.274418f, 0.340392f,   0.274612f, 0.340181f,   0.273845f, 0.339259f,
          0.272299f, 0.337765f,   0.270155f, 0.335837f,   0.267594f, 0.333614f,
          0.264800f, 0.331235f,   0.261953f, 0.328838f,   0.259235f, 0.326564f,
          0.258685f, 0.388448f,   0.261126f, 0.389569f,   0.262232f, 0.390009f,
          0.262190f, 0.389864f,   0.261183f, 0.389229f,   0.259398f, 0.388199f,
          0.257019f, 0.386870f,   0.254232f, 0.385338f,   0.251221f, 0.383699f,
          0.248172f, 0.382047f,   0.245270f, 0.380480f,   0.251523f, 0.442630f,
          0.253784f, 0.443206f,   0.254713f, 0.443433f,   0.254497f, 0.443358f,
          0.253322f, 0.443031f,   0.251374f, 0.442501f,   0.248840f, 0.441818f,
          0.245905f, 0.441030f,   0.242756f, 0.440187f,   0.239580f, 0.439338f,
          0.236563f, 0.438532f,   0.249099f, 0.500000f,   0.251286f, 0.500000f,
          0.252144f, 0.500000f,   0.251861f, 0.500000f,   0.250621f, 0.500000f,
          0.248613f, 0.500000f,   0.246022f, 0.500000f,   0.243035f, 0.500000f,
          0.239839f, 0.500000f,   0.236618f, 0.500000f,   0.233561f, 0.500000f,
          0.251901f, 0.557371f,   0.254059f, 0.556794f,   0.254907f, 0.556568f,
          0.254627f, 0.556642f,   0.253404f, 0.556969f,   0.251421f, 0.557499f,
          0.248864f, 0.558182f,   0.245915f, 0.558970f,   0.242760f, 0.559813f,
          0.239581f, 0.560662f,   0.236563f, 0.561469f,   0.260027f, 0.611553f,
          0.262104f, 0.610431f,   0.262920f, 0.609991f,   0.262650f, 0.610136f,
          0.261473f, 0.610772f,   0.259566f, 0.611802f,   0.257105f, 0.613130f,
          0.254268f, 0.614662f,   0.251232f, 0.616301f,   0.248174f, 0.617953f,
          0.245270f, 0.619521f,   0.273061f, 0.661874f,   0.275007f, 0.660247f,
          0.275771f, 0.659608f,   0.275519f, 0.659819f,   0.274416f, 0.660741f,
          0.272629f, 0.662235f,   0.270324f, 0.664164f,   0.267666f, 0.666386f,
          0.264821f, 0.668765f,   0.261956f, 0.671162f,   0.259236f, 0.673437f,
          0.290585f, 0.707664f,   0.292355f, 0.705577f,   0.293050f, 0.704757f,
          0.292820f, 0.705028f,   0.291817f, 0.706211f,   0.290192f, 0.708128f,
          0.288095f, 0.710601f,   0.285678f, 0.713453f,   0.283091f, 0.716505f,
          0.280485f, 0.719579f,   0.278011f, 0.722498f,   0.312181f, 0.748252f,
          0.313734f, 0.745756f,   0.314343f, 0.744776f,   0.314142f, 0.745100f,
          0.313262f, 0.746514f,   0.311836f, 0.748806f,   0.309997f, 0.751763f,
          0.307876f, 0.755172f,   0.305606f, 0.758820f,   0.303320f, 0.762495f,
          0.301149f, 0.765984f,   0.337431f, 0.782965f,   0.338730f, 0.780120f,
          0.339240f, 0.779004f,   0.339071f, 0.779372f,   0.338336f, 0.780984f,
          0.337143f, 0.783597f,   0.335604f, 0.786967f,   0.333830f, 0.790853f,
          0.331931f, 0.795011f,   0.330019f, 0.799200f,   0.328203f, 0.803177f,
          0.365918f, 0.811133f,   0.366931f, 0.808005f,   0.367328f, 0.806777f,
          0.367197f, 0.807183f,   0.366623f, 0.808955f,   0.365693f, 0.811827f,
          0.364494f, 0.815533f,   0.363111f, 0.819806f,   0.361631f, 0.824378f,
          0.360140f, 0.828984f,   0.358725f, 0.833357f,   0.397224f, 0.832084f,
          0.397922f, 0.828745f,   0.398196f, 0.827435f,   0.398105f, 0.827868f,
          0.397710f, 0.829759f,   0.397069f, 0.832825f,   0.396243f, 0.836781f,
          0.395290f, 0.841341f,   0.394270f, 0.846221f,   0.393243f, 0.851137f,
          0.392267f, 0.855804f,   0.430932f, 0.845147f,   0.431291f, 0.841677f,
          0.431432f, 0.840315f,   0.431385f, 0.840765f,   0.431182f, 0.842731f,
          0.430852f, 0.845917f,   0.430427f, 0.850028f,   0.429937f, 0.854768f,
          0.429412f, 0.859840f,   0.428884f, 0.864950f,   0.428383f, 0.869800f,
          0.466623f, 0.849650f,   0.466623f, 0.846135f,   0.466623f, 0.844755f,
          0.466623f, 0.845211f,   0.466623f, 0.847203f,   0.466623f, 0.850431f,
          0.466623f, 0.854595f,   0.466623f, 0.859397f,   0.466623f, 0.864535f,
          0.466623f, 0.869712f,   0.466623f, 0.874625f,   0.502315f, 0.845147f,
          0.501956f, 0.841677f,   0.501815f, 0.840315f,   0.501862f, 0.840765f,
          0.502065f, 0.842731f,   0.502394f, 0.845917f,   0.502819f, 0.850028f,
          0.503310f, 0.854768f,   0.503834f, 0.859840f,   0.504363f, 0.864950f,
          0.504864f, 0.869800f,   0.536022f, 0.832084f,   0.535325f, 0.828745f,
          0.535051f, 0.827435f,   0.535141f, 0.827868f,   0.535536f, 0.829759f,
          0.536177f, 0.832825f,   0.537004f, 0.836780f,   0.537957f, 0.841341f,
          0.538977f, 0.846221f,   0.540004f, 0.851137f,   0.540979f, 0.855804f,
          0.567328f, 0.811133f,   0.566316f, 0.808005f,   0.565919f, 0.806777f,
          0.566050f, 0.807183f,   0.566623f, 0.808955f,   0.567553f, 0.811827f,
          0.568753f, 0.815533f,   0.570136f, 0.819806f,   0.571616f, 0.824378f,
          0.573106f, 0.828984f,   0.574522f, 0.833357f,   0.595815f, 0.782965f,
          0.594517f, 0.780120f,   0.594007f, 0.779004f,   0.594175f, 0.779372f,
          0.594911f, 0.780984f,   0.596104f, 0.783597f,   0.597643f, 0.786967f,
          0.599417f, 0.790853f,   0.601315f, 0.795011f,   0.603228f, 0.799200f,
          0.605043f, 0.803177f,   0.621066f, 0.748252f,   0.619513f, 0.745756f,
          0.618903f, 0.744776f,   0.619105f, 0.745100f,   0.619985f, 0.746514f,
          0.621410f, 0.748806f,   0.623250f, 0.751763f,   0.625371f, 0.755172f,
          0.627640f, 0.758820f,   0.629927f, 0.762495f,   0.632097f, 0.765984f,
          0.642661f, 0.707664f,   0.640892f, 0.705576f,   0.640197f, 0.704757f,
          0.640426f, 0.705028f,   0.641429f, 0.706211f,   0.643054f, 0.708128f,
          0.645151f, 0.710601f,   0.647568f, 0.713453f,   0.650156f, 0.716505f,
          0.652762f, 0.719579f,   0.655236f, 0.722497f,   0.660185f, 0.661874f,
          0.658239f, 0.660247f,   0.657475f, 0.659608f,   0.657728f, 0.659819f,
          0.658830f, 0.660741f,   0.660617f, 0.662235f,   0.662923f, 0.664163f,
          0.665581f, 0.666386f,   0.668425f, 0.668765f,   0.671291f, 0.671162f,
          0.674011f, 0.673436f,   0.673219f, 0.611552f,   0.671142f, 0.610431f,
          0.670327f, 0.609991f,   0.670596f, 0.610136f,   0.671773f, 0.610771f,
          0.673680f, 0.611801f,   0.676141f, 0.613130f,   0.678978f, 0.614662f,
          0.682014f, 0.616301f,   0.685073f, 0.617953f,   0.687976f, 0.619520f,
          0.681346f, 0.557370f,   0.679187f, 0.556794f,   0.678340f, 0.556567f,
          0.678620f, 0.556642f,   0.679843f, 0.556969f,   0.681825f, 0.557499f,
          0.684383f, 0.558182f,   0.687331f, 0.558970f,   0.690487f, 0.559813f,
          0.693666f, 0.560662f,   0.696683f, 0.561468f,   0.708148f, 0.435468f,
          0.711299f, 0.500000f,   0.719383f, 0.432467f,   0.722681f, 0.500000f,
          0.730157f, 0.429588f,   0.733596f, 0.500000f,   0.740241f, 0.426893f,
          0.743811f, 0.500000f,   0.749405f, 0.424445f,   0.753095f, 0.500000f,
          0.757419f, 0.422304f,   0.761213f, 0.500000f,   0.764053f, 0.420531f,
          0.767933f, 0.500000f,   0.769075f, 0.419189f,   0.773022f, 0.500000f,
          0.772258f, 0.418339f,   0.776246f, 0.500000f,   0.773370f, 0.418042f,
          0.777372f, 0.500000f,   0.699007f, 0.374523f,   0.709816f, 0.368687f,
          0.720183f, 0.363089f,   0.729886f, 0.357850f,   0.738703f, 0.353089f,
          0.746413f, 0.348926f,   0.752796f, 0.345480f,   0.757628f, 0.342870f,
          0.760690f, 0.341217f,   0.761760f, 0.340639f,   0.684346f, 0.317920f,
          0.694473f, 0.309451f,   0.704186f, 0.301328f,   0.713276f, 0.293726f,
          0.721537f, 0.286817f,   0.728761f, 0.280776f,   0.734741f, 0.275775f,
          0.739269f, 0.271989f,   0.742138f, 0.269589f,   0.743140f, 0.268751f,
          0.664635f, 0.266415f,   0.673845f, 0.255549f,   0.682679f, 0.245129f,
          0.690946f, 0.235376f,   0.698459f, 0.226513f,   0.705029f, 0.218763f,
          0.710467f, 0.212348f,   0.714586f, 0.207490f,   0.717195f, 0.204412f,
          0.718106f, 0.203337f,   0.640343f, 0.220761f,   0.648424f, 0.207772f,
          0.656174f, 0.195315f,   0.663427f, 0.183656f,   0.670018f, 0.173061f,
          0.675782f, 0.163796f,   0.680553f, 0.156127f,   0.684166f, 0.150320f,
          0.686455f, 0.146640f,   0.687255f, 0.145355f,   0.611941f, 0.181715f,
          0.618701f, 0.166910f,   0.625183f, 0.152711f,   0.631251f, 0.139422f,
          0.636765f, 0.127345f,   0.641586f, 0.116784f,   0.645577f, 0.108043f,
          0.648600f, 0.101423f,   0.650514f, 0.097230f,   0.651183f, 0.095764f,
          0.579898f, 0.150031f,   0.585167f, 0.133752f,   0.590221f, 0.118140f,
          0.594950f, 0.103528f,   0.599248f, 0.090249f,   0.603007f, 0.078637f,
          0.606118f, 0.069025f,   0.608473f, 0.061747f,   0.609966f, 0.057136f,
          0.610488f, 0.055524f,   0.544685f, 0.126465f,   0.548316f, 0.109090f,
          0.551798f, 0.092426f,   0.555057f, 0.076830f,   0.558019f, 0.062657f,
          0.560609f, 0.050264f,   0.562753f, 0.040004f,   0.564377f, 0.032236f,
          0.565405f, 0.027314f,   0.565765f, 0.025594f,   0.506770f, 0.111771f,
          0.508637f, 0.093713f,   0.510428f, 0.076394f,   0.512104f, 0.060184f,
          0.513627f, 0.045454f,   0.514960f, 0.032573f,   0.516062f, 0.021910f,
          0.516897f, 0.013836f,   0.517426f, 0.008720f,   0.517611f, 0.006933f,
          0.466623f, 0.106706f,   0.466623f, 0.088412f,   0.466623f, 0.070867f,
          0.466623f, 0.054446f,   0.466623f, 0.039523f,   0.466623f, 0.026473f,
          0.466623f, 0.015672f,   0.466623f, 0.007493f,   0.466623f, 0.002310f,
          0.466623f, 0.000499f,   0.426477f, 0.111771f,   0.424609f, 0.093713f,
          0.422818f, 0.076394f,   0.421142f, 0.060184f,   0.419619f, 0.045454f,
          0.418287f, 0.032573f,   0.417184f, 0.021910f,   0.416349f, 0.013836f,
          0.415820f, 0.008721f,   0.415635f, 0.006933f,   0.388562f, 0.126465f,
          0.384931f, 0.109090f,   0.381448f, 0.092426f,   0.378189f, 0.076830f,
          0.375227f, 0.062657f,   0.372637f, 0.050264f,   0.370493f, 0.040004f,
          0.368870f, 0.032236f,   0.367841f, 0.027314f,   0.367482f, 0.025595f,
          0.353348f, 0.150031f,   0.348079f, 0.133752f,   0.343026f, 0.118140f,
          0.338296f, 0.103528f,   0.333998f, 0.090249f,   0.330240f, 0.078637f,
          0.327129f, 0.069026f,   0.324773f, 0.061747f,   0.323280f, 0.057136f,
          0.322759f, 0.055524f,   0.321305f, 0.181715f,   0.314546f, 0.166910f,
          0.308063f, 0.152711f,   0.301996f, 0.139422f,   0.296482f, 0.127345f,
          0.291660f, 0.116785f,   0.287669f, 0.108043f,   0.284647f, 0.101424f,
          0.282732f, 0.097230f,   0.282063f, 0.095764f,   0.292903f, 0.220761f,
          0.284823f, 0.207772f,   0.277073f, 0.195315f,   0.269820f, 0.183656f,
          0.263228f, 0.173061f,   0.257464f, 0.163796f,   0.252693f, 0.156127f,
          0.249080f, 0.150320f,   0.246791f, 0.146640f,   0.245991f, 0.145355f,
          0.268612f, 0.266415f,   0.259401f, 0.255550f,   0.250568f, 0.245129f,
          0.242300f, 0.235376f,   0.234787f, 0.226514f,   0.228217f, 0.218763f,
          0.222779f, 0.212348f,   0.218661f, 0.207490f,   0.216052f, 0.204412f,
          0.215140f, 0.203337f,   0.248901f, 0.317921f,   0.238773f, 0.309451f,
          0.229061f, 0.301329f,   0.219970f, 0.293726f,   0.211709f, 0.286818f,
          0.204485f, 0.280776f,   0.198505f, 0.275776f,   0.193977f, 0.271989f,
          0.191109f, 0.269590f,   0.190106f, 0.268751f,   0.234240f, 0.374524f,
          0.223430f, 0.368687f,   0.213064f, 0.363089f,   0.203361f, 0.357850f,
          0.194544f, 0.353090f,   0.186833f, 0.348926f,   0.180451f, 0.345480f,
          0.175618f, 0.342871f,   0.172556f, 0.341217f,   0.171486f, 0.340640f,
          0.225099f, 0.435468f,   0.213864f, 0.432467f,   0.203089f, 0.429588f,
          0.193005f, 0.426894f,   0.183841f, 0.424445f,   0.175827f, 0.422304f,
          0.169194f, 0.420532f,   0.164171f, 0.419190f,   0.160988f, 0.418339f,
          0.159877f, 0.418042f,   0.221947f, 0.500000f,   0.210566f, 0.500000f,
          0.199651f, 0.500000f,   0.189435f, 0.500000f,   0.180151f, 0.500000f,
          0.172033f, 0.500000f,   0.165313f, 0.500000f,   0.160225f, 0.500000f,
          0.157001f, 0.500000f,   0.155874f, 0.500000f,   0.225099f, 0.564532f,
          0.213864f, 0.567533f,   0.203090f, 0.570412f,   0.193005f, 0.573107f,
          0.183841f, 0.575555f,   0.175827f, 0.577696f,   0.169194f, 0.579469f,
          0.164171f, 0.580811f,   0.160989f, 0.581661f,   0.159877f, 0.581958f,
          0.234240f, 0.625477f,   0.223430f, 0.631313f,   0.213064f, 0.636911f,
          0.203361f, 0.642150f,   0.194544f, 0.646911f,   0.186833f, 0.651074f,
          0.180451f, 0.654520f,   0.175618f, 0.657130f,   0.172556f, 0.658783f,
          0.171486f, 0.659361f,   0.248901f, 0.682080f,   0.238773f, 0.690549f,
          0.229061f, 0.698672f,   0.219970f, 0.706274f,   0.211709f, 0.713183f,
          0.204485f, 0.719224f,   0.198505f, 0.724225f,   0.193977f, 0.728011f,
          0.191109f, 0.730411f,   0.190106f, 0.731249f,   0.268612f, 0.733585f,
          0.259401f, 0.744451f,   0.250568f, 0.754871f,   0.242300f, 0.764624f,
          0.234787f, 0.773487f,   0.228217f, 0.781237f,   0.222779f, 0.787652f,
          0.218661f, 0.792510f,   0.216052f, 0.795588f,   0.215140f, 0.796663f,
          0.292903f, 0.779239f,   0.284823f, 0.792228f,   0.277073f, 0.804685f,
          0.269820f, 0.816344f,   0.263228f, 0.826939f,   0.257464f, 0.836204f,
          0.252693f, 0.843873f,   0.249080f, 0.849680f,   0.246791f, 0.853360f,
          0.245991f, 0.854645f,   0.321305f, 0.818285f,   0.314546f, 0.833090f,
          0.308063f, 0.847289f,   0.301996f, 0.860578f,   0.296482f, 0.872655f,
          0.291660f, 0.883216f,   0.287669f, 0.891957f,   0.284647f, 0.898577f,
          0.282732f, 0.902771f,   0.282063f, 0.904236f,   0.353348f, 0.849969f,
          0.348079f, 0.866248f,   0.343026f, 0.881860f,   0.338296f, 0.896472f,
          0.333998f, 0.909751f,   0.330240f, 0.921363f,   0.327129f, 0.930975f,
          0.324773f, 0.938253f,   0.323280f, 0.942864f,   0.322759f, 0.944476f,
          0.388562f, 0.873535f,   0.384931f, 0.890910f,   0.381448f, 0.907574f,
          0.378189f, 0.923170f,   0.375227f, 0.937343f,   0.372637f, 0.949737f,
          0.370493f, 0.959996f,   0.368870f, 0.967764f,   0.367841f, 0.972686f,
          0.367482f, 0.974406f,   0.426477f, 0.888229f,   0.424609f, 0.906287f,
          0.422819f, 0.923606f,   0.421142f, 0.939816f,   0.419619f, 0.954546f,
          0.418287f, 0.967427f,   0.417184f, 0.978090f,   0.416349f, 0.986164f,
          0.415820f, 0.991280f,   0.415636f, 0.993067f,   0.466623f, 0.893294f,
          0.466623f, 0.911588f,   0.466623f, 0.929133f,   0.466623f, 0.945554f,
          0.466623f, 0.960477f,   0.466623f, 0.973526f,   0.466623f, 0.984328f,
          0.466623f, 0.992507f,   0.466623f, 0.997690f,   0.466623f, 0.999501f,
          0.506770f, 0.888229f,   0.508637f, 0.906287f,   0.510428f, 0.923606f,
          0.512104f, 0.939816f,   0.513628f, 0.954546f,   0.514960f, 0.967427f,
          0.516062f, 0.978090f,   0.516897f, 0.986164f,   0.517426f, 0.991279f,
          0.517611f, 0.993067f,   0.544685f, 0.873535f,   0.548316f, 0.890910f,
          0.551798f, 0.907574f,   0.555057f, 0.923170f,   0.558019f, 0.937343f,
          0.560609f, 0.949736f,   0.562753f, 0.959996f,   0.564377f, 0.967764f,
          0.565405f, 0.972686f,   0.565765f, 0.974406f,   0.579899f, 0.849969f,
          0.585168f, 0.866248f,   0.590221f, 0.881860f,   0.594950f, 0.896472f,
          0.599248f, 0.909751f,   0.603007f, 0.921363f,   0.606118f, 0.930975f,
          0.608474f, 0.938253f,   0.609966f, 0.942864f,   0.610488f, 0.944475f,
          0.611941f, 0.818285f,   0.618701f, 0.833090f,   0.625184f, 0.847289f,
          0.631251f, 0.860578f,   0.636765f, 0.872655f,   0.641586f, 0.883215f,
          0.645577f, 0.891957f,   0.648600f, 0.898576f,   0.650514f, 0.902770f,
          0.651183f, 0.904236f,   0.640343f, 0.779239f,   0.648424f, 0.792228f,
          0.656174f, 0.804685f,   0.663427f, 0.816344f,   0.670018f, 0.826939f,
          0.675782f, 0.836204f,   0.680553f, 0.843873f,   0.684166f, 0.849680f,
          0.686455f, 0.853360f,   0.687255f, 0.854645f,   0.664635f, 0.733585f,
          0.673845f, 0.744450f,   0.682679f, 0.754871f,   0.690946f, 0.764624f,
          0.698459f, 0.773486f,   0.705029f, 0.781237f,   0.710468f, 0.787652f,
          0.714586f, 0.792510f,   0.717195f, 0.795588f,   0.718106f, 0.796663f,
          0.684346f, 0.682079f,   0.694473f, 0.690549f,   0.704186f, 0.698671f,
          0.713277f, 0.706274f,   0.721537f, 0.713182f,   0.728761f, 0.719224f,
          0.734741f, 0.724224f,   0.739269f, 0.728011f,   0.742138f, 0.730410f,
          0.743140f, 0.731249f,   0.699007f, 0.625476f,   0.709816f, 0.631313f,
          0.720183f, 0.636911f,   0.729886f, 0.642150f,   0.738703f, 0.646910f,
          0.746413f, 0.651074f,   0.752796f, 0.654520f,   0.757629f, 0.657129f,
          0.760691f, 0.658783f,   0.761760f, 0.659360f,   0.708148f, 0.564532f,
          0.719383f, 0.567533f,   0.730157f, 0.570412f,   0.740241f, 0.573106f,
          0.749405f, 0.575555f,   0.757419f, 0.577696f,   0.764053f, 0.579468f,
          0.769076f, 0.580810f,   0.772258f, 0.581661f,   0.773370f, 0.581958f,
          0.771223f, 0.418616f,   0.775197f, 0.500000f,   0.765395f, 0.420173f,
          0.769293f, 0.500000f,   0.756806f, 0.422468f,   0.760592f, 0.500000f,
          0.746376f, 0.425254f,   0.750026f, 0.500000f,   0.735027f, 0.428287f,
          0.738529f, 0.500000f,   0.723677f, 0.431319f,   0.727031f, 0.500000f,
          0.713248f, 0.434106f,   0.716466f, 0.500000f,   0.704659f, 0.436400f,
          0.707765f, 0.500000f,   0.698830f, 0.437958f,   0.701860f, 0.500000f,
          0.696683f, 0.438531f,   0.699685f, 0.500000f,   0.759694f, 0.341755f,
          0.754087f, 0.344783f,   0.745823f, 0.349245f,   0.735788f, 0.354663f,
          0.724868f, 0.360559f,   0.713948f, 0.366456f,   0.703913f, 0.371874f,
          0.695650f, 0.376336f,   0.690042f, 0.379364f,   0.687976f, 0.380479f,
          0.741205f, 0.270370f,   0.735951f, 0.274764f,   0.728208f, 0.281239f,
          0.718807f, 0.289101f,   0.708576f, 0.297657f,   0.698344f, 0.306213f,
          0.688943f, 0.314076f,   0.681200f, 0.320551f,   0.675947f, 0.324945f,
          0.674011f, 0.326563f,   0.716346f, 0.205413f,   0.711568f, 0.211050f,
          0.704526f, 0.219356f,   0.695976f, 0.229443f,   0.686671f, 0.240420f,
          0.677366f, 0.251396f,   0.668816f, 0.261483f,   0.661774f, 0.269789f,
          0.656996f, 0.275426f,   0.655236f, 0.277502f,   0.685711f, 0.147837f,
          0.681519f, 0.154575f,   0.675341f, 0.164505f,   0.667839f, 0.176563f,
          0.659676f, 0.189685f,   0.651513f, 0.202807f,   0.644011f, 0.214865f,
          0.637834f, 0.224795f,   0.633642f, 0.231533f,   0.632097f, 0.234016f,
          0.649891f, 0.098594f,   0.646385f, 0.106274f,   0.641217f, 0.117593f,
          0.634942f, 0.131337f,   0.628113f, 0.146294f,   0.621285f, 0.161250f,
          0.615010f, 0.174994f,   0.609842f, 0.186313f,   0.606335f, 0.193994f,
          0.605043f, 0.196823f,   0.609481f, 0.058636f,   0.606747f, 0.067081f,
          0.602719f, 0.079526f,   0.597828f, 0.094638f,   0.592505f, 0.111084f,
          0.587182f, 0.127529f,   0.582290f, 0.142642f,   0.578262f, 0.155087f,
          0.575529f, 0.163532f,   0.574521f, 0.166643f,   0.565071f, 0.028915f,
          0.563187f, 0.037929f,   0.560411f, 0.051212f,   0.557040f, 0.067342f,
          0.553372f, 0.084895f,   0.549704f, 0.102448f,   0.546333f, 0.118578f,
          0.543557f, 0.131861f,   0.541673f, 0.140875f,   0.540979f, 0.144196f,
          0.517254f, 0.010385f,   0.516285f, 0.019753f,   0.514858f, 0.033559f,
          0.513124f, 0.050323f,   0.511237f, 0.068566f,   0.509351f, 0.086810f,
          0.507617f, 0.103574f,   0.506190f, 0.117380f,   0.505221f, 0.126748f,
          0.504864f, 0.130200f,   0.466623f, 0.003996f,   0.466623f, 0.013486f,
          0.466623f, 0.027473f,   0.466623f, 0.044455f,   0.466623f, 0.062937f,
          0.466623f, 0.081419f,   0.466623f, 0.098402f,   0.466623f, 0.112388f,
          0.466623f, 0.121878f,   0.466623f, 0.125375f,   0.415992f, 0.010385f,
          0.416961f, 0.019753f,   0.418389f, 0.033559f,   0.420122f, 0.050323f,
          0.422009f, 0.068567f,   0.423895f, 0.086810f,   0.425629f, 0.103574f,
          0.427057f, 0.117380f,   0.428025f, 0.126748f,   0.428382f, 0.130200f,
          0.368176f, 0.028915f,   0.370059f, 0.037929f,   0.372835f, 0.051212f,
          0.376206f, 0.067342f,   0.379874f, 0.084895f,   0.383543f, 0.102448f,
          0.386914f, 0.118578f,   0.389689f, 0.131861f,   0.391573f, 0.140875f,
          0.392267f, 0.144196f,   0.323766f, 0.058636f,   0.326499f, 0.067081f,
          0.330527f, 0.079526f,   0.335419f, 0.094638f,   0.340742f, 0.111084f,
          0.346065f, 0.127530f,   0.350956f, 0.142642f,   0.354984f, 0.155087f,
          0.357718f, 0.163532f,   0.358725f, 0.166643f,   0.283355f, 0.098594f,
          0.286862f, 0.106274f,   0.292029f, 0.117593f,   0.298304f, 0.131337f,
          0.305133f, 0.146294f,   0.311962f, 0.161250f,   0.318237f, 0.174995f,
          0.323405f, 0.186313f,   0.326911f, 0.193994f,   0.328203f, 0.196823f,
          0.247536f, 0.147837f,   0.251728f, 0.154576f,   0.257905f, 0.164506f,
          0.265407f, 0.176564f,   0.273570f, 0.189685f,   0.281734f, 0.202807f,
          0.289235f, 0.214865f,   0.295413f, 0.224795f,   0.299605f, 0.231534f,
          0.301149f, 0.234016f,   0.216900f, 0.205413f,   0.221679f, 0.211050f,
          0.228720f, 0.219357f,   0.237271f, 0.229443f,   0.246576f, 0.240420f,
          0.255880f, 0.251396f,   0.264431f, 0.261483f,   0.271472f, 0.269789f,
          0.276251f, 0.275426f,   0.278011f, 0.277503f,   0.192042f, 0.270370f,
          0.197296f, 0.274764f,   0.205038f, 0.281239f,   0.214440f, 0.289101f,
          0.224671f, 0.297657f,   0.234902f, 0.306214f,   0.244304f, 0.314076f,
          0.252046f, 0.320551f,   0.257300f, 0.324945f,   0.259235f, 0.326564f,
          0.173552f, 0.341755f,   0.179160f, 0.344783f,   0.187423f, 0.349245f,
          0.197458f, 0.354663f,   0.208378f, 0.360560f,   0.219298f, 0.366456f,
          0.229333f, 0.371874f,   0.237597f, 0.376336f,   0.243204f, 0.379364f,
          0.245270f, 0.380480f,   0.162024f, 0.418616f,   0.167852f, 0.420173f,
          0.176441f, 0.422468f,   0.186870f, 0.425254f,   0.198220f, 0.428287f,
          0.209570f, 0.431319f,   0.219999f, 0.434106f,   0.228588f, 0.436401f,
          0.234416f, 0.437958f,   0.236563f, 0.438532f,   0.158049f, 0.500000f,
          0.163954f, 0.500000f,   0.172655f, 0.500000f,   0.183220f, 0.500000f,
          0.194718f, 0.500000f,   0.206215f, 0.500000f,   0.216781f, 0.500000f,
          0.225482f, 0.500000f,   0.231386f, 0.500000f,   0.233561f, 0.500000f,
          0.162024f, 0.581384f,   0.167852f, 0.579827f,   0.176441f, 0.577532f,
          0.186870f, 0.574746f,   0.198220f, 0.571713f,   0.209570f, 0.568681f,
          0.219999f, 0.565894f,   0.228588f, 0.563600f,   0.234416f, 0.562042f,
          0.236563f, 0.561469f,   0.173552f, 0.658245f,   0.179160f, 0.655217f,
          0.187424f, 0.650755f,   0.197458f, 0.645337f,   0.208378f, 0.639441f,
          0.219298f, 0.633544f,   0.229333f, 0.628126f,   0.237597f, 0.623664f,
          0.243204f, 0.620636f,   0.245270f, 0.619521f,   0.192042f, 0.729630f,
          0.197296f, 0.725236f,   0.205038f, 0.718761f,   0.214440f, 0.710899f,
          0.224671f, 0.702343f,   0.234902f, 0.693787f,   0.244304f, 0.685924f,
          0.252046f, 0.679449f,   0.257300f, 0.675055f,   0.259236f, 0.673437f,
          0.216901f, 0.794587f,   0.221679f, 0.788950f,   0.228720f, 0.780644f,
          0.237271f, 0.770557f,   0.246576f, 0.759580f,   0.255880f, 0.748604f,
          0.264431f, 0.738517f,   0.271472f, 0.730211f,   0.276251f, 0.724574f,
          0.278011f, 0.722498f,   0.247536f, 0.852163f,   0.251728f, 0.845425f,
          0.257906f, 0.835495f,   0.265407f, 0.823437f,   0.273570f, 0.810315f,
          0.281734f, 0.797193f,   0.289235f, 0.785135f,   0.295413f, 0.775205f,
          0.299605f, 0.768467f,   0.301149f, 0.765984f,   0.283355f, 0.901406f,
          0.286862f, 0.893726f,   0.292029f, 0.882407f,   0.298304f, 0.868663f,
          0.305133f, 0.853706f,   0.311962f, 0.838750f,   0.318237f, 0.825006f,
          0.323405f, 0.813687f,   0.326911f, 0.806006f,   0.328203f, 0.803177f,
          0.323766f, 0.941364f,   0.326499f, 0.932919f,   0.330528f, 0.920474f,
          0.335419f, 0.905362f,   0.340742f, 0.888916f,   0.346065f, 0.872471f,
          0.350956f, 0.857358f,   0.354985f, 0.844913f,   0.357718f, 0.836468f,
          0.358725f, 0.833357f,   0.368176f, 0.971085f,   0.370060f, 0.962071f,
          0.372836f, 0.948788f,   0.376206f, 0.932658f,   0.379875f, 0.915105f,
          0.383543f, 0.897552f,   0.386914f, 0.881422f,   0.389690f, 0.868139f,
          0.391573f, 0.859125f,   0.392267f, 0.855804f,   0.415993f, 0.989615f,
          0.416961f, 0.980247f,   0.418389f, 0.966441f,   0.420123f, 0.949677f,
          0.422009f, 0.931434f,   0.423896f, 0.913190f,   0.425629f, 0.896426f,
          0.427057f, 0.882620f,   0.428026f, 0.873252f,   0.428383f, 0.869800f,
          0.466623f, 0.996004f,   0.466623f, 0.986513f,   0.466623f, 0.972527f,
          0.466623f, 0.955544f,   0.466623f, 0.937063f,   0.466623f, 0.918581f,
          0.466623f, 0.901598f,   0.466623f, 0.887612f,   0.466623f, 0.878122f,
          0.466623f, 0.874625f,   0.517254f, 0.989615f,   0.516285f, 0.980247f,
          0.514858f, 0.966441f,   0.513124f, 0.949677f,   0.511238f, 0.931433f,
          0.509351f, 0.913190f,   0.507617f, 0.896426f,   0.506190f, 0.882620f,
          0.505221f, 0.873252f,   0.504864f, 0.869800f,   0.565071f, 0.971085f,
          0.563187f, 0.962071f,   0.560411f, 0.948788f,   0.557040f, 0.932658f,
          0.553372f, 0.915105f,   0.549704f, 0.897552f,   0.546333f, 0.881422f,
          0.543557f, 0.868139f,   0.541673f, 0.859125f,   0.540979f, 0.855804f,
          0.609481f, 0.941364f,   0.606747f, 0.932919f,   0.602719f, 0.920474f,
          0.597828f, 0.905362f,   0.592505f, 0.888916f,   0.587182f, 0.872470f,
          0.582290f, 0.857358f,   0.578262f, 0.844913f,   0.575529f, 0.836468f,
          0.574522f, 0.833357f,   0.649891f, 0.901406f,   0.646385f, 0.893726f,
          0.641217f, 0.882407f,   0.634942f, 0.868663f,   0.628113f, 0.853706f,
          0.621285f, 0.838750f,   0.615010f, 0.825005f,   0.609842f, 0.813687f,
          0.606335f, 0.806006f,   0.605043f, 0.803177f,   0.685711f, 0.852163f,
          0.681519f, 0.845424f,   0.675341f, 0.835494f,   0.667840f, 0.823436f,
          0.659676f, 0.810315f,   0.651513f, 0.797193f,   0.644011f, 0.785135f,
          0.637834f, 0.775205f,   0.633642f, 0.768466f,   0.632097f, 0.765984f,
          0.716346f, 0.794587f,   0.711568f, 0.788950f,   0.704526f, 0.780643f,
          0.695976f, 0.770557f,   0.686671f, 0.759580f,   0.677366f, 0.748604f,
          0.668816f, 0.738517f,   0.661774f, 0.730211f,   0.656996f, 0.724574f,
          0.655236f, 0.722497f,   0.741205f, 0.729630f,   0.735951f, 0.725236f,
          0.728208f, 0.718761f,   0.718807f, 0.710899f,   0.708576f, 0.702343f,
          0.698345f, 0.693786f,   0.688943f, 0.685924f,   0.681201f, 0.679449f,
          0.675947f, 0.675055f,   0.674011f, 0.673436f,   0.759694f, 0.658245f,
          0.754087f, 0.655217f,   0.745823f, 0.650755f,   0.735788f, 0.645337f,
          0.724868f, 0.639440f,   0.713948f, 0.633544f,   0.703914f, 0.628126f,
          0.695650f, 0.623664f,   0.690042f, 0.620636f,   0.687976f, 0.619520f,
          0.771223f, 0.581384f,   0.765395f, 0.579827f,   0.756806f, 0.577532f,
          0.746376f, 0.574746f,   0.735027f, 0.571713f,   0.723677f, 0.568681f,
          0.713248f, 0.565894f,   0.704659f, 0.563599f,   0.698830f, 0.562042f,
          0.696683f, 0.561468f,   0.696143f, 0.438676f,   0.699137f, 0.500000f,
          0.693738f, 0.439318f,   0.696702f, 0.500000f,   0.688298f, 0.440772f,
          0.691190f, 0.500000f,   0.678647f, 0.443351f,   0.681413f, 0.500000f,
          0.663612f, 0.447367f,   0.666182f, 0.500000f,   0.642021f, 0.453136f,
          0.644310f, 0.500000f,   0.612700f, 0.460971f,   0.614606f, 0.500000f,
          0.574475f, 0.471184f,   0.575883f, 0.500000f,   0.526174f, 0.484089f,
          0.526951f, 0.500000f,   0.466623f, 0.500000f,   0.687456f, 0.380760f,
          0.685143f, 0.382009f,   0.679908f, 0.384836f,   0.670622f, 0.389850f,
          0.656157f, 0.397660f,   0.635383f, 0.408877f,   0.607171f, 0.424110f,
          0.570393f, 0.443969f,   0.523920f, 0.469062f,   0.673524f, 0.326971f,
          0.671356f, 0.328783f,   0.666452f, 0.332885f,   0.657752f, 0.340161f,
          0.644199f, 0.351495f,   0.624736f, 0.367772f,   0.598304f, 0.389876f,
          0.563847f, 0.418693f,   0.520306f, 0.455106f,   0.654792f, 0.278025f,
          0.652821f, 0.280350f,   0.648361f, 0.285612f,   0.640448f, 0.294946f,
          0.628123f, 0.309486f,   0.610421f, 0.330368f,   0.586383f, 0.358725f,
          0.555045f, 0.395693f,   0.515445f, 0.442407f,   0.631708f, 0.234641f,
          0.629979f, 0.237421f,   0.626066f, 0.243711f,   0.619124f, 0.254869f,
          0.608310f, 0.272251f,   0.592781f, 0.297214f,   0.571691f, 0.331113f,
          0.544197f, 0.375307f,   0.509456f, 0.431150f,   0.604718f, 0.197536f,
          0.603271f, 0.200704f,   0.599998f, 0.207874f,   0.594191f, 0.220592f,
          0.585145f, 0.240405f,   0.572155f, 0.268858f,   0.554513f, 0.307498f,
          0.531515f, 0.357871f,   0.502453f, 0.421523f,   0.574268f, 0.167427f,
          0.573140f, 0.170910f,   0.570589f, 0.178794f,   0.566062f, 0.192778f,
          0.559011f, 0.214563f,   0.548885f, 0.245849f,   0.535133f, 0.288335f,
          0.517206f, 0.343722f,   0.494553f, 0.413711f,   0.540805f, 0.145032f,
          0.540027f, 0.148750f,   0.538269f, 0.157165f,   0.535150f, 0.172091f,
          0.530291f, 0.195343f,   0.523312f, 0.228735f,   0.513836f, 0.274082f,
          0.501481f, 0.333199f,   0.485870f, 0.407900f,   0.504774f, 0.131069f,
          0.504374f, 0.134933f,   0.503470f, 0.143679f,   0.501866f, 0.159192f,
          0.499367f, 0.183359f,   0.495778f, 0.218064f,   0.490904f, 0.265195f,
          0.484550f, 0.326638f,   0.476522f, 0.404277f,   0.466623f, 0.126255f,
          0.466623f, 0.130170f,   0.466623f, 0.139030f,   0.466623f, 0.154745f,
          0.466623f, 0.179227f,   0.466623f, 0.214386f,   0.466623f, 0.262132f,
          0.466623f, 0.324376f,   0.466623f, 0.403028f,   0.428472f, 0.131069f,
          0.428872f, 0.134933f,   0.429776f, 0.143679f,   0.431380f, 0.159192f,
          0.433880f, 0.183359f,   0.437468f, 0.218064f,   0.442342f, 0.265195f,
          0.448696f, 0.326638f,   0.456725f, 0.404277f,   0.392442f, 0.145032f,
          0.393219f, 0.148750f,   0.394977f, 0.157165f,   0.398097f, 0.172091f,
          0.402956f, 0.195343f,   0.409934f, 0.228735f,   0.419411f, 0.274082f,
          0.431765f, 0.333199f,   0.447376f, 0.407900f,   0.358978f, 0.167427f,
          0.360106f, 0.170910f,   0.362658f, 0.178794f,   0.367184f, 0.192779f,
          0.374235f, 0.214563f,   0.384362f, 0.245849f,   0.398113f, 0.288335f,
          0.416041f, 0.343722f,   0.438694f, 0.413711f,   0.328528f, 0.197536f,
          0.329975f, 0.200704f,   0.333248f, 0.207874f,   0.339055f, 0.220592f,
          0.348101f, 0.240405f,   0.361092f, 0.268858f,   0.378733f, 0.307498f,
          0.401732f, 0.357871f,   0.430793f, 0.421523f,   0.301538f, 0.234641f,
          0.303267f, 0.237421f,   0.307181f, 0.243711f,   0.314122f, 0.254869f,
          0.324936f, 0.272251f,   0.340466f, 0.297214f,   0.361556f, 0.331114f,
          0.389049f, 0.375307f,   0.423790f, 0.431150f,   0.278454f, 0.278025f,
          0.280425f, 0.280351f,   0.284886f, 0.285613f,   0.292798f, 0.294946f,
          0.305124f, 0.309487f,   0.322825f, 0.330368f,   0.346864f, 0.358725f,
          0.378202f, 0.395693f,   0.417801f, 0.442407f,   0.259723f, 0.326971f,
          0.261890f, 0.328784f,   0.266795f, 0.332885f,   0.275495f, 0.340161f,
          0.289047f, 0.351495f,   0.308511f, 0.367772f,   0.334942f, 0.389877f,
          0.369400f, 0.418693f,   0.412941f, 0.455106f,   0.245791f, 0.380761f,
          0.248104f, 0.382010f,   0.253339f, 0.384836f,   0.262624f, 0.389850f,
          0.277090f, 0.397661f,   0.297864f, 0.408878f,   0.326075f, 0.424111f,
          0.362853f, 0.443969f,   0.409326f, 0.469062f,   0.237104f, 0.438676f,
          0.239508f, 0.439318f,   0.244949f, 0.440772f,   0.254600f, 0.443351f,
          0.269634f, 0.447368f,   0.291225f, 0.453137f,   0.320547f, 0.460971f,
          0.358771f, 0.471184f,   0.407072f, 0.484089f,   0.234109f, 0.500000f,
          0.236545f, 0.500000f,   0.242056f, 0.500000f,   0.251833f, 0.500000f,
          0.267064f, 0.500000f,   0.288937f, 0.500000f,   0.318641f, 0.500000f,
          0.357364f, 0.500000f,   0.406295f, 0.500000f,   0.237104f, 0.561324f,
          0.239508f, 0.560682f,   0.244949f, 0.559228f,   0.254600f, 0.556649f,
          0.269634f, 0.552633f,   0.291225f, 0.546864f,   0.320547f, 0.539029f,
          0.358771f, 0.528816f,   0.407072f, 0.515911f,   0.245791f, 0.619240f,
          0.248104f, 0.617991f,   0.253339f, 0.615164f,   0.262625f, 0.610150f,
          0.277090f, 0.602340f,   0.297864f, 0.591122f,   0.326075f, 0.575890f,
          0.362853f, 0.556031f,   0.409326f, 0.530938f,   0.259723f, 0.673029f,
          0.261890f, 0.671217f,   0.266795f, 0.667115f,   0.275495f, 0.659839f,
          0.289048f, 0.648505f,   0.308511f, 0.632228f,   0.334942f, 0.610124f,
          0.369400f, 0.581307f,   0.412941f, 0.544894f,   0.278454f, 0.721975f,
          0.280425f, 0.719650f,   0.284886f, 0.714388f,   0.292798f, 0.705054f,
          0.305124f, 0.690514f,   0.322825f, 0.669632f,   0.346864f, 0.641275f,
          0.378202f, 0.604307f,   0.417801f, 0.557593f,   0.301538f, 0.765359f,
          0.303267f, 0.762579f,   0.307181f, 0.756289f,   0.314123f, 0.745131f,
          0.324936f, 0.727749f,   0.340466f, 0.702786f,   0.361556f, 0.668887f,
          0.389049f, 0.624693f,   0.423790f, 0.568850f,   0.328528f, 0.802464f,
          0.329975f, 0.799296f,   0.333249f, 0.792126f,   0.339055f, 0.779408f,
          0.348101f, 0.759595f,   0.361092f, 0.731142f,   0.378733f, 0.692502f,
          0.401732f, 0.642129f,   0.430793f, 0.578477f,   0.358979f, 0.832573f,
          0.360106f, 0.829090f,   0.362658f, 0.821206f,   0.367184f, 0.807222f,
          0.374235f, 0.785437f,   0.384362f, 0.754151f,   0.398113f, 0.711665f,
          0.416041f, 0.656278f,   0.438694f, 0.586289f,   0.392442f, 0.854968f,
          0.393219f, 0.851250f,   0.394978f, 0.842835f,   0.398097f, 0.827909f,
          0.402956f, 0.804657f,   0.409934f, 0.771265f,   0.419411f, 0.725918f,
          0.431765f, 0.666801f,   0.447376f, 0.592100f,   0.428472f, 0.868931f,
          0.428872f, 0.865067f,   0.429776f, 0.856321f,   0.431381f, 0.840808f,
          0.433880f, 0.816641f,   0.437469f, 0.781936f,   0.442342f, 0.734805f,
          0.448696f, 0.673362f,   0.456725f, 0.595723f,   0.466623f, 0.873745f,
          0.466623f, 0.869830f,   0.466623f, 0.860970f,   0.466623f, 0.845255f,
          0.466623f, 0.820773f,   0.466623f, 0.785614f,   0.466623f, 0.737868f,
          0.466623f, 0.675624f,   0.466623f, 0.596972f,   0.504774f, 0.868931f,
          0.504375f, 0.865067f,   0.503470f, 0.856321f,   0.501866f, 0.840808f,
          0.499367f, 0.816641f,   0.495778f, 0.781936f,   0.490904f, 0.734805f,
          0.484551f, 0.673362f,   0.476522f, 0.595723f,   0.540805f, 0.854968f,
          0.540028f, 0.851250f,   0.538269f, 0.842835f,   0.535150f, 0.827909f,
          0.530291f, 0.804657f,   0.523312f, 0.771265f,   0.513836f, 0.725918f,
          0.501481f, 0.666801f,   0.485870f, 0.592100f,   0.574268f, 0.832573f,
          0.573141f, 0.829090f,   0.570589f, 0.821206f,   0.566062f, 0.807221f,
          0.559011f, 0.785437f,   0.548885f, 0.754151f,   0.535133f, 0.711665f,
          0.517206f, 0.656278f,   0.494553f, 0.586289f,   0.604718f, 0.802464f,
          0.603272f, 0.799296f,   0.599998f, 0.792126f,   0.594191f, 0.779408f,
          0.585145f, 0.759595f,   0.572155f, 0.731142f,   0.554513f, 0.692502f,
          0.531515f, 0.642129f,   0.502453f, 0.578477f,   0.631708f, 0.765359f,
          0.629979f, 0.762579f,   0.626066f, 0.756289f,   0.619124f, 0.745131f,
          0.608310f, 0.727749f,   0.592781f, 0.702786f,   0.571691f, 0.668886f,
          0.544197f, 0.624693f,   0.509456f, 0.568850f,   0.654792f, 0.721975f,
          0.652821f, 0.719649f,   0.648361f, 0.714387f,   0.640448f, 0.705054f,
          0.628123f, 0.690513f,   0.610421f, 0.669632f,   0.586383f, 0.641275f,
          0.555045f, 0.604307f,   0.515446f, 0.557593f,   0.673524f, 0.673029f,
          0.671356f, 0.671216f,   0.666452f, 0.667115f,   0.657752f, 0.659839f,
          0.644199f, 0.648505f,   0.624736f, 0.632228f,   0.598304f, 0.610123f,
          0.563847f, 0.581307f,   0.520306f, 0.544894f,   0.687456f, 0.619239f,
          0.685143f, 0.617990f,   0.679908f, 0.615164f,   0.670622f, 0.610150f,
          0.656157f, 0.602339f,   0.635383f, 0.591122f,   0.607171f, 0.575889f,
          0.570393f, 0.556031f,   0.523920f, 0.530938f,   0.696143f, 0.561324f,
          0.693739f, 0.560682f,   0.688298f, 0.559228f,   0.678647f, 0.556649f,
          0.663612f, 0.552632f,   0.642021f, 0.546863f,   0.612700f, 0.539029f,
          0.574475f, 0.528816f,   0.526174f, 0.515911f,   0.218024f, 0.500000f,
          0.218459f, 0.479770f,   0.186878f, 0.479770f,   0.186809f, 0.500000f,
          0.158237f, 0.479770f,   0.158484f, 0.500000f,   0.132626f, 0.479770f,
          0.133143f, 0.500000f,   0.110136f, 0.479770f,   0.110878f, 0.500000f,
          0.090858f, 0.479770f,   0.091782f, 0.500000f,   0.074881f, 0.479770f,
          0.075949f, 0.500000f,   0.062297f, 0.479770f,   0.063473f, 0.500000f,
          0.053196f, 0.479770f,   0.054446f, 0.500000f,   0.047669f, 0.479770f,
          0.048961f, 0.500000f,   0.045807f, 0.479770f,   0.047112f, 0.500000f,
          0.219640f, 0.464036f,   0.187066f, 0.464036f,   0.157567f, 0.464036f,
          0.131225f, 0.464036f,   0.108124f, 0.464036f,   0.088348f, 0.464036f,
          0.071981f, 0.464036f,   0.059105f, 0.464036f,   0.049805f, 0.464036f,
          0.044163f, 0.464036f,   0.042264f, 0.464036f,   0.221380f, 0.452797f,
          0.187343f, 0.452797f,   0.156578f, 0.452797f,   0.129159f, 0.452797f,
          0.105159f, 0.452797f,   0.084650f, 0.452797f,   0.067707f, 0.452797f,
          0.054401f, 0.452797f,   0.044807f, 0.452797f,   0.038997f, 0.452797f,
          0.037044f, 0.452797f,   0.223493f, 0.446054f,   0.187679f, 0.446054f,
          0.155378f, 0.446054f,   0.126651f, 0.446054f,   0.101558f, 0.446054f,
          0.080160f, 0.446054f,   0.062517f, 0.446054f,   0.048690f, 0.446054f,
          0.038738f, 0.446054f,   0.032723f, 0.446054f,   0.030704f, 0.446054f,
          0.225793f, 0.443806f,   0.188044f, 0.443806f,   0.154072f, 0.443806f,
          0.123921f, 0.443806f,   0.097640f, 0.443806f,   0.075273f, 0.443806f,
          0.056869f, 0.443806f,   0.042474f, 0.443806f,   0.032134f, 0.443806f,
          0.025896f, 0.443806f,   0.023806f, 0.443806f,   0.228092f, 0.446054f,
          0.188410f, 0.446054f,   0.152766f, 0.446054f,   0.121192f, 0.446054f,
          0.093721f, 0.446054f,   0.070387f, 0.446054f,   0.051222f, 0.446054f,
          0.036258f, 0.446054f,   0.025530f, 0.446054f,   0.019068f, 0.446054f,
          0.016907f, 0.446054f,   0.230205f, 0.452797f,   0.188746f, 0.452797f,
          0.151565f, 0.452797f,   0.118684f, 0.452797f,   0.090121f, 0.452797f,
          0.065897f, 0.452797f,   0.046032f, 0.452797f,   0.030547f, 0.452797f,
          0.019461f, 0.452797f,   0.012794f, 0.452797f,   0.010568f, 0.452797f,
          0.231945f, 0.464036f,   0.189023f, 0.464036f,   0.150577f, 0.464036f,
          0.116618f, 0.464036f,   0.087155f, 0.464036f,   0.062199f, 0.464036f,
          0.041758f, 0.464036f,   0.025843f, 0.464036f,   0.014463f, 0.464036f,
          0.007628f, 0.464036f,   0.005347f, 0.464036f,   0.233126f, 0.479770f,
          0.189210f, 0.479770f,   0.149906f, 0.479770f,   0.115216f, 0.479770f,
          0.085143f, 0.479770f,   0.059689f, 0.479770f,   0.038858f, 0.479770f,
          0.022651f, 0.479770f,   0.011071f, 0.479770f,   0.004122f, 0.479770f,
          0.001805f, 0.479770f,   0.233561f, 0.500000f,   0.189280f, 0.500000f,
          0.149659f, 0.500000f,   0.114700f, 0.500000f,   0.084402f, 0.500000f,
          0.058765f, 0.500000f,   0.037789f, 0.500000f,   0.021475f, 0.500000f,
          0.009822f, 0.500000f,   0.002830f, 0.500000f,   0.000500f, 0.500000f,
          0.233126f, 0.520230f,   0.189210f, 0.520230f,   0.149906f, 0.520230f,
          0.115216f, 0.520230f,   0.085143f, 0.520230f,   0.059689f, 0.520230f,
          0.038858f, 0.520230f,   0.022651f, 0.520230f,   0.011072f, 0.520230f,
          0.004122f, 0.520230f,   0.001805f, 0.520230f,   0.231945f, 0.535964f,
          0.189023f, 0.535964f,   0.150577f, 0.535964f,   0.116618f, 0.535964f,
          0.087155f, 0.535964f,   0.062199f, 0.535964f,   0.041758f, 0.535964f,
          0.025843f, 0.535964f,   0.014463f, 0.535964f,   0.007628f, 0.535964f,
          0.005347f, 0.535964f,   0.230205f, 0.547203f,   0.188746f, 0.547203f,
          0.151565f, 0.547203f,   0.118684f, 0.547203f,   0.090121f, 0.547203f,
          0.065897f, 0.547203f,   0.046032f, 0.547203f,   0.030547f, 0.547203f,
          0.019461f, 0.547203f,   0.012794f, 0.547203f,   0.010568f, 0.547203f,
          0.228092f, 0.553946f,   0.188410f, 0.553946f,   0.152766f, 0.553946f,
          0.121192f, 0.553946f,   0.093721f, 0.553946f,   0.070387f, 0.553946f,
          0.051222f, 0.553946f,   0.036258f, 0.553946f,   0.025530f, 0.553946f,
          0.019068f, 0.553946f,   0.016907f, 0.553946f,   0.225793f, 0.556194f,
          0.188044f, 0.556194f,   0.154072f, 0.556194f,   0.123921f, 0.556194f,
          0.097640f, 0.556194f,   0.075273f, 0.556194f,   0.056869f, 0.556194f,
          0.042474f, 0.556194f,   0.032134f, 0.556194f,   0.025896f, 0.556194f,
          0.023806f, 0.556194f,   0.223493f, 0.553946f,   0.187679f, 0.553946f,
          0.155378f, 0.553946f,   0.126651f, 0.553946f,   0.101558f, 0.553946f,
          0.080160f, 0.553946f,   0.062517f, 0.553946f,   0.048690f, 0.553946f,
          0.038738f, 0.553946f,   0.032723f, 0.553946f,   0.030704f, 0.553946f,
          0.221380f, 0.547203f,   0.187343f, 0.547203f,   0.156578f, 0.547203f,
          0.129159f, 0.547203f,   0.105159f, 0.547203f,   0.084650f, 0.547203f,
          0.067707f, 0.547203f,   0.054401f, 0.547203f,   0.044807f, 0.547203f,
          0.038997f, 0.547203f,   0.037044f, 0.547203f,   0.219640f, 0.535964f,
          0.187066f, 0.535964f,   0.157567f, 0.535964f,   0.131225f, 0.535964f,
          0.108124f, 0.535964f,   0.088348f, 0.535964f,   0.071981f, 0.535964f,
          0.059105f, 0.535964f,   0.049805f, 0.535964f,   0.044163f, 0.535964f,
          0.042264f, 0.535964f,   0.218459f, 0.520230f,   0.186878f, 0.520230f,
          0.158237f, 0.520230f,   0.132626f, 0.520230f,   0.110136f, 0.520230f,
          0.090858f, 0.520230f,   0.074881f, 0.520230f,   0.062297f, 0.520230f,
          0.053196f, 0.520230f,   0.047669f, 0.520230f,   0.045807f, 0.520230f,
          0.046774f, 0.479770f,   0.048060f, 0.500000f,   0.049737f, 0.479770f,
          0.050965f, 0.500000f,   0.054787f, 0.479770f,   0.055922f, 0.500000f,
          0.062016f, 0.479770f,   0.063022f, 0.500000f,   0.071517f, 0.479770f,
          0.072360f, 0.500000f,   0.083382f, 0.479770f,   0.084029f, 0.500000f,
          0.097701f, 0.479770f,   0.098121f, 0.500000f,   0.114567f, 0.479770f,
          0.114731f, 0.500000f,   0.134073f, 0.479770f,   0.133951f, 0.500000f,
          0.156309f, 0.479770f,   0.155874f, 0.500000f,   0.043284f, 0.464036f,
          0.046402f, 0.464036f,   0.051707f, 0.464036f,   0.059286f, 0.464036f,
          0.069229f, 0.464036f,   0.081624f, 0.464036f,   0.096560f, 0.464036f,
          0.114123f, 0.464036f,   0.134404f, 0.464036f,   0.157490f, 0.464036f,
          0.038141f, 0.452797f,   0.041488f, 0.452797f,   0.047167f, 0.452797f,
          0.055263f, 0.452797f,   0.065858f, 0.452797f,   0.079035f, 0.452797f,
          0.094878f, 0.452797f,   0.113469f, 0.452797f,   0.134892f, 0.452797f,
          0.159230f, 0.452797f,   0.031895f, 0.446054f,   0.035520f, 0.446054f,
          0.041655f, 0.446054f,   0.050377f, 0.446054f,   0.061764f, 0.446054f,
          0.075891f, 0.446054f,   0.092835f, 0.446054f,   0.112675f, 0.446054f,
          0.135485f, 0.446054f,   0.161343f, 0.446054f,   0.025099f, 0.443806f,
          0.029026f, 0.443806f,   0.035657f, 0.443806f,   0.045061f, 0.443806f,
          0.057308f, 0.443806f,   0.072469f, 0.443806f,   0.090613f, 0.443806f,
          0.111810f, 0.443806f,   0.136130f, 0.443806f,   0.163643f, 0.443806f,
          0.018303f, 0.446054f,   0.022532f, 0.446054f,   0.029659f, 0.446054f,
          0.039744f, 0.446054f,   0.052853f, 0.446054f,   0.069047f, 0.446054f,
          0.088390f, 0.446054f,   0.110945f, 0.446054f,   0.136775f, 0.446054f,
          0.165942f, 0.446054f,   0.012058f, 0.452797f,   0.016565f, 0.452797f,
          0.024146f, 0.452797f,   0.034859f, 0.452797f,   0.048759f, 0.452797f,
          0.065903f, 0.452797f,   0.086348f, 0.452797f,   0.110151f, 0.452797f,
          0.137368f, 0.452797f,   0.168055f, 0.452797f,   0.006915f, 0.464036f,
          0.011651f, 0.464036f,   0.019607f, 0.464036f,   0.030836f, 0.464036f,
          0.045387f, 0.464036f,   0.063314f, 0.464036f,   0.084666f, 0.464036f,
          0.109496f, 0.464036f,   0.137856f, 0.464036f,   0.169796f, 0.464036f,
          0.003425f, 0.479770f,   0.008316f, 0.479770f,   0.016527f, 0.479770f,
          0.028105f, 0.479770f,   0.043099f, 0.479770f,   0.061556f, 0.479770f,
          0.083525f, 0.479770f,   0.109052f, 0.479770f,   0.138187f, 0.479770f,
          0.170976f, 0.479770f,   0.002139f, 0.500000f,   0.007087f, 0.500000f,
          0.015392f, 0.500000f,   0.027100f, 0.500000f,   0.042256f, 0.500000f,
          0.060909f, 0.500000f,   0.083104f, 0.500000f,   0.108889f, 0.500000f,
          0.138309f, 0.500000f,   0.171412f, 0.500000f,   0.003425f, 0.520230f,
          0.008316f, 0.520230f,   0.016527f, 0.520230f,   0.028105f, 0.520230f,
          0.043099f, 0.520230f,   0.061556f, 0.520230f,   0.083525f, 0.520230f,
          0.109052f, 0.520230f,   0.138187f, 0.520230f,   0.170977f, 0.520230f,
          0.006915f, 0.535964f,   0.011651f, 0.535964f,   0.019607f, 0.535964f,
          0.030836f, 0.535964f,   0.045387f, 0.535964f,   0.063314f, 0.535964f,
          0.084666f, 0.535964f,   0.109496f, 0.535964f,   0.137856f, 0.535964f,
          0.169796f, 0.535964f,   0.012058f, 0.547203f,   0.016565f, 0.547203f,
          0.024146f, 0.547203f,   0.034859f, 0.547203f,   0.048759f, 0.547203f,
          0.065903f, 0.547203f,   0.086348f, 0.547203f,   0.110151f, 0.547203f,
          0.137368f, 0.547203f,   0.168055f, 0.547203f,   0.018303f, 0.553946f,
          0.022532f, 0.553946f,   0.029659f, 0.553946f,   0.039744f, 0.553946f,
          0.052853f, 0.553946f,   0.069047f, 0.553946f,   0.088390f, 0.553946f,
          0.110945f, 0.553946f,   0.136775f, 0.553946f,   0.165942f, 0.553946f,
          0.025099f, 0.556194f,   0.029026f, 0.556194f,   0.035657f, 0.556194f,
          0.045061f, 0.556194f,   0.057308f, 0.556194f,   0.072469f, 0.556194f,
          0.090613f, 0.556194f,   0.111810f, 0.556194f,   0.136130f, 0.556194f,
          0.163643f, 0.556194f,   0.031895f, 0.553946f,   0.035520f, 0.553946f,
          0.041655f, 0.553946f,   0.050377f, 0.553946f,   0.061764f, 0.553946f,
          0.075891f, 0.553946f,   0.092835f, 0.553946f,   0.112675f, 0.553946f,
          0.135485f, 0.553946f,   0.161343f, 0.553946f,   0.038141f, 0.547203f,
          0.041488f, 0.547203f,   0.047167f, 0.547203f,   0.055263f, 0.547203f,
          0.065858f, 0.547203f,   0.079035f, 0.547203f,   0.094878f, 0.547203f,
          0.113469f, 0.547203f,   0.134892f, 0.547203f,   0.159230f, 0.547203f,
          0.043284f, 0.535964f,   0.046402f, 0.535964f,   0.051707f, 0.535964f,
          0.059286f, 0.535964f,   0.069229f, 0.535964f,   0.081624f, 0.535964f,
          0.096560f, 0.535964f,   0.114123f, 0.535964f,   0.134404f, 0.535964f,
          0.157490f, 0.535964f,   0.046774f, 0.520230f,   0.049737f, 0.520230f,
          0.054787f, 0.520230f,   0.062016f, 0.520230f,   0.071517f, 0.520230f,
          0.083382f, 0.520230f,   0.097701f, 0.520230f,   0.114567f, 0.520230f,
          0.134073f, 0.520230f,   0.156309f, 0.520230f,   0.730760f, 0.500000f,
          0.730760f, 0.455494f,   0.767956f, 0.456269f,   0.767413f, 0.500000f,
          0.795548f, 0.458370f,   0.794650f, 0.500000f,   0.815355f, 0.461466f,
          0.814243f, 0.500000f,   0.829194f, 0.465226f,   0.827962f, 0.500000f,
          0.838885f, 0.469318f,   0.837580f, 0.500000f,   0.846245f, 0.473410f,
          0.844867f, 0.500000f,   0.853093f, 0.477170f,   0.851595f, 0.500000f,
          0.861247f, 0.480266f,   0.859535f, 0.500000f,   0.872525f, 0.482368f,
          0.870457f, 0.500000f,   0.888745f, 0.483142f,   0.886135f, 0.500000f,
          0.730760f, 0.420879f,   0.769430f, 0.422255f,   0.797985f, 0.425991f,
          0.818373f, 0.431496f,   0.832539f, 0.438180f,   0.842428f, 0.445454f,
          0.849986f, 0.452729f,   0.857160f, 0.459413f,   0.865895f, 0.464918f,
          0.878136f, 0.468654f,   0.895830f, 0.470030f,   0.730760f, 0.396154f,
          0.771601f, 0.397960f,   0.801577f, 0.402863f,   0.822821f, 0.410088f,
          0.837467f, 0.418861f,   0.847648f, 0.428409f,   0.855499f, 0.437956f,
          0.863153f, 0.446730f,   0.872744f, 0.453955f,   0.886405f, 0.458858f,
          0.906271f, 0.460664f,   0.730760f, 0.381319f,   0.774238f, 0.383383f,
          0.805938f, 0.388986f,   0.828222f, 0.397243f,   0.843451f, 0.407270f,
          0.853988f, 0.418182f,   0.862193f, 0.429093f,   0.870431f, 0.439120f,
          0.881061f, 0.447377f,   0.896447f, 0.452980f,   0.918950f, 0.455045f,
          0.730760f, 0.376374f,   0.777108f, 0.378524f,   0.810685f, 0.384361f,
          0.834100f, 0.392962f,   0.849963f, 0.403406f,   0.860886f, 0.414773f,
          0.869478f, 0.426139f,   0.878350f, 0.436583f,   0.890112f, 0.445185f,
          0.907374f, 0.451021f,   0.932747f, 0.453172f,   0.730760f, 0.381319f,
          0.779978f, 0.383383f,   0.815431f, 0.388986f,   0.839977f, 0.397243f,
          0.856476f, 0.407270f,   0.867785f, 0.418182f,   0.876763f, 0.429093f,
          0.886270f, 0.439120f,   0.899163f, 0.447377f,   0.918302f, 0.452980f,
          0.946544f, 0.455045f,   0.730760f, 0.396154f,   0.782615f, 0.397960f,
          0.819792f, 0.402863f,   0.845378f, 0.410088f,   0.862460f, 0.418861f,
          0.874124f, 0.428409f,   0.883458f, 0.437956f,   0.893547f, 0.446730f,
          0.907480f, 0.453955f,   0.928343f, 0.458858f,   0.959223f, 0.460664f,
          0.730760f, 0.420879f,   0.784787f, 0.422255f,   0.823384f, 0.425991f,
          0.849826f, 0.431496f,   0.867388f, 0.438180f,   0.879345f, 0.445454f,
          0.888971f, 0.452729f,   0.899541f, 0.459413f,   0.914330f, 0.464918f,
          0.936613f, 0.468654f,   0.969664f, 0.470030f,   0.730760f, 0.455494f,
          0.786261f, 0.456269f,   0.825821f, 0.458370f,   0.852845f, 0.461466f,
          0.870732f, 0.465226f,   0.882887f, 0.469318f,   0.892712f, 0.473410f,
          0.903608f, 0.477170f,   0.918978f, 0.480266f,   0.942224f, 0.482368f,
          0.976749f, 0.483142f,   0.730760f, 0.500000f,   0.786804f, 0.500000f,
          0.826719f, 0.500000f,   0.853957f, 0.500000f,   0.871964f, 0.500000f,
          0.884192f, 0.500000f,   0.894090f, 0.500000f,   0.905106f, 0.500000f,
          0.920690f, 0.500000f,   0.944291f, 0.500000f,   0.979359f, 0.500000f,
          0.730760f, 0.544505f,   0.786261f, 0.543731f,   0.825821f, 0.541630f,
          0.852845f, 0.538534f,   0.870732f, 0.534773f,   0.882887f, 0.530682f,
          0.892712f, 0.526590f,   0.903608f, 0.522830f,   0.918978f, 0.519733f,
          0.942224f, 0.517632f,   0.976749f, 0.516858f,   0.730760f, 0.579121f,
          0.784787f, 0.577745f,   0.823384f, 0.574009f,   0.849826f, 0.568504f,
          0.867388f, 0.561820f,   0.879345f, 0.554545f,   0.888971f, 0.547271f,
          0.899541f, 0.540586f,   0.914330f, 0.535082f,   0.936613f, 0.531346f,
          0.969664f, 0.529970f,   0.730760f, 0.603846f,   0.782615f, 0.602040f,
          0.819792f, 0.597137f,   0.845378f, 0.589912f,   0.862460f, 0.581138f,
          0.874124f, 0.571591f,   0.883458f, 0.562043f,   0.893547f, 0.553270f,
          0.907480f, 0.546045f,   0.928343f, 0.541142f,   0.959223f, 0.539335f,
          0.730760f, 0.618681f,   0.779978f, 0.616617f,   0.815431f, 0.611014f,
          0.839977f, 0.602756f,   0.856476f, 0.592730f,   0.867785f, 0.581818f,
          0.876763f, 0.570907f,   0.886270f, 0.560880f,   0.899163f, 0.552622f,
          0.918302f, 0.547019f,   0.946544f, 0.544955f,   0.730760f, 0.623626f,
          0.777108f, 0.621476f,   0.810685f, 0.615639f,   0.834100f, 0.607038f,
          0.849963f, 0.596593f,   0.860886f, 0.585227f,   0.869478f, 0.573861f,
          0.878350f, 0.563416f,   0.890112f, 0.554815f,   0.907374f, 0.548978f,
          0.932747f, 0.546828f,   0.730760f, 0.618681f,   0.774238f, 0.616617f,
          0.805938f, 0.611014f,   0.828222f, 0.602756f,   0.843451f, 0.592730f,
          0.853988f, 0.581818f,   0.862194f, 0.570907f,   0.870431f, 0.560880f,
          0.881061f, 0.552622f,   0.896447f, 0.547019f,   0.918950f, 0.544955f,
          0.730760f, 0.603846f,   0.771601f, 0.602040f,   0.801577f, 0.597137f,
          0.822821f, 0.589912f,   0.837467f, 0.581138f,   0.847648f, 0.571591f,
          0.855499f, 0.562043f,   0.863153f, 0.553270f,   0.872744f, 0.546045f,
          0.886405f, 0.541142f,   0.906271f, 0.539335f,   0.730760f, 0.579121f,
          0.769430f, 0.577745f,   0.797985f, 0.574009f,   0.818373f, 0.568504f,
          0.832539f, 0.561820f,   0.842428f, 0.554545f,   0.849986f, 0.547271f,
          0.857160f, 0.540586f,   0.865895f, 0.535082f,   0.878136f, 0.531346f,
          0.895830f, 0.529970f,   0.730760f, 0.544505f,   0.767956f, 0.543731f,
          0.795548f, 0.541630f,   0.815355f, 0.538534f,   0.829194f, 0.534773f,
          0.838885f, 0.530682f,   0.846245f, 0.526590f,   0.853093f, 0.522830f,
          0.861247f, 0.519733f,   0.872525f, 0.517632f,   0.888745f, 0.516858f,
          0.893500f, 0.483330f,   0.890765f, 0.500000f,   0.898000f, 0.483843f,
          0.895208f, 0.500000f,   0.902065f, 0.484598f,   0.899279f, 0.500000f,
          0.905518f, 0.485515f,   0.902791f, 0.500000f,   0.908180f, 0.486513f,
          0.905556f, 0.500000f,   0.909875f, 0.487511f,   0.907390f, 0.500000f,
          0.910423f, 0.488428f,   0.908105f, 0.500000f,   0.909648f, 0.489184f,
          0.907514f, 0.500000f,   0.907370f, 0.489696f,   0.905432f, 0.500000f,
          0.903412f, 0.489885f,   0.901672f, 0.500000f,   0.900926f, 0.470365f,
          0.905576f, 0.471277f,   0.909625f, 0.472619f,   0.912919f, 0.474250f,
          0.915302f, 0.476024f,   0.916620f, 0.477798f,   0.916717f, 0.479428f,
          0.915438f, 0.480771f,   0.912630f, 0.481682f,   0.908136f, 0.482018f,
          0.911869f, 0.461105f,   0.916741f, 0.462301f,   0.920767f, 0.464063f,
          0.923827f, 0.466203f,   0.925798f, 0.468531f,   0.926560f, 0.470860f,
          0.925992f, 0.473000f,   0.923972f, 0.474762f,   0.920381f, 0.475958f,
          0.915096f, 0.476398f,   0.925156f, 0.455548f,   0.930299f, 0.456915f,
          0.934297f, 0.458929f,   0.937072f, 0.461374f,   0.938542f, 0.464036f,
          0.938630f, 0.466697f,   0.937254f, 0.469143f,   0.934335f, 0.471157f,
          0.929793f, 0.472523f,   0.923549f, 0.473027f,   0.939616f, 0.453696f,
          0.945053f, 0.455120f,   0.949021f, 0.457218f,   0.951485f, 0.459765f,
          0.952412f, 0.462537f,   0.951765f, 0.465310f,   0.949510f, 0.467857f,
          0.945612f, 0.469955f,   0.940036f, 0.471378f,   0.932747f, 0.471903f,
          0.954077f, 0.455548f,   0.959807f, 0.456915f,   0.963744f, 0.458929f,
          0.965899f, 0.461374f,   0.966281f, 0.464036f,   0.964900f, 0.466697f,
          0.961766f, 0.469143f,   0.956889f, 0.471157f,   0.950279f, 0.472523f,
          0.941945f, 0.473027f,   0.967364f, 0.461105f,   0.973364f, 0.462301f,
          0.977274f, 0.464063f,   0.979144f, 0.466203f,   0.979025f, 0.468531f,
          0.976970f, 0.470860f,   0.973028f, 0.473000f,   0.967252f, 0.474762f,
          0.959691f, 0.475958f,   0.950397f, 0.476398f,   0.978307f, 0.470365f,
          0.984529f, 0.471277f,   0.988416f, 0.472619f,   0.990051f, 0.474250f,
          0.989521f, 0.476024f,   0.986910f, 0.477798f,   0.982303f, 0.479428f,
          0.975785f, 0.480771f,   0.967442f, 0.481682f,   0.957358f, 0.482018f,
          0.985732f, 0.483330f,   0.992105f, 0.483843f,   0.995976f, 0.484598f,
          0.997453f, 0.485515f,   0.996643f, 0.486513f,   0.993655f, 0.487511f,
          0.988597f, 0.488428f,   0.981576f, 0.489184f,   0.972702f, 0.489696f,
          0.962082f, 0.489885f,   0.988468f, 0.500000f,   0.994897f, 0.500000f,
          0.998762f, 0.500000f,   1.000180f, 0.500000f,   0.999267f, 0.500000f,
          0.996140f, 0.500000f,   0.990915f, 0.500000f,   0.983710f, 0.500000f,
          0.974640f, 0.500000f,   0.963822f, 0.500000f,   0.985732f, 0.516669f,
          0.992105f, 0.516157f,   0.995976f, 0.515401f,   0.997453f, 0.514484f,
          0.996643f, 0.513486f,   0.993655f, 0.512488f,   0.988597f, 0.511571f,
          0.981576f, 0.510816f,   0.972702f, 0.510303f,   0.962082f, 0.510115f,
          0.978307f, 0.529634f,   0.984529f, 0.528723f,   0.988416f, 0.527380f,
          0.990051f, 0.525750f,   0.989521f, 0.523976f,   0.986910f, 0.522202f,
          0.982303f, 0.520571f,   0.975785f, 0.519229f,   0.967442f, 0.518317f,
          0.957358f, 0.517982f,   0.967364f, 0.538895f,   0.973364f, 0.537699f,
          0.977274f, 0.535937f,   0.979144f, 0.533797f,   0.979025f, 0.531468f,
          0.976970f, 0.529140f,   0.973028f, 0.527000f,   0.967252f, 0.525238f,
          0.959691f, 0.524042f,   0.950397f, 0.523601f,   0.954077f, 0.544451f,
          0.959806f, 0.543085f,   0.963744f, 0.541071f,   0.965899f, 0.538625f,
          0.966281f, 0.535964f,   0.964900f, 0.533302f,   0.961766f, 0.530857f,
          0.956889f, 0.528843f,   0.950279f, 0.527476f,   0.941945f, 0.526973f,
          0.939616f, 0.546304f,   0.945053f, 0.544880f,   0.949021f, 0.542782f,
          0.951485f, 0.540235f,   0.952412f, 0.537462f,   0.951765f, 0.534690f,
          0.949510f, 0.532143f,   0.945612f, 0.530045f,   0.940036f, 0.528621f,
          0.932747f, 0.528097f,   0.925156f, 0.544451f,   0.930299f, 0.543085f,
          0.934297f, 0.541071f,   0.937072f, 0.538625f,   0.938542f, 0.535964f,
          0.938630f, 0.533303f,   0.937254f, 0.530857f,   0.934335f, 0.528843f,
          0.929793f, 0.527476f,   0.923549f, 0.526973f,   0.911869f, 0.538895f,
          0.916741f, 0.537699f,   0.920768f, 0.535937f,   0.923827f, 0.533797f,
          0.925798f, 0.531468f,   0.926560f, 0.529140f,   0.925992f, 0.527000f,
          0.923972f, 0.525238f,   0.920381f, 0.524042f,   0.915096f, 0.523601f,
          0.900926f, 0.529634f,   0.905576f, 0.528723f,   0.909625f, 0.527380f,
          0.912919f, 0.525750f,   0.915302f, 0.523976f,   0.916620f, 0.522202f,
          0.916717f, 0.520571f,   0.915438f, 0.519229f,   0.912630f, 0.518318f,
          0.908136f, 0.517982f,   0.893500f, 0.516669f,   0.898000f, 0.516157f,
          0.902065f, 0.515401f,   0.905518f, 0.514484f,   0.908180f, 0.513486f,
          0.909875f, 0.512488f,   0.910423f, 0.511571f,   0.909648f, 0.510816f,
          0.907370f, 0.510303f,   0.903412f, 0.510115f,   0.496472f, 0.491996f,
          0.496859f, 0.500000f,   0.466623f, 0.500000f,   0.513988f, 0.487299f,
          0.514603f, 0.500000f,   0.521565f, 0.485268f,   0.522278f, 0.500000f,
          0.521596f, 0.485261f,   0.522309f, 0.500000f,   0.516473f, 0.486636f,
          0.517120f, 0.500000f,   0.508589f, 0.488753f,   0.509134f, 0.500000f,
          0.500336f, 0.490970f,   0.500775f, 0.500000f,   0.494109f, 0.492645f,
          0.494466f, 0.500000f,   0.492298f, 0.493137f,   0.492633f, 0.500000f,
          0.497298f, 0.491804f,   0.497698f, 0.500000f,   0.495347f, 0.484448f,
          0.512204f, 0.475321f,   0.519495f, 0.471374f,   0.519525f, 0.471360f,
          0.514594f, 0.474032f,   0.507007f, 0.478144f,   0.499064f, 0.482450f,
          0.493070f, 0.485703f,   0.491327f, 0.486656f,   0.496137f, 0.484064f,
          0.493543f, 0.477446f,   0.509340f, 0.464211f,   0.516173f, 0.458486f,
          0.516200f, 0.458465f,   0.511579f, 0.462339f,   0.504468f, 0.468302f,
          0.497024f, 0.474544f,   0.491405f, 0.479259f,   0.489769f, 0.480639f,
          0.494275f, 0.476875f,   0.491114f, 0.471082f,   0.505487f, 0.454112f,
          0.511703f, 0.446772f,   0.511728f, 0.446745f,   0.507523f, 0.451711f,
          0.501052f, 0.459354f,   0.494278f, 0.467356f,   0.489165f, 0.473398f,
          0.487675f, 0.475163f,   0.491772f, 0.470334f,   0.488119f, 0.465447f,
          0.500734f, 0.445171f,   0.506190f, 0.436400f,   0.506211f, 0.436367f,
          0.502520f, 0.442300f,   0.496839f, 0.451431f,   0.490893f, 0.460989f,
          0.486403f, 0.468206f,   0.485093f, 0.470311f,   0.488686f, 0.464535f,
          0.484614f, 0.460633f,   0.495171f, 0.437531f,   0.499737f, 0.427538f,
          0.499754f, 0.427499f,   0.496665f, 0.434258f,   0.491910f, 0.444659f,
          0.486932f, 0.455547f,   0.483173f, 0.463767f,   0.482075f, 0.466162f,
          0.485079f, 0.459576f,   0.480655f, 0.456730f,   0.488889f, 0.431337f,
          0.492450f, 0.420353f,   0.492463f, 0.420309f,   0.490053f, 0.427737f,
          0.486343f, 0.439169f,   0.482460f, 0.451134f,   0.479527f, 0.460166f,
          0.478668f, 0.462795f,   0.481010f, 0.455552f,   0.476299f, 0.453829f,
          0.481976f, 0.426733f,   0.484432f, 0.415013f,   0.484441f, 0.414966f,
          0.482778f, 0.422891f,   0.480220f, 0.435088f,   0.477541f, 0.447854f,
          0.475518f, 0.457489f,   0.474925f, 0.460291f,   0.476537f, 0.452559f,
          0.471603f, 0.452021f,   0.474525f, 0.423865f,   0.475788f, 0.411686f,
          0.475793f, 0.411637f,   0.474937f, 0.419872f,   0.473620f, 0.432545f,
          0.472241f, 0.445809f,   0.471199f, 0.455820f,   0.470893f, 0.458730f,
          0.471722f, 0.450693f,   0.466623f, 0.451399f,   0.466623f, 0.422877f,
          0.466623f, 0.410539f,   0.466623f, 0.410489f,   0.466623f, 0.418831f,
          0.466623f, 0.431668f,   0.466623f, 0.445105f,   0.466623f, 0.455245f,
          0.466623f, 0.458192f,   0.466623f, 0.450050f,   0.461644f, 0.452021f,
          0.458722f, 0.423865f,   0.457458f, 0.411686f,   0.457454f, 0.411637f,
          0.458309f, 0.419872f,   0.459626f, 0.432545f,   0.461005f, 0.445809f,
          0.462047f, 0.455820f,   0.462354f, 0.458730f,   0.461524f, 0.450693f,
          0.456948f, 0.453829f,   0.451270f, 0.426733f,   0.448814f, 0.415013f,
          0.448806f, 0.414966f,   0.450468f, 0.422891f,   0.453026f, 0.435088f,
          0.455705f, 0.447854f,   0.457729f, 0.457489f,   0.458322f, 0.460291f,
          0.456709f, 0.452559f,   0.452592f, 0.456730f,   0.444358f, 0.431337f,
          0.440797f, 0.420353f,   0.440784f, 0.420309f,   0.443194f, 0.427737f,
          0.446903f, 0.439169f,   0.450787f, 0.451134f,   0.453720f, 0.460166f,
          0.454578f, 0.462795f,   0.452237f, 0.455552f,   0.448633f, 0.460633f,
          0.438076f, 0.437531f,   0.433509f, 0.427538f,   0.433492f, 0.427499f,
          0.436582f, 0.434258f,   0.441337f, 0.444659f,   0.446315f, 0.455547f,
          0.450074f, 0.463767f,   0.451172f, 0.466162f,   0.448167f, 0.459576f,
          0.445127f, 0.465447f,   0.432513f, 0.445171f,   0.427057f, 0.436400f,
          0.427036f, 0.436367f,   0.430727f, 0.442300f,   0.436407f, 0.451431f,
          0.442354f, 0.460989f,   0.446843f, 0.468206f,   0.448153f, 0.470311f,
          0.444560f, 0.464535f,   0.442132f, 0.471082f,   0.427760f, 0.454112f,
          0.421543f, 0.446772f,   0.421519f, 0.446745f,   0.425724f, 0.451711f,
          0.432195f, 0.459354f,   0.438968f, 0.467356f,   0.444082f, 0.473398f,
          0.445572f, 0.475163f,   0.441475f, 0.470334f,   0.439704f, 0.477446f,
          0.423906f, 0.464211f,   0.417073f, 0.458486f,   0.417046f, 0.458465f,
          0.421667f, 0.462340f,   0.428779f, 0.468302f,   0.436223f, 0.474544f,
          0.441842f, 0.479259f,   0.443477f, 0.480639f,   0.438972f, 0.476875f,
          0.437899f, 0.484448f,   0.421043f, 0.475321f,   0.413751f, 0.471374f,
          0.413722f, 0.471360f,   0.418652f, 0.474032f,   0.426240f, 0.478144f,
          0.434182f, 0.482450f,   0.440176f, 0.485703f,   0.441919f, 0.486656f,
          0.437110f, 0.484064f,   0.436775f, 0.491996f,   0.419258f, 0.487299f,
          0.411681f, 0.485268f,   0.411651f, 0.485261f,   0.416774f, 0.486636f,
          0.424658f, 0.488753f,   0.432910f, 0.490970f,   0.439138f, 0.492645f,
          0.440948f, 0.493137f,   0.435949f, 0.491804f,   0.436387f, 0.500000f,
          0.418644f, 0.500000f,   0.410968f, 0.500000f,   0.410937f, 0.500000f,
          0.416126f, 0.500000f,   0.424113f, 0.500000f,   0.432472f, 0.500000f,
          0.438780f, 0.500000f,   0.440614f, 0.500000f,   0.435548f, 0.500000f,
          0.436775f, 0.508004f,   0.419258f, 0.512701f,   0.411681f, 0.514732f,
          0.411651f, 0.514739f,   0.416774f, 0.513364f,   0.424658f, 0.511247f,
          0.432910f, 0.509030f,   0.439138f, 0.507355f,   0.440948f, 0.506863f,
          0.435949f, 0.508196f,   0.437899f, 0.515552f,   0.421043f, 0.524679f,
          0.413751f, 0.528626f,   0.413722f, 0.528640f,   0.418652f, 0.525968f,
          0.426240f, 0.521856f,   0.434182f, 0.517550f,   0.440176f, 0.514297f,
          0.441919f, 0.513344f,   0.437110f, 0.515936f,   0.439704f, 0.522554f,
          0.423906f, 0.535789f,   0.417073f, 0.541514f,   0.417046f, 0.541535f,
          0.421667f, 0.537661f,   0.428779f, 0.531698f,   0.436223f, 0.525456f,
          0.441842f, 0.520741f,   0.443477f, 0.519361f,   0.438972f, 0.523125f,
          0.442132f, 0.528918f,   0.427760f, 0.545888f,   0.421543f, 0.553228f,
          0.421519f, 0.553255f,   0.425724f, 0.548289f,   0.432195f, 0.540646f,
          0.438968f, 0.532644f,   0.444082f, 0.526602f,   0.445572f, 0.524837f,
          0.441475f, 0.529666f,   0.445127f, 0.534553f,   0.432513f, 0.554829f,
          0.427057f, 0.563600f,   0.427036f, 0.563633f,   0.430727f, 0.557700f,
          0.436407f, 0.548569f,   0.442354f, 0.539011f,   0.446843f, 0.531794f,
          0.448153f, 0.529689f,   0.444560f, 0.535465f,   0.448633f, 0.539367f,
          0.438076f, 0.562469f,   0.433509f, 0.572462f,   0.433492f, 0.572501f,
          0.436582f, 0.565742f,   0.441337f, 0.555341f,   0.446315f, 0.544453f,
          0.450074f, 0.536233f,   0.451172f, 0.533838f,   0.448167f, 0.540424f,
          0.452592f, 0.543270f,   0.444358f, 0.568663f,   0.440797f, 0.579647f,
          0.440784f, 0.579691f,   0.443194f, 0.572263f,   0.446903f, 0.560831f,
          0.450787f, 0.548866f,   0.453720f, 0.539834f,   0.454578f, 0.537205f,
          0.452237f, 0.544448f,   0.456948f, 0.546171f,   0.451270f, 0.573267f,
          0.448814f, 0.584987f,   0.448806f, 0.585034f,   0.450468f, 0.577109f,
          0.453026f, 0.564912f,   0.455705f, 0.552146f,   0.457729f, 0.542511f,
          0.458322f, 0.539709f,   0.456709f, 0.547441f,   0.461644f, 0.547979f,
          0.458722f, 0.576135f,   0.457458f, 0.588314f,   0.457454f, 0.588363f,
          0.458309f, 0.580128f,   0.459626f, 0.567455f,   0.461005f, 0.554191f,
          0.462048f, 0.544180f,   0.462354f, 0.541270f,   0.461524f, 0.549307f,
          0.466623f, 0.548601f,   0.466623f, 0.577123f,   0.466623f, 0.589461f,
          0.466623f, 0.589511f,   0.466623f, 0.581169f,   0.466623f, 0.568332f,
          0.466623f, 0.554895f,   0.466623f, 0.544755f,   0.466623f, 0.541808f,
          0.466623f, 0.549950f,   0.471603f, 0.547979f,   0.474525f, 0.576135f,
          0.475788f, 0.588314f,   0.475793f, 0.588363f,   0.474937f, 0.580128f,
          0.473620f, 0.567455f,   0.472241f, 0.554191f,   0.471199f, 0.544180f,
          0.470893f, 0.541270f,   0.471722f, 0.549307f,   0.476299f, 0.546171f,
          0.481977f, 0.573267f,   0.484432f, 0.584987f,   0.484441f, 0.585034f,
          0.482778f, 0.577109f,   0.480220f, 0.564912f,   0.477541f, 0.552146f,
          0.475518f, 0.542511f,   0.474925f, 0.539709f,   0.476537f, 0.547441f,
          0.480655f, 0.543270f,   0.488889f, 0.568663f,   0.492450f, 0.579647f,
          0.492463f, 0.579691f,   0.490053f, 0.572263f,   0.486343f, 0.560831f,
          0.482460f, 0.548866f,   0.479527f, 0.539834f,   0.478668f, 0.537205f,
          0.481010f, 0.544448f,   0.484614f, 0.539367f,   0.495171f, 0.562469f,
          0.499737f, 0.572462f,   0.499754f, 0.572501f,   0.496665f, 0.565742f,
          0.491910f, 0.555341f,   0.486932f, 0.544453f,   0.483173f, 0.536233f,
          0.482075f, 0.533838f,   0.485079f, 0.540424f,   0.488119f, 0.534553f,
          0.500734f, 0.554829f,   0.506190f, 0.563600f,   0.506211f, 0.563633f,
          0.502520f, 0.557700f,   0.496839f, 0.548569f,   0.490893f, 0.539011f,
          0.486403f, 0.531794f,   0.485093f, 0.529689f,   0.488686f, 0.535465f,
          0.491114f, 0.528918f,   0.505487f, 0.545888f,   0.511703f, 0.553228f,
          0.511728f, 0.553255f,   0.507523f, 0.548289f,   0.501052f, 0.540646f,
          0.494278f, 0.532644f,   0.489165f, 0.526602f,   0.487675f, 0.524837f,
          0.491772f, 0.529666f,   0.493543f, 0.522554f,   0.509340f, 0.535789f,
          0.516173f, 0.541514f,   0.516200f, 0.541535f,   0.511579f, 0.537660f,
          0.504468f, 0.531698f,   0.497024f, 0.525456f,   0.491405f, 0.520741f,
          0.489769f, 0.519361f,   0.494275f, 0.523125f,   0.495347f, 0.515552f,
          0.512204f, 0.524679f,   0.519495f, 0.528626f,   0.519525f, 0.528640f,
          0.514594f, 0.525968f,   0.507007f, 0.521856f,   0.499064f, 0.517550f,
          0.493070f, 0.514297f,   0.491327f, 0.513344f,   0.496137f, 0.515936f,
          0.496472f, 0.508004f,   0.513988f, 0.512701f,   0.521565f, 0.514732f,
          0.521596f, 0.514739f,   0.516473f, 0.513364f,   0.508589f, 0.511247f,
          0.500336f, 0.509030f,   0.494109f, 0.507355f,   0.492298f, 0.506863f,
          0.497298f, 0.508196f,   0.509476f, 0.488550f,   0.510035f, 0.500000f,
          0.526623f, 0.483969f,   0.527406f, 0.500000f,   0.547267f, 0.478453f,
          0.548319f, 0.500000f,   0.569936f, 0.472396f,   0.571284f, 0.500000f,
          0.593156f, 0.466192f,   0.594807f, 0.500000f,   0.615457f, 0.460234f,
          0.617399f, 0.500000f,   0.635365f, 0.454915f,   0.637566f, 0.500000f,
          0.651407f, 0.450628f,   0.653819f, 0.500000f,   0.662113f, 0.447768f,
          0.664664f, 0.500000f,   0.666009f, 0.446727f,   0.668610f, 0.500000f,
          0.507854f, 0.477737f,   0.524352f, 0.468829f,   0.544215f, 0.458104f,
          0.566025f, 0.446327f,   0.588367f, 0.434264f,   0.609824f, 0.422678f,
          0.628978f, 0.412336f,   0.644414f, 0.404001f,   0.654714f, 0.398439f,
          0.658462f, 0.396415f,   0.505253f, 0.467695f,   0.520710f, 0.454768f,
          0.539320f, 0.439205f,   0.559754f, 0.422115f,   0.580687f, 0.404610f,
          0.600789f, 0.387798f,   0.618735f, 0.372790f,   0.633197f, 0.360696f,
          0.642848f, 0.352625f,   0.646359f, 0.349688f,   0.501755f, 0.458556f,
          0.515813f, 0.441973f,   0.532738f, 0.422007f,   0.551323f, 0.400084f,
          0.570360f, 0.377626f,   0.588643f, 0.356059f,   0.604964f, 0.336805f,
          0.618117f, 0.321290f,   0.626893f, 0.310936f,   0.630087f, 0.307169f,
          0.497446f, 0.450456f,   0.509779f, 0.430631f,   0.524627f, 0.406764f,
          0.540932f, 0.380555f,   0.557634f, 0.353709f,   0.573674f, 0.327926f,
          0.587993f, 0.304910f,   0.599532f, 0.286362f,   0.607232f, 0.273984f,
          0.610034f, 0.269480f,   0.492406f, 0.443528f,   0.502723f, 0.420931f,
          0.515144f, 0.393726f,   0.528783f, 0.363853f,   0.542754f, 0.333253f,
          0.556172f, 0.303865f,   0.568150f, 0.277630f,   0.577802f, 0.256488f,
          0.584243f, 0.242380f,   0.586587f, 0.237247f,   0.486721f, 0.437907f,
          0.494763f, 0.413061f,   0.504445f, 0.383147f,   0.515077f, 0.350301f,
          0.525967f, 0.316654f,   0.536426f, 0.284340f,   0.545763f, 0.255494f,
          0.553287f, 0.232248f,   0.558308f, 0.216736f,   0.560135f, 0.211091f,
          0.480473f, 0.433726f,   0.486015f, 0.407206f,   0.492687f, 0.375279f,
          0.500014f, 0.340220f,   0.507519f, 0.304308f,   0.514727f, 0.269818f,
          0.521161f, 0.239029f,   0.526346f, 0.214218f,   0.529806f, 0.197661f,
          0.531065f, 0.191636f,   0.473746f, 0.431119f,   0.476596f, 0.403556f,
          0.480028f, 0.370373f,   0.483796f, 0.333935f,   0.487656f, 0.296610f,
          0.491362f, 0.260764f,   0.494672f, 0.228764f,   0.497338f, 0.202976f,
          0.499118f, 0.185768f,   0.499765f, 0.179507f,   0.466623f, 0.430220f,
          0.466623f, 0.402298f,   0.466623f, 0.368681f,   0.466623f, 0.331768f,
          0.466623f, 0.293956f,   0.466623f, 0.257642f,   0.466623f, 0.225225f,
          0.466623f, 0.199101f,   0.466623f, 0.181668f,   0.466623f, 0.175325f,
          0.459500f, 0.431119f,   0.456650f, 0.403556f,   0.453219f, 0.370373f,
          0.449451f, 0.333935f,   0.445591f, 0.296610f,   0.441884f, 0.260764f,
          0.438575f, 0.228764f,   0.435908f, 0.202976f,   0.434129f, 0.185768f,
          0.433481f, 0.179507f,   0.452773f, 0.433726f,   0.447231f, 0.407206f,
          0.440559f, 0.375279f,   0.433232f, 0.340220f,   0.425727f, 0.304308f,
          0.418520f, 0.269818f,   0.412086f, 0.239030f,   0.406900f, 0.214218f,
          0.403440f, 0.197661f,   0.402181f, 0.191636f,   0.446525f, 0.437907f,
          0.438483f, 0.413061f,   0.428801f, 0.383147f,   0.418170f, 0.350301f,
          0.407279f, 0.316654f,   0.396820f, 0.284341f,   0.387483f, 0.255494f,
          0.379959f, 0.232248f,   0.374938f, 0.216736f,   0.373111f, 0.211091f,
          0.440840f, 0.443528f,   0.430523f, 0.420932f,   0.418102f, 0.393726f,
          0.404463f, 0.363853f,   0.390492f, 0.333253f,   0.377075f, 0.303865f,
          0.365097f, 0.277630f,   0.355444f, 0.256488f,   0.349003f, 0.242381f,
          0.346659f, 0.237247f,   0.435801f, 0.450456f,   0.423468f, 0.430631f,
          0.408619f, 0.406764f,   0.392314f, 0.380555f,   0.375613f, 0.353709f,
          0.359573f, 0.327926f,   0.345254f, 0.304910f,   0.333715f, 0.286362f,
          0.326015f, 0.273985f,   0.323213f, 0.269481f,   0.431491f, 0.458556f,
          0.417433f, 0.441973f,   0.400508f, 0.422007f,   0.381924f, 0.400084f,
          0.362886f, 0.377626f,   0.344604f, 0.356059f,   0.328282f, 0.336806f,
          0.315130f, 0.321290f,   0.306353f, 0.310937f,   0.303159f, 0.307169f,
          0.427994f, 0.467695f,   0.412537f, 0.454768f,   0.393927f, 0.439205f,
          0.373492f, 0.422115f,   0.352560f, 0.404610f,   0.332457f, 0.387798f,
          0.314511f, 0.372790f,   0.300049f, 0.360696f,   0.290399f, 0.352625f,
          0.286887f, 0.349688f,   0.425393f, 0.477737f,   0.408894f, 0.468829f,
          0.389032f, 0.458104f,   0.367221f, 0.446327f,   0.344879f, 0.434264f,
          0.323423f, 0.422678f,   0.304268f, 0.412336f,   0.288833f, 0.404001f,
          0.278532f, 0.398440f,   0.274784f, 0.396416f,   0.423771f, 0.488550f,
          0.406624f, 0.483969f,   0.385980f, 0.478453f,   0.363311f, 0.472397f,
          0.340090f, 0.466192f,   0.317790f, 0.460234f,   0.297882f, 0.454915f,
          0.281839f, 0.450629f,   0.271134f, 0.447768f,   0.267238f, 0.446727f,
          0.423212f, 0.500000f,   0.405841f, 0.500000f,   0.384927f, 0.500000f,
          0.361963f, 0.500000f,   0.338439f, 0.500000f,   0.315848f, 0.500000f,
          0.295680f, 0.500000f,   0.279428f, 0.500000f,   0.268583f, 0.500000f,
          0.264636f, 0.500000f,   0.423771f, 0.511450f,   0.406624f, 0.516031f,
          0.385980f, 0.521547f,   0.363311f, 0.527604f,   0.340090f, 0.533808f,
          0.317790f, 0.539766f,   0.297882f, 0.545085f,   0.281839f, 0.549372f,
          0.271134f, 0.552232f,   0.267238f, 0.553273f,   0.425393f, 0.522263f,
          0.408894f, 0.531171f,   0.389032f, 0.541896f,   0.367221f, 0.553673f,
          0.344879f, 0.565736f,   0.323423f, 0.577322f,   0.304268f, 0.587664f,
          0.288833f, 0.595999f,   0.278532f, 0.601561f,   0.274784f, 0.603584f,
          0.427994f, 0.532305f,   0.412537f, 0.545232f,   0.393927f, 0.560795f,
          0.373492f, 0.577885f,   0.352560f, 0.595390f,   0.332457f, 0.612202f,
          0.314511f, 0.627210f,   0.300049f, 0.639304f,   0.290399f, 0.647375f,
          0.286887f, 0.650312f,   0.431491f, 0.541444f,   0.417433f, 0.558027f,
          0.400508f, 0.577993f,   0.381924f, 0.599916f,   0.362886f, 0.622374f,
          0.344604f, 0.643941f,   0.328282f, 0.663195f,   0.315130f, 0.678710f,
          0.306353f, 0.689064f,   0.303159f, 0.692831f,   0.435801f, 0.549544f,
          0.423468f, 0.569369f,   0.408619f, 0.593236f,   0.392314f, 0.619445f,
          0.375613f, 0.646291f,   0.359573f, 0.672074f,   0.345254f, 0.695090f,
          0.333715f, 0.713638f,   0.326015f, 0.726016f,   0.323213f, 0.730520f,
          0.440840f, 0.556472f,   0.430523f, 0.579069f,   0.418102f, 0.606274f,
          0.404463f, 0.636147f,   0.390492f, 0.666747f,   0.377075f, 0.696135f,
          0.365097f, 0.722370f,   0.355444f, 0.743512f,   0.349003f, 0.757620f,
          0.346659f, 0.762753f,   0.446525f, 0.562093f,   0.438483f, 0.586939f,
          0.428801f, 0.616853f,   0.418170f, 0.649699f,   0.407279f, 0.683346f,
          0.396820f, 0.715660f,   0.387483f, 0.744506f,   0.379959f, 0.767752f,
          0.374938f, 0.783264f,   0.373111f, 0.788909f,   0.452773f, 0.566274f,
          0.447231f, 0.592794f,   0.440559f, 0.624721f,   0.433232f, 0.659780f,
          0.425727f, 0.695692f,   0.418520f, 0.730182f,   0.412086f, 0.760971f,
          0.406901f, 0.785782f,   0.403440f, 0.802339f,   0.402181f, 0.808364f,
          0.459500f, 0.568881f,   0.456650f, 0.596444f,   0.453219f, 0.629627f,
          0.449451f, 0.666065f,   0.445591f, 0.703390f,   0.441884f, 0.739236f,
          0.438575f, 0.771236f,   0.435908f, 0.797024f,   0.434129f, 0.814232f,
          0.433481f, 0.820493f,   0.466623f, 0.569780f,   0.466623f, 0.597702f,
          0.466623f, 0.631319f,   0.466623f, 0.668232f,   0.466623f, 0.706044f,
          0.466623f, 0.742358f,   0.466623f, 0.774775f,   0.466623f, 0.800899f,
          0.466623f, 0.818332f,   0.466623f, 0.824675f,   0.473746f, 0.568881f,
          0.476596f, 0.596444f,   0.480028f, 0.629627f,   0.483796f, 0.666065f,
          0.487656f, 0.703390f,   0.491363f, 0.739236f,   0.494672f, 0.771236f,
          0.497338f, 0.797024f,   0.499118f, 0.814232f,   0.499765f, 0.820493f,
          0.480473f, 0.566274f,   0.486015f, 0.592794f,   0.492688f, 0.624721f,
          0.500014f, 0.659780f,   0.507519f, 0.695692f,   0.514727f, 0.730182f,
          0.521161f, 0.760970f,   0.526346f, 0.785782f,   0.529806f, 0.802339f,
          0.531065f, 0.808364f,   0.486721f, 0.562093f,   0.494763f, 0.586939f,
          0.504445f, 0.616853f,   0.515077f, 0.649699f,   0.525967f, 0.683346f,
          0.536426f, 0.715659f,   0.545763f, 0.744506f,   0.553287f, 0.767752f,
          0.558308f, 0.783264f,   0.560135f, 0.788909f,   0.492406f, 0.556472f,
          0.502723f, 0.579068f,   0.515144f, 0.606274f,   0.528783f, 0.636147f,
          0.542754f, 0.666747f,   0.556172f, 0.696135f,   0.568150f, 0.722370f,
          0.577802f, 0.743512f,   0.584243f, 0.757619f,   0.586587f, 0.762753f,
          0.497446f, 0.549544f,   0.509779f, 0.569369f,   0.524627f, 0.593236f,
          0.540932f, 0.619445f,   0.557634f, 0.646291f,   0.573674f, 0.672074f,
          0.587993f, 0.695090f,   0.599532f, 0.713638f,   0.607232f, 0.726015f,
          0.610034f, 0.730519f,   0.501755f, 0.541444f,   0.515813f, 0.558027f,
          0.532738f, 0.577993f,   0.551323f, 0.599916f,   0.570360f, 0.622374f,
          0.588643f, 0.643941f,   0.604964f, 0.663194f,   0.618117f, 0.678710f,
          0.626893f, 0.689063f,   0.630087f, 0.692831f,   0.505253f, 0.532305f,
          0.520710f, 0.545232f,   0.539320f, 0.560795f,   0.559754f, 0.577885f,
          0.580687f, 0.595390f,   0.600789f, 0.612202f,   0.618735f, 0.627210f,
          0.633197f, 0.639304f,   0.642848f, 0.647375f,   0.646359f, 0.650312f,
          0.507854f, 0.522263f,   0.524352f, 0.531171f,   0.544215f, 0.541896f,
          0.566025f, 0.553673f,   0.588367f, 0.565736f,   0.609824f, 0.577322f,
          0.628978f, 0.587664f,   0.644414f, 0.595999f,   0.654714f, 0.601560f,
          0.658462f, 0.603584f,   0.509476f, 0.511450f,   0.526623f, 0.516031f,
          0.547267f, 0.521547f,   0.569936f, 0.527603f,   0.593156f, 0.533808f,
          0.615457f, 0.539766f,   0.635365f, 0.545085f,   0.651407f, 0.549371f,
          0.662113f, 0.552232f,   0.666009f, 0.553273f
    };
    static const unsigned int teapot_indices[] = {
           0,    1,    2,    2,    3,    0,    3,    2,    4,    4,    5,    3,
           5,    4,    6,    6,    7,    5,    7,    6,    8,    8,    9,    7,
           9,    8,   10,   10,   11,    9,   11,   10,   12,   12,   13,   11,
          13,   12,   14,   14,   15,   13,   15,   14,   16,   16,   17,   15,
          17,   16,   18,   18,   19,   17,   19,   18,   20,   20,   21,   19,
           1,   22,   23,   23,    2,    1,    2,   23,   24,   24,    4,    2,
           4,   24,   25,   25,    6,    4,    6,   25,   26,   26,    8,    6,
           8,   26,   27,   27,   10,    8,   10,   27,   28,   28,   12,   10,
          12,   28,   29,   29,   14,   12,   14,   29,   30,   30,   16,   14,
          16,   30,   31,   31,   18,   16,   18,   31,   32,   32,   20,   18,
          22,   33,   34,   34,   23,   22,   23,   34,   35,   35,   24,   23,
          24,   35,   36,   36,   25,   24,   25,   36,   37,   37,   26,   25,
          26,   37,   38,   38,   27,   26,   27,   38,   39,   39,   28,   27,
          28,   39,   40,   40,   29,   28,   29,   40,   41,   41,   30,   29,
          30,   41,   42,   42,   31,   30,   31,   42,   43,   43,   32,   31,
          33,   44,   45,   45,   34,   33,   34,   45,   46,   46,   35,   34,
          35,   46,   47,   47,   36,   35,   36,   47,   48,   48,   37,   36,
          37,   48,   49,   49,   38,   37,   38,   49,   50,   50,   39,   38,
          39,   50,   51,   51,   40,   39,   40,   51,   52,   52,   41,   40,
          41,   52,   53,   53,   42,   41,   42,   53,   54,   54,   43,   42,
          44,   55,   56,   56,   45,   44,   45,   56,   57,   57,   46,   45,
          46,   57,   58,   58,   47,   46,   47,   58,   59,   59,   48,   47,
          48,   59,   60,   60,   49,   48,   49,   60,   61,   61,   50,   49,
          50,   61,   62,   62,   51,   50,   51,   62,   63,   63,   52,   51,
          52,   63,   64,   64,   53,   52,   53,   64,   65,   65,   54,   53,
          55,   66,   67,   67,   56,   55,   56,   67,   68,   68,   57,   56,
          57,   68,   69,   69,   58,   57,   58,   69,   70,   70,   59,   58,
          59,   70,   71,   71,   60,   59,   60,   71,   72,   72,   61,   60,
          61,   72,   73,   73,   62,   61,   62,   73,   74,   74,   63,   62,
          63,   74,   75,   75,   64,   63,   64,   75,   76,   76,   65,   64,
          66,   77,   78,   78,   67,   66,   67,   78,   79,   79,   68,   67,
          68,   79,   80,   80,   69,   68,   69,   80,   81,   81,   70,   69,
          70,   81,   82,   82,   71,   70,   71,   82,   83,   83,   72,   71,
          72,   83,   84,   84,   73,   72,   73,   84,   85,   85,   74,   73,
          74,   85,   86,   86,   75,   74,   75,   86,   87,   87,   76,   75,
          77,   88,   89,   89,   78,   77,   78,   89,   90,   90,   79,   78,
          79,   90,   91,   91,   80,   79,   80,   91,   92,   92,   81,   80,
          81,   92,   93,   93,   82,   81,   82,   93,   94,   94,   83,   82,
          83,   94,   95,   95,   84,   83,   84,   95,   96,   96,   85,   84,
          85,   96,   97,   97,   86,   85,   86,   97,   98,   98,   87,   86,
          88,   99,  100,  100,   89,   88,   89,  100,  101,  101,   90,   89,
          90,  101,  102,  102,   91,   90,   91,  102,  103,  103,   92,   91,
          92,  103,  104,  104,   93,   92,   93,  104,  105,  105,   94,   93,
          94,  105,  106,  106,   95,   94,   95,  106,  107,  107,   96,   95,
          96,  107,  108,  108,   97,   96,   97,  108,  109,  109,   98,   97,
          99,  110,  111,  111,  100,   99,  100,  111,  112,  112,  101,  100,
         101,  112,  113,  113,  102,  101,  102,  113,  114,  114,  103,  102,
         103,  114,  115,  115,  104,  103,  104,  115,  116,  116,  105,  104,
         105,  116,  117,  117,  106,  105,  106,  117,  118,  118,  107,  106,
         107,  118,  119,  119,  108,  107,  108,  119,  120,  120,  109,  108,
         110,  121,  122,  122,  111,  110,  111,  122,  123,  123,  112,  111,
         112,  123,  124,  124,  113,  112,  113,  124,  125,  125,  114,  113,
         114,  125,  126,  126,  115,  114,  115,  126,  127,  127,  116,  115,
         116,  127,  128,  128,  117,  116,  117,  128,  129,  129,  118,  117,
         118,  129,  130,  130,  119,  118,  119,  130,  131,  131,  120,  119,
         121,  132,  133,  133,  122,  121,  122,  133,  134,  134,  123,  122,
         123,  134,  135,  135,  124,  123,  124,  135,  136,  136,  125,  124,
         125,  136,  137,  137,  126,  125,  126,  137,  138,  138,  127,  126,
         127,  138,  139,  139,  128,  127,  128,  139,  140,  140,  129,  128,
         129,  140,  141,  141,  130,  129,  130,  141,  142,  142,  131,  130,
         132,  143,  144,  144,  133,  132,  133,  144,  145,  145,  134,  133,
         134,  145,  146,  146,  135,  134,  135,  146,  147,  147,  136,  135,
         136,  147,  148,  148,  137,  136,  137,  148,  149,  149,  138,  137,
         138,  149,  150,  150,  139,  138,  139,  150,  151,  151,  140,  139,
         140,  151,  152,  152,  141,  140,  141,  152,  153,  153,  142,  141,
         143,  154,  155,  155,  144,  143,  144,  155,  156,  156,  145,  144,
         145,  156,  157,  157,  146,  145,  146,  157,  158,  158,  147,  146,
         147,  158,  159,  159,  148,  147,  148,  159,  160,  160,  149,  148,
         149,  160,  161,  161,  150,  149,  150,  161,  162,  162,  151,  150,
         151,  162,  163,  163,  152,  151,  152,  163,  164,  164,  153,  152,
         154,  165,  166,  166,  155,  154,  155,  166,  167,  167,  156,  155,
         156,  167,  168,  168,  157,  156,  157,  168,  169,  169,  158,  157,
         158,  169,  170,  170,  159,  158,  159,  170,  171,  171,  160,  159,
         160,  171,  172,  172,  161,  160,  161,  172,  173,  173,  162,  161,
         162,  173,  174,  174,  163,  162,  163,  174,  175,  175,  164,  163,
         165,  176,  177,  177,  166,  165,  166,  177,  178,  178,  167,  166,
         167,  178,  179,  179,  168,  167,  168,  179,  180,  180,  169,  168,
         169,  180,  181,  181,  170,  169,  170,  181,  182,  182,  171,  170,
         171,  182,  183,  183,  172,  171,  172,  183,  184,  184,  173,  172,
         173,  184,  185,  185,  174,  173,  174,  185,  186,  186,  175,  174,
         176,  187,  188,  188,  177,  176,  177,  188,  189,  189,  178,  177,
         178,  189,  190,  190,  179,  178,  179,  190,  191,  191,  180,  179,
         180,  191,  192,  192,  181,  180,  181,  192,  193,  193,  182,  181,
         182,  193,  194,  194,  183,  182,  183,  194,  195,  195,  184,  183,
         184,  195,  196,  196,  185,  184,  185,  196,  197,  197,  186,  185,
         187,  198,  199,  199,  188,  187,  188,  199,  200,  200,  189,  188,
         189,  200,  201,  201,  190,  189,  190,  201,  202,  202,  191,  190,
         191,  202,  203,  203,  192,  191,  192,  203,  204,  204,  193,  192,
         193,  204,  205,  205,  194,  193,  194,  205,  206,  206,  195,  194,
         195,  206,  207,  207,  196,  195,  196,  207,  208,  208,  197,  196,
         198,  209,  210,  210,  199,  198,  199,  210,  211,  211,  200,  199,
         200,  211,  212,  212,  201,  200,  201,  212,  213,  213,  202,  201,
         202,  213,  214,  214,  203,  202,  203,  214,  215,  215,  204,  203,
         204,  215,  216,  216,  205,  204,  205,  216,  217,  217,  206,  205,
         206,  217,  218,  218,  207,  206,  207,  218,  219,  219,  208,  207,
         209,  220,  221,  221,  210,  209,  210,  221,  222,  222,  211,  210,
         211,  222,  223,  223,  212,  211,  212,  223,  224,  224,  213,  212,
         213,  224,  225,  225,  214,  213,  214,  225,  226,  226,  215,  214,
         215,  226,  227,  227,  216,  215,  216,  227,  228,  228,  217,  216,
         217,  228,  229,  229,  218,  217,  218,  229,  230,  230,  219,  218,
         220,  231,  232,  232,  221,  220,  221,  232,  233,  233,  222,  221,
         222,  233,  234,  234,  223,  222,  223,  234,  235,  235,  224,  223,
         224,  235,  236,  236,  225,  224,  225,  236,  237,  237,  226,  225,
         226,  237,  238,  238,  227,  226,  227,  238,  239,  239,  228,  227,
         228,  239,  240,  240,  229,  228,  229,  240,  241,  241,  230,  229,
         231,  242,  243,  243,  232,  231,  232,  243,  244,  244,  233,  232,
         233,  244,  245,  245,  234,  233,  234,  245,  246,  246,  235,  234,
         235,  246,  247,  247,  236,  235,  236,  247,  248,  248,  237,  236,
         237,  248,  249,  249,  238,  237,  238,  249,  250,  250,  239,  238,
         239,  250,  251,  251,  240,  239,  240,  251,  252,  252,  241,  240,
         242,  253,  254,  254,  243,  242,  243,  254,  255,  255,  244,  243,
         244,  255,  256,  256,  245,  244,  245,  256,  257,  257,  246,  245,
         246,  257,  258,  258,  247,  246,  247,  258,  259,  259,  248,  247,
         248,  259,  260,  260,  249,  248,  249,  260,  261,  261,  250,  249,
         250,  261,  262,  262,  251,  250,  251,  262,  263,  263,  252,  251,
         253,  264,  265,  265,  254,  253,  254,  265,  266,  266,  255,  254,
         255,  266,  267,  267,  256,  255,  256,  267,  268,  268,  257,  256,
         257,  268,  269,  269,  258,  257,  258,  269,  270,  270,  259,  258,
         259,  270,  271,  271,  260,  259,  260,  271,  272,  272,  261,  260,
         261,  272,  273,  273,  262,  261,  262,  273,  274,  274,  263,  262,
         264,  275,  276,  276,  265,  264,  265,  276,  277,  277,  266,  265,
         266,  277,  278,  278,  267,  266,  267,  278,  279,  279,  268,  267,
         268,  279,  280,  280,  269,  268,  269,  280,  281,  281,  270,  269,
         270,  281,  282,  282,  271,  270,  271,  282,  283,  283,  272,  271,
         272,  283,  284,  284,  273,  272,  273,  284,  285,  285,  274,  273,
         275,  286,  287,  287,  276,  275,  276,  287,  288,  288,  277,  276,
         277,  288,  289,  289,  278,  277,  278,  289,  290,  290,  279,  278,
         279,  290,  291,  291,  280,  279,  280,  291,  292,  292,  281,  280,
         281,  292,  293,  293,  282,  281,  282,  293,  294,  294,  283,  282,
         283,  294,  295,  295,  284,  283,  284,  295,  296,  296,  285,  284,
         286,  297,  298,  298,  287,  286,  287,  298,  299,  299,  288,  287,
         288,  299,  300,  300,  289,  288,  289,  300,  301,  301,  290,  289,
         290,  301,  302,  302,  291,  290,  291,  302,  303,  303,  292,  291,
         292,  303,  304,  304,  293,  292,  293,  304,  305,  305,  294,  293,
         294,  305,  306,  306,  295,  294,  295,  306,  307,  307,  296,  295,
         297,  308,  309,  309,  298,  297,  298,  309,  310,  310,  299,  298,
         299,  310,  311,  311,  300,  299,  300,  311,  312,  312,  301,  300,
         301,  312,  313,  313,  302,  301,  302,  313,  314,  314,  303,  302,
         303,  314,  315,  315,  304,  303,  304,  315,  316,  316,  305,  304,
         305,  316,  317,  317,  306,  305,  306,  317,  318,  318,  307,  306,
         308,  319,  320,  320,  309,  308,  309,  320,  321,  321,  310,  309,
         310,  321,  322,  322,  311,  310,  311,  322,  323,  323,  312,  311,
         312,  323,  324,  324,  313,  312,  313,  324,  325,  325,  314,  313,
         314,  325,  326,  326,  315,  314,  315,  326,  327,  327,  316,  315,
         316,  327,  328,  328,  317,  316,  317,  328,  329,  329,  318,  317,
         319,  330,  331,  331,  320,  319,  320,  331,  332,  332,  321,  320,
         321,  332,  333,  333,  322,  321,  322,  333,  334,  334,  323,  322,
         323,  334,  335,  335,  324,  323,  324,  335,  336,  336,  325,  324,
         325,  336,  337,  337,  326,  325,  326,  337,  338,  338,  327,  326,
         327,  338,  339,  339,  328,  327,  328,  339,  340,  340,  329,  328,
         330,  341,  342,  342,  331,  330,  331,  342,  343,  343,  332,  331,
         332,  343,  344,  344,  333,  332,  333,  344,  345,  345,  334,  333,
         334,  345,  346,  346,  335,  334,  335,  346,  347,  347,  336,  335,
         336,  347,  348,  348,  337,  336,  337,  348,  349,  349,  338,  337,
         338,  349,  350,  350,  339,  338,  339,  350,  351,  351,  340,  339,
         341,  352,  353,  353,  342,  341,  342,  353,  354,  354,  343,  342,
         343,  354,  355,  355,  344,  343,  344,  355,  356,  356,  345,  344,
         345,  356,  357,  357,  346,  345,  346,  357,  358,  358,  347,  346,
         347,  358,  359,  359,  348,  347,  348,  359,  360,  360,  349,  348,
         349,  360,  361,  361,  350,  349,  350,  361,  362,  362,  351,  350,
         352,  363,  364,  364,  353,  352,  353,  364,  365,  365,  354,  353,
         354,  365,  366,  366,  355,  354,  355,  366,  367,  367,  356,  355,
         356,  367,  368,  368,  357,  356,  357,  368,  369,  369,  358,  357,
         358,  369,  370,  370,  359,  358,  359,  370,  371,  371,  360,  359,
         360,  371,  372,  372,  361,  360,  361,  372,  373,  373,  362,  361,
         363,  374,  375,  375,  364,  363,  364,  375,  376,  376,  365,  364,
         365,  376,  377,  377,  366,  365,  366,  377,  378,  378,  367,  366,
         367,  378,  379,  379,  368,  367,  368,  379,  380,  380,  369,  368,
         369,  380,  381,  381,  370,  369,  370,  381,  382,  382,  371,  370,
         371,  382,  383,  383,  372,  371,  372,  383,  384,  384,  373,  372,
         374,  385,  386,  386,  375,  374,  375,  386,  387,  387,  376,  375,
         376,  387,  388,  388,  377,  376,  377,  388,  389,  389,  378,  377,
         378,  389,  390,  390,  379,  378,  379,  390,  391,  391,  380,  379,
         380,  391,  392,  392,  381,  380,  381,  392,  393,  393,  382,  381,
         382,  393,  394,  394,  383,  382,  383,  394,  395,  395,  384,  383,
         385,  396,  397,  397,  386,  385,  386,  397,  398,  398,  387,  386,
         387,  398,  399,  399,  388,  387,  388,  399,  400,  400,  389,  388,
         389,  400,  401,  401,  390,  389,  390,  401,  402,  402,  391,  390,
         391,  402,  403,  403,  392,  391,  392,  403,  404,  404,  393,  392,
         393,  404,  405,  405,  394,  393,  394,  405,  406,  406,  395,  394,
         396,  407,  408,  408,  397,  396,  397,  408,  409,  409,  398,  397,
         398,  409,  410,  410,  399,  398,  399,  410,  411,  411,  400,  399,
         400,  411,  412,  412,  401,  400,  401,  412,  413,  413,  402,  401,
         402,  413,  414,  414,  403,  402,  403,  414,  415,  415,  404,  403,
         404,  415,  416,  416,  405,  404,  405,  416,  417,  417,  406,  405,
         407,  418,  419,  419,  408,  407,  408,  419,  420,  420,  409,  408,
         409,  420,  421,  421,  410,  409,  410,  421,  422,  422,  411,  410,
         411,  422,  423,  423,  412,  411,  412,  423,  424,  424,  413,  412,
         413,  424,  425,  425,  414,  413,  414,  425,  426,  426,  415,  414,
         415,  426,  427,  427,  416,  415,  416,  427,  428,  428,  417,  416,
         418,  429,  430,  430,  419,  418,  419,  430,  431,  431,  420,  419,
         420,  431,  432,  432,  421,  420,  421,  432,  433,  433,  422,  421,
         422,  433,  434,  434,  423,  422,  423,  434,  435,  435,  424,  423,
         424,  435,  436,  436,  425,  424,  425,  436,  437,  437,  426,  425,
         426,  437,  438,  438,  427,  426,  427,  438,  439,  439,  428,  427,
         429,    0,    3,    3,  430,  429,  430,    3,    5,    5,  431,  430,
         431,    5,    7,    7,  432,  431,  432,    7,    9,    9,  433,  432,
         433,    9,   11,   11,  434,  433,  434,   11,   13,   13,  435,  434,
         435,   13,   15,   15,  436,  435,  436,   15,   17,   17,  437,  436,
         437,   17,   19,   19,  438,  437,  438,   19,   21,   21,  439,  438,
          21,   20,  440,  440,  441,   21,  441,  440,  442,  442,  443,  441,
         443,  442,  444,  444,  445,  443,  445,  444,  446,  446,  447,  445,
         447,  446,  448,  448,  449,  447,  449,  448,  450,  450,  451,  449,
         451,  450,  452,  452,  453,  451,  453,  452,  454,  454,  455,  453,
         455,  454,  456,  456,  457,  455,  457,  456,  458,  458,  459,  457,
          20,   32,  460,  460,  440,   20,  440,  460,  461,  461,  442,  440,
         442,  461,  462,  462,  444,  442,  444,  462,  463,  463,  446,  444,
         446,  463,  464,  464,  448,  446,  448,  464,  465,  465,  450,  448,
         450,  465,  466,  466,  452,  450,  452,  466,  467,  467,  454,  452,
         454,  467,  468,  468,  456,  454,  456,  468,  469,  469,  458,  456,
          32,   43,  470,  470,  460,   32,  460,  470,  471,  471,  461,  460,
         461,  471,  472,  472,  462,  461,  462,  472,  473,  473,  463,  462,
         463,  473,  474,  474,  464,  463,  464,  474,  475,  475,  465,  464,
         465,  475,  476,  476,  466,  465,  466,  476,  477,  477,  467,  466,
         467,  477,  478,  478,  468,  467,  468,  478,  479,  479,  469,  468,
          43,   54,  480,  480,  470,   43,  470,  480,  481,  481,  471,  470,
         471,  481,  482,  482,  472,  471,  472,  482,  483,  483,  473,  472,
         473,  483,  484,  484,  474,  473,  474,  484,  485,  485,  475,  474,
         475,  485,  486,  486,  476,  475,  476,  486,  487,  487,  477,  476,
         477,  487,  488,  488,  478,  477,  478,  488,  489,  489,  479,  478,
          54,   65,  490,  490,  480,   54,  480,  490,  491,  491,  481,  480,
         481,  491,  492,  492,  482,  481,  482,  492,  493,  493,  483,  482,
         483,  493,  494,  494,  484,  483,  484,  494,  495,  495,  485,  484,
         485,  495,  496,  496,  486,  485,  486,  496,  497,  497,  487,  486,
         487,  497,  498,  498,  488,  487,  488,  498,  499,  499,  489,  488,
          65,   76,  500,  500,  490,   65,  490,  500,  501,  501,  491,  490,
         491,  501,  502,  502,  492,  491,  492,  502,  503,  503,  493,  492,
         493,  503,  504,  504,  494,  493,  494,  504,  505,  505,  495,  494,
         495,  505,  506,  506,  496,  495,  496,  506,  507,  507,  497,  496,
         497,  507,  508,  508,  498,  497,  498,  508,  509,  509,  499,  498,
          76,   87,  510,  510,  500,   76,  500,  510,  511,  511,  501,  500,
         501,  511,  512,  512,  502,  501,  502,  512,  513,  513,  503,  502,
         503,  513,  514,  514,  504,  503,  504,  514,  515,  515,  505,  504,
         505,  515,  516,  516,  506,  505,  506,  516,  517,  517,  507,  506,
         507,  517,  518,  518,  508,  507,  508,  518,  519,  519,  509,  508,
          87,   98,  520,  520,  510,   87,  510,  520,  521,  521,  511,  510,
         511,  521,  522,  522,  512,  511,  512,  522,  523,  523,  513,  512,
         513,  523,  524,  524,  514,  513,  514,  524,  525,  525,  515,  514,
         515,  525,  526,  526,  516,  515,  516,  526,  527,  527,  517,  516,
         517,  527,  528,  528,  518,  517,  518,  528,  529,  529,  519,  518,
          98,  109,  530,  530,  520,   98,  520,  530,  531,  531,  521,  520,
         521,  531,  532,  532,  522,  521,  522,  532,  533,  533,  523,  522,
         523,  533,  534,  534,  524,  523,  524,  534,  535,  535,  525,  524,
         525,  535,  536,  536,  526,  525,  526,  536,  537,  537,  527,  526,
         527,  537,  538,  538,  528,  527,  528,  538,  539,  539,  529,  528,
         109,  120,  540,  540,  530,  109,  530,  540,  541,  541,  531,  530,
         531,  541,  542,  542,  532,  531,  532,  542,  543,  543,  533,  532,
         533,  543,  544,  544,  534,  533,  534,  544,  545,  545,  535,  534,
         535,  545,  546,  546,  536,  535,  536,  546,  547,  547,  537,  536,
         537,  547,  548,  548,  538,  537,  538,  548,  549,  549,  539,  538,
         120,  131,  550,  550,  540,  120,  540,  550,  551,  551,  541,  540,
         541,  551,  552,  552,  542,  541,  542,  552,  553,  553,  543,  542,
         543,  553,  554,  554,  544,  543,  544,  554,  555,  555,  545,  544,
         545,  555,  556,  556,  546,  545,  546,  556,  557,  557,  547,  546,
         547,  557,  558,  558,  548,  547,  548,  558,  559,  559,  549,  548,
         131,  142,  560,  560,  550,  131,  550,  560,  561,  561,  551,  550,
         551,  561,  562,  562,  552,  551,  552,  562,  563,  563,  553,  552,
         553,  563,  564,  564,  554,  553,  554,  564,  565,  565,  555,  554,
         555,  565,  566,  566,  556,  555,  556,  566,  567,  567,  557,  556,
         557,  567,  568,  568,  558,  557,  558,  568,  569,  569,  559,  558,
         142,  153,  570,  570,  560,  142,  560,  570,  571,  571,  561,  560,
         561,  571,  572,  572,  562,  561,  562,  572,  573,  573,  563,  562,
         563,  573,  574,  574,  564,  563,  564,  574,  575,  575,  565,  564,
         565,  575,  576,  576,  566,  565,  566,  576,  577,  577,  567,  566,
         567,  577,  578,  578,  568,  567,  568,  578,  579,  579,  569,  568,
         153,  164,  580,  580,  570,  153,  570,  580,  581,  581,  571,  570,
         571,  581,  582,  582,  572,  571,  572,  582,  583,  583,  573,  572,
         573,  583,  584,  584,  574,  573,  574,  584,  585,  585,  575,  574,
         575,  585,  586,  586,  576,  575,  576,  586,  587,  587,  577,  576,
         577,  587,  588,  588,  578,  577,  578,  588,  589,  589,  579,  578,
         164,  175,  590,  590,  580,  164,  580,  590,  591,  591,  581,  580,
         581,  591,  592,  592,  582,  581,  582,  592,  593,  593,  583,  582,
         583,  593,  594,  594,  584,  583,  584,  594,  595,  595,  585,  584,
         585,  595,  596,  596,  586,  585,  586,  596,  597,  597,  587,  586,
         587,  597,  598,  598,  588,  587,  588,  598,  599,  599,  589,  588,
         175,  186,  600,  600,  590,  175,  590,  600,  601,  601,  591,  590,
         591,  601,  602,  602,  592,  591,  592,  602,  603,  603,  593,  592,
         593,  603,  604,  604,  594,  593,  594,  604,  605,  605,  595,  594,
         595,  605,  606,  606,  596,  595,  596,  606,  607,  607,  597,  596,
         597,  607,  608,  608,  598,  597,  598,  608,  609,  609,  599,  598,
         186,  197,  610,  610,  600,  186,  600,  610,  611,  611,  601,  600,
         601,  611,  612,  612,  602,  601,  602,  612,  613,  613,  603,  602,
         603,  613,  614,  614,  604,  603,  604,  614,  615,  615,  605,  604,
         605,  615,  616,  616,  606,  605,  606,  616,  617,  617,  607,  606,
         607,  617,  618,  618,  608,  607,  608,  618,  619,  619,  609,  608,
         197,  208,  620,  620,  610,  197,  610,  620,  621,  621,  611,  610,
         611,  621,  622,  622,  612,  611,  612,  622,  623,  623,  613,  612,
         613,  623,  624,  624,  614,  613,  614,  624,  625,  625,  615,  614,
         615,  625,  626,  626,  616,  615,  616,  626,  627,  627,  617,  616,
         617,  627,  628,  628,  618,  617,  618,  628,  629,  629,  619,  618,
         208,  219,  630,  630,  620,  208,  620,  630,  631,  631,  621,  620,
         621,  631,  632,  632,  622,  621,  622,  632,  633,  633,  623,  622,
         623,  633,  634,  634,  624,  623,  624,  634,  635,  635,  625,  624,
         625,  635,  636,  636,  626,  625,  626,  636,  637,  637,  627,  626,
         627,  637,  638,  638,  628,  627,  628,  638,  639,  639,  629,  628,
         219,  230,  640,  640,  630,  219,  630,  640,  641,  641,  631,  630,
         631,  641,  642,  642,  632,  631,  632,  642,  643,  643,  633,  632,
         633,  643,  644,  644,  634,  633,  634,  644,  645,  645,  635,  634,
         635,  645,  646,  646,  636,  635,  636,  646,  647,  647,  637,  636,
         637,  647,  648,  648,  638,  637,  638,  648,  649,  649,  639,  638,
         230,  241,  650,  650,  640,  230,  640,  650,  651,  651,  641,  640,
         641,  651,  652,  652,  642,  641,  642,  652,  653,  653,  643,  642,
         643,  653,  654,  654,  644,  643,  644,  654,  655,  655,  645,  644,
         645,  655,  656,  656,  646,  645,  646,  656,  657,  657,  647,  646,
         647,  657,  658,  658,  648,  647,  648,  658,  659,  659,  649,  648,
         241,  252,  660,  660,  650,  241,  650,  660,  661,  661,  651,  650,
         651,  661,  662,  662,  652,  651,  652,  662,  663,  663,  653,  652,
         653,  663,  664,  664,  654,  653,  654,  664,  665,  665,  655,  654,
         655,  665,  666,  666,  656,  655,  656,  666,  667,  667,  657,  656,
         657,  667,  668,  668,  658,  657,  658,  668,  669,  669,  659,  658,
         252,  263,  670,  670,  660,  252,  660,  670,  671,  671,  661,  660,
         661,  671,  672,  672,  662,  661,  662,  672,  673,  673,  663,  662,
         663,  673,  674,  674,  664,  663,  664,  674,  675,  675,  665,  664,
         665,  675,  676,  676,  666,  665,  666,  676,  677,  677,  667,  666,
         667,  677,  678,  678,  668,  667,  668,  678,  679,  679,  669,  668,
         263,  274,  680,  680,  670,  263,  670,  680,  681,  681,  671,  670,
         671,  681,  682,  682,  672,  671,  672,  682,  683,  683,  673,  672,
         673,  683,  684,  684,  674,  673,  674,  684,  685,  685,  675,  674,
         675,  685,  686,  686,  676,  675,  676,  686,  687,  687,  677,  676,
         677,  687,  688,  688,  678,  677,  678,  688,  689,  689,  679,  678,
         274,  285,  690,  690,  680,  274,  680,  690,  691,  691,  681,  680,
         681,  691,  692,  692,  682,  681,  682,  692,  693,  693,  683,  682,
         683,  693,  694,  694,  684,  683,  684,  694,  695,  695,  685,  684,
         685,  695,  696,  696,  686,  685,  686,  696,  697,  697,  687,  686,
         687,  697,  698,  698,  688,  687,  688,  698,  699,  699,  689,  688,
         285,  296,  700,  700,  690,  285,  690,  700,  701,  701,  691,  690,
         691,  701,  702,  702,  692,  691,  692,  702,  703,  703,  693,  692,
         693,  703,  704,  704,  694,  693,  694,  704,  705,  705,  695,  694,
         695,  705,  706,  706,  696,  695,  696,  706,  707,  707,  697,  696,
         697,  707,  708,  708,  698,  697,  698,  708,  709,  709,  699,  698,
         296,  307,  710,  710,  700,  296,  700,  710,  711,  711,  701,  700,
         701,  711,  712,  712,  702,  701,  702,  712,  713,  713,  703,  702,
         703,  713,  714,  714,  704,  703,  704,  714,  715,  715,  705,  704,
         705,  715,  716,  716,  706,  705,  706,  716,  717,  717,  707,  706,
         707,  717,  718,  718,  708,  707,  708,  718,  719,  719,  709,  708,
         307,  318,  720,  720,  710,  307,  710,  720,  721,  721,  711,  710,
         711,  721,  722,  722,  712,  711,  712,  722,  723,  723,  713,  712,
         713,  723,  724,  724,  714,  713,  714,  724,  725,  725,  715,  714,
         715,  725,  726,  726,  716,  715,  716,  726,  727,  727,  717,  716,
         717,  727,  728,  728,  718,  717,  718,  728,  729,  729,  719,  718,
         318,  329,  730,  730,  720,  318,  720,  730,  731,  731,  721,  720,
         721,  731,  732,  732,  722,  721,  722,  732,  733,  733,  723,  722,
         723,  733,  734,  734,  724,  723,  724,  734,  735,  735,  725,  724,
         725,  735,  736,  736,  726,  725,  726,  736,  737,  737,  727,  726,
         727,  737,  738,  738,  728,  727,  728,  738,  739,  739,  729,  728,
         329,  340,  740,  740,  730,  329,  730,  740,  741,  741,  731,  730,
         731,  741,  742,  742,  732,  731,  732,  742,  743,  743,  733,  732,
         733,  743,  744,  744,  734,  733,  734,  744,  745,  745,  735,  734,
         735,  745,  746,  746,  736,  735,  736,  746,  747,  747,  737,  736,
         737,  747,  748,  748,  738,  737,  738,  748,  749,  749,  739,  738,
         340,  351,  750,  750,  740,  340,  740,  750,  751,  751,  741,  740,
         741,  751,  752,  752,  742,  741,  742,  752,  753,  753,  743,  742,
         743,  753,  754,  754,  744,  743,  744,  754,  755,  755,  745,  744,
         745,  755,  756,  756,  746,  745,  746,  756,  757,  757,  747,  746,
         747,  757,  758,  758,  748,  747,  748,  758,  759,  759,  749,  748,
         351,  362,  760,  760,  750,  351,  750,  760,  761,  761,  751,  750,
         751,  761,  762,  762,  752,  751,  752,  762,  763,  763,  753,  752,
         753,  763,  764,  764,  754,  753,  754,  764,  765,  765,  755,  754,
         755,  765,  766,  766,  756,  755,  756,  766,  767,  767,  757,  756,
         757,  767,  768,  768,  758,  757,  758,  768,  769,  769,  759,  758,
         362,  373,  770,  770,  760,  362,  760,  770,  771,  771,  761,  760,
         761,  771,  772,  772,  762,  761,  762,  772,  773,  773,  763,  762,
         763,  773,  774,  774,  764,  763,  764,  774,  775,  775,  765,  764,
         765,  775,  776,  776,  766,  765,  766,  776,  777,  777,  767,  766,
         767,  777,  778,  778,  768,  767,  768,  778,  779,  779,  769,  768,
         373,  384,  780,  780,  770,  373,  770,  780,  781,  781,  771,  770,
         771,  781,  782,  782,  772,  771,  772,  782,  783,  783,  773,  772,
         773,  783,  784,  784,  774,  773,  774,  784,  785,  785,  775,  774,
         775,  785,  786,  786,  776,  775,  776,  786,  787,  787,  777,  776,
         777,  787,  788,  788,  778,  777,  778,  788,  789,  789,  779,  778,
         384,  395,  790,  790,  780,  384,  780,  790,  791,  791,  781,  780,
         781,  791,  792,  792,  782,  781,  782,  792,  793,  793,  783,  782,
         783,  793,  794,  794,  784,  783,  784,  794,  795,  795,  785,  784,
         785,  795,  796,  796,  786,  785,  786,  796,  797,  797,  787,  786,
         787,  797,  798,  798,  788,  787,  788,  798,  799,  799,  789,  788,
         395,  406,  800,  800,  790,  395,  790,  800,  801,  801,  791,  790,
         791,  801,  802,  802,  792,  791,  792,  802,  803,  803,  793,  792,
         793,  803,  804,  804,  794,  793,  794,  804,  805,  805,  795,  794,
         795,  805,  806,  806,  796,  795,  796,  806,  807,  807,  797,  796,
         797,  807,  808,  808,  798,  797,  798,  808,  809,  809,  799,  798,
         406,  417,  810,  810,  800,  406,  800,  810,  811,  811,  801,  800,
         801,  811,  812,  812,  802,  801,  802,  812,  813,  813,  803,  802,
         803,  813,  814,  814,  804,  803,  804,  814,  815,  815,  805,  804,
         805,  815,  816,  816,  806,  805,  806,  816,  817,  817,  807,  806,
         807,  817,  818,  818,  808,  807,  808,  818,  819,  819,  809,  808,
         417,  428,  820,  820,  810,  417,  810,  820,  821,  821,  811,  810,
         811,  821,  822,  822,  812,  811,  812,  822,  823,  823,  813,  812,
         813,  823,  824,  824,  814,  813,  814,  824,  825,  825,  815,  814,
         815,  825,  826,  826,  816,  815,  816,  826,  827,  827,  817,  816,
         817,  827,  828,  828,  818,  817,  818,  828,  829,  829,  819,  818,
         428,  439,  830,  830,  820,  428,  820,  830,  831,  831,  821,  820,
         821,  831,  832,  832,  822,  821,  822,  832,  833,  833,  823,  822,
         823,  833,  834,  834,  824,  823,  824,  834,  835,  835,  825,  824,
         825,  835,  836,  836,  826,  825,  826,  836,  837,  837,  827,  826,
         827,  837,  838,  838,  828,  827,  828,  838,  839,  839,  829,  828,
         439,   21,  441,  441,  830,  439,  830,  441,  443,  443,  831,  830,
         831,  443,  445,  445,  832,  831,  832,  445,  447,  447,  833,  832,
         833,  447,  449,  449,  834,  833,  834,  449,  451,  451,  835,  834,
         835,  451,  453,  453,  836,  835,  836,  453,  455,  455,  837,  836,
         837,  455,  457,  457,  838,  837,  838,  457,  459,  459,  839,  838,
         459,  458,  840,  840,  841,  459,  841,  840,  842,  842,  843,  841,
         843,  842,  844,  844,  845,  843,  845,  844,  846,  846,  847,  845,
         847,  846,  848,  848,  849,  847,  849,  848,  850,  850,  851,  849,
         851,  850,  852,  852,  853,  851,  853,  852,  854,  854,  855,  853,
         855,  854,  856,  856,  857,  855,  857,  856,  858,  858,  859,  857,
         458,  469,  860,  860,  840,  458,  840,  860,  861,  861,  842,  840,
         842,  861,  862,  862,  844,  842,  844,  862,  863,  863,  846,  844,
         846,  863,  864,  864,  848,  846,  848,  864,  865,  865,  850,  848,
         850,  865,  866,  866,  852,  850,  852,  866,  867,  867,  854,  852,
         854,  867,  868,  868,  856,  854,  856,  868,  869,  869,  858,  856,
         469,  479,  870,  870,  860,  469,  860,  870,  871,  871,  861,  860,
         861,  871,  872,  872,  862,  861,  862,  872,  873,  873,  863,  862,
         863,  873,  874,  874,  864,  863,  864,  874,  875,  875,  865,  864,
         865,  875,  876,  876,  866,  865,  866,  876,  877,  877,  867,  866,
         867,  877,  878,  878,  868,  867,  868,  878,  879,  879,  869,  868,
         479,  489,  880,  880,  870,  479,  870,  880,  881,  881,  871,  870,
         871,  881,  882,  882,  872,  871,  872,  882,  883,  883,  873,  872,
         873,  883,  884,  884,  874,  873,  874,  884,  885,  885,  875,  874,
         875,  885,  886,  886,  876,  875,  876,  886,  887,  887,  877,  876,
         877,  887,  888,  888,  878,  877,  878,  888,  889,  889,  879,  878,
         489,  499,  890,  890,  880,  489,  880,  890,  891,  891,  881,  880,
         881,  891,  892,  892,  882,  881,  882,  892,  893,  893,  883,  882,
         883,  893,  894,  894,  884,  883,  884,  894,  895,  895,  885,  884,
         885,  895,  896,  896,  886,  885,  886,  896,  897,  897,  887,  886,
         887,  897,  898,  898,  888,  887,  888,  898,  899,  899,  889,  888,
         499,  509,  900,  900,  890,  499,  890,  900,  901,  901,  891,  890,
         891,  901,  902,  902,  892,  891,  892,  902,  903,  903,  893,  892,
         893,  903,  904,  904,  894,  893,  894,  904,  905,  905,  895,  894,
         895,  905,  906,  906,  896,  895,  896,  906,  907,  907,  897,  896,
         897,  907,  908,  908,  898,  897,  898,  908,  909,  909,  899,  898,
         509,  519,  910,  910,  900,  509,  900,  910,  911,  911,  901,  900,
         901,  911,  912,  912,  902,  901,  902,  912,  913,  913,  903,  902,
         903,  913,  914,  914,  904,  903,  904,  914,  915,  915,  905,  904,
         905,  915,  916,  916,  906,  905,  906,  916,  917,  917,  907,  906,
         907,  917,  918,  918,  908,  907,  908,  918,  919,  919,  909,  908,
         519,  529,  920,  920,  910,  519,  910,  920,  921,  921,  911,  910,
         911,  921,  922,  922,  912,  911,  912,  922,  923,  923,  913,  912,
         913,  923,  924,  924,  914,  913,  914,  924,  925,  925,  915,  914,
         915,  925,  926,  926,  916,  915,  916,  926,  927,  927,  917,  916,
         917,  927,  928,  928,  918,  917,  918,  928,  929,  929,  919,  918,
         529,  539,  930,  930,  920,  529,  920,  930,  931,  931,  921,  920,
         921,  931,  932,  932,  922,  921,  922,  932,  933,  933,  923,  922,
         923,  933,  934,  934,  924,  923,  924,  934,  935,  935,  925,  924,
         925,  935,  936,  936,  926,  925,  926,  936,  937,  937,  927,  926,
         927,  937,  938,  938,  928,  927,  928,  938,  939,  939,  929,  928,
         539,  549,  940,  940,  930,  539,  930,  940,  941,  941,  931,  930,
         931,  941,  942,  942,  932,  931,  932,  942,  943,  943,  933,  932,
         933,  943,  944,  944,  934,  933,  934,  944,  945,  945,  935,  934,
         935,  945,  946,  946,  936,  935,  936,  946,  947,  947,  937,  936,
         937,  947,  948,  948,  938,  937,  938,  948,  949,  949,  939,  938,
         549,  559,  950,  950,  940,  549,  940,  950,  951,  951,  941,  940,
         941,  951,  952,  952,  942,  941,  942,  952,  953,  953,  943,  942,
         943,  953,  954,  954,  944,  943,  944,  954,  955,  955,  945,  944,
         945,  955,  956,  956,  946,  945,  946,  956,  957,  957,  947,  946,
         947,  957,  958,  958,  948,  947,  948,  958,  959,  959,  949,  948,
         559,  569,  960,  960,  950,  559,  950,  960,  961,  961,  951,  950,
         951,  961,  962,  962,  952,  951,  952,  962,  963,  963,  953,  952,
         953,  963,  964,  964,  954,  953,  954,  964,  965,  965,  955,  954,
         955,  965,  966,  966,  956,  955,  956,  966,  967,  967,  957,  956,
         957,  967,  968,  968,  958,  957,  958,  968,  969,  969,  959,  958,
         569,  579,  970,  970,  960,  569,  960,  970,  971,  971,  961,  960,
         961,  971,  972,  972,  962,  961,  962,  972,  973,  973,  963,  962,
         963,  973,  974,  974,  964,  963,  964,  974,  975,  975,  965,  964,
         965,  975,  976,  976,  966,  965,  966,  976,  977,  977,  967,  966,
         967,  977,  978,  978,  968,  967,  968,  978,  979,  979,  969,  968,
         579,  589,  980,  980,  970,  579,  970,  980,  981,  981,  971,  970,
         971,  981,  982,  982,  972,  971,  972,  982,  983,  983,  973,  972,
         973,  983,  984,  984,  974,  973,  974,  984,  985,  985,  975,  974,
         975,  985,  986,  986,  976,  975,  976,  986,  987,  987,  977,  976,
         977,  987,  988,  988,  978,  977,  978,  988,  989,  989,  979,  978,
         589,  599,  990,  990,  980,  589,  980,  990,  991,  991,  981,  980,
         981,  991,  992,  992,  982,  981,  982,  992,  993,  993,  983,  982,
         983,  993,  994,  994,  984,  983,  984,  994,  995,  995,  985,  984,
         985,  995,  996,  996,  986,  985,  986,  996,  997,  997,  987,  986,
         987,  997,  998,  998,  988,  987,  988,  998,  999,  999,  989,  988,
         599,  609, 1000, 1000,  990,  599,  990, 1000, 1001, 1001,  991,  990,
         991, 1001, 1002, 1002,  992,  991,  992, 1002, 1003, 1003,  993,  992,
         993, 1003, 1004, 1004,  994,  993,  994, 1004, 1005, 1005,  995,  994,
         995, 1005, 1006, 1006,  996,  995,  996, 1006, 1007, 1007,  997,  996,
         997, 1007, 1008, 1008,  998,  997,  998, 1008, 1009, 1009,  999,  998,
         609,  619, 1010, 1010, 1000,  609, 1000, 1010, 1011, 1011, 1001, 1000,
        1001, 1011, 1012, 1012, 1002, 1001, 1002, 1012, 1013, 1013, 1003, 1002,
        1003, 1013, 1014, 1014, 1004, 1003, 1004, 1014, 1015, 1015, 1005, 1004,
        1005, 1015, 1016, 1016, 1006, 1005, 1006, 1016, 1017, 1017, 1007, 1006,
        1007, 1017, 1018, 1018, 1008, 1007, 1008, 1018, 1019, 1019, 1009, 1008,
         619,  629, 1020, 1020, 1010,  619, 1010, 1020, 1021, 1021, 1011, 1010,
        1011, 1021, 1022, 1022, 1012, 1011, 1012, 1022, 1023, 1023, 1013, 1012,
        1013, 1023, 1024, 1024, 1014, 1013, 1014, 1024, 1025, 1025, 1015, 1014,
        1015, 1025, 1026, 1026, 1016, 1015, 1016, 1026, 1027, 1027, 1017, 1016,
        1017, 1027, 1028, 1028, 1018, 1017, 1018, 1028, 1029, 1029, 1019, 1018,
         629,  639, 1030, 1030, 1020,  629, 1020, 1030, 1031, 1031, 1021, 1020,
        1021, 1031, 1032, 1032, 1022, 1021, 1022, 1032, 1033, 1033, 1023, 1022,
        1023, 1033, 1034, 1034, 1024, 1023, 1024, 1034, 1035, 1035, 1025, 1024,
        1025, 1035, 1036, 1036, 1026, 1025, 1026, 1036, 1037, 1037, 1027, 1026,
        1027, 1037, 1038, 1038, 1028, 1027, 1028, 1038, 1039, 1039, 1029, 1028,
         639,  649, 1040, 1040, 1030,  639, 1030, 1040, 1041, 1041, 1031, 1030,
        1031, 1041, 1042, 1042, 1032, 1031, 1032, 1042, 1043, 1043, 1033, 1032,
        1033, 1043, 1044, 1044, 1034, 1033, 1034, 1044, 1045, 1045, 1035, 1034,
        1035, 1045, 1046, 1046, 1036, 1035, 1036, 1046, 1047, 1047, 1037, 1036,
        1037, 1047, 1048, 1048, 1038, 1037, 1038, 1048, 1049, 1049, 1039, 1038,
         649,  659, 1050, 1050, 1040,  649, 1040, 1050, 1051, 1051, 1041, 1040,
        1041, 1051, 1052, 1052, 1042, 1041, 1042, 1052, 1053, 1053, 1043, 1042,
        1043, 1053, 1054, 1054, 1044, 1043, 1044, 1054, 1055, 1055, 1045, 1044,
        1045, 1055, 1056, 1056, 1046, 1045, 1046, 1056, 1057, 1057, 1047, 1046,
        1047, 1057, 1058, 1058, 1048, 1047, 1048, 1058, 1059, 1059, 1049, 1048,
         659,  669, 1060, 1060, 1050,  659, 1050, 1060, 1061, 1061, 1051, 1050,
        1051, 1061, 1062, 1062, 1052, 1051, 1052, 1062, 1063, 1063, 1053, 1052,
        1053, 1063, 1064, 1064, 1054, 1053, 1054, 1064, 1065, 1065, 1055, 1054,
        1055, 1065, 1066, 1066, 1056, 1055, 1056, 1066, 1067, 1067, 1057, 1056,
        1057, 1067, 1068, 1068, 1058, 1057, 1058, 1068, 1069, 1069, 1059, 1058,
         669,  679, 1070, 1070, 1060,  669, 1060, 1070, 1071, 1071, 1061, 1060,
        1061, 1071, 1072, 1072, 1062, 1061, 1062, 1072, 1073, 1073, 1063, 1062,
        1063, 1073, 1074, 1074, 1064, 1063, 1064, 1074, 1075, 1075, 1065, 1064,
        1065, 1075, 1076, 1076, 1066, 1065, 1066, 1076, 1077, 1077, 1067, 1066,
        1067, 1077, 1078, 1078, 1068, 1067, 1068, 1078, 1079, 1079, 1069, 1068,
         679,  689, 1080, 1080, 1070,  679, 1070, 1080, 1081, 1081, 1071, 1070,
        1071, 1081, 1082, 1082, 1072, 1071, 1072, 1082, 1083, 1083, 1073, 1072,
        1073, 1083, 1084, 1084, 1074, 1073, 1074, 1084, 1085, 1085, 1075, 1074,
        1075, 1085, 1086, 1086, 1076, 1075, 1076, 1086, 1087, 1087, 1077, 1076,
        1077, 1087, 1088, 1088, 1078, 1077, 1078, 1088, 1089, 1089, 1079, 1078,
         689,  699, 1090, 1090, 1080,  689, 1080, 1090, 1091, 1091, 1081, 1080,
        1081, 1091, 1092, 1092, 1082, 1081, 1082, 1092, 1093, 1093, 1083, 1082,
        1083, 1093, 1094, 1094, 1084, 1083, 1084, 1094, 1095, 1095, 1085, 1084,
        1085, 1095, 1096, 1096, 1086, 1085, 1086, 1096, 1097, 1097, 1087, 1086,
        1087, 1097, 1098, 1098, 1088, 1087, 1088, 1098, 1099, 1099, 1089, 1088,
         699,  709, 1100, 1100, 1090,  699, 1090, 1100, 1101, 1101, 1091, 1090,
        1091, 1101, 1102, 1102, 1092, 1091, 1092, 1102, 1103, 1103, 1093, 1092,
        1093, 1103, 1104, 1104, 1094, 1093, 1094, 1104, 1105, 1105, 1095, 1094,
        1095, 1105, 1106, 1106, 1096, 1095, 1096, 1106, 1107, 1107, 1097, 1096,
        1097, 1107, 1108, 1108, 1098, 1097, 1098, 1108, 1109, 1109, 1099, 1098,
         709,  719, 1110, 1110, 1100,  709, 1100, 1110, 1111, 1111, 1101, 1100,
        1101, 1111, 1112, 1112, 1102, 1101, 1102, 1112, 1113, 1113, 1103, 1102,
        1103, 1113, 1114, 1114, 1104, 1103, 1104, 1114, 1115, 1115, 1105, 1104,
        1105, 1115, 1116, 1116, 1106, 1105, 1106, 1116, 1117, 1117, 1107, 1106,
        1107, 1117, 1118, 1118, 1108, 1107, 1108, 1118, 1119, 1119, 1109, 1108,
         719,  729, 1120, 1120, 1110,  719, 1110, 1120, 1121, 1121, 1111, 1110,
        1111, 1121, 1122, 1122, 1112, 1111, 1112, 1122, 1123, 1123, 1113, 1112,
        1113, 1123, 1124, 1124, 1114, 1113, 1114, 1124, 1125, 1125, 1115, 1114,
        1115, 1125, 1126, 1126, 1116, 1115, 1116, 1126, 1127, 1127, 1117, 1116,
        1117, 1127, 1128, 1128, 1118, 1117, 1118, 1128, 1129, 1129, 1119, 1118,
         729,  739, 1130, 1130, 1120,  729, 1120, 1130, 1131, 1131, 1121, 1120,
        1121, 1131, 1132, 1132, 1122, 1121, 1122, 1132, 1133, 1133, 1123, 1122,
        1123, 1133, 1134, 1134, 1124, 1123, 1124, 1134, 1135, 1135, 1125, 1124,
        1125, 1135, 1136, 1136, 1126, 1125, 1126, 1136, 1137, 1137, 1127, 1126,
        1127, 1137, 1138, 1138, 1128, 1127, 1128, 1138, 1139, 1139, 1129, 1128,
         739,  749, 1140, 1140, 1130,  739, 1130, 1140, 1141, 1141, 1131, 1130,
        1131, 1141, 1142, 1142, 1132, 1131, 1132, 1142, 1143, 1143, 1133, 1132,
        1133, 1143, 1144, 1144, 1134, 1133, 1134, 1144, 1145, 1145, 1135, 1134,
        1135, 1145, 1146, 1146, 1136, 1135, 1136, 1146, 1147, 1147, 1137, 1136,
        1137, 1147, 1148, 1148, 1138, 1137, 1138, 1148, 1149, 1149, 1139, 1138,
         749,  759, 1150, 1150, 1140,  749, 1140, 1150, 1151, 1151, 1141, 1140,
        1141, 1151, 1152, 1152, 1142, 1141, 1142, 1152, 1153, 1153, 1143, 1142,
        1143, 1153, 1154, 1154, 1144, 1143, 1144, 1154, 1155, 1155, 1145, 1144,
        1145, 1155, 1156, 1156, 1146, 1145, 1146, 1156, 1157, 1157, 1147, 1146,
        1147, 1157, 1158, 1158, 1148, 1147, 1148, 1158, 1159, 1159, 1149, 1148,
         759,  769, 1160, 1160, 1150,  759, 1150, 1160, 1161, 1161, 1151, 1150,
        1151, 1161, 1162, 1162, 1152, 1151, 1152, 1162, 1163, 1163, 1153, 1152,
        1153, 1163, 1164, 1164, 1154, 1153, 1154, 1164, 1165, 1165, 1155, 1154,
        1155, 1165, 1166, 1166, 1156, 1155, 1156, 1166, 1167, 1167, 1157, 1156,
        1157, 1167, 1168, 1168, 1158, 1157, 1158, 1168, 1169, 1169, 1159, 1158,
         769,  779, 1170, 1170, 1160,  769, 1160, 1170, 1171, 1171, 1161, 1160,
        1161, 1171, 1172, 1172, 1162, 1161, 1162, 1172, 1173, 1173, 1163, 1162,
        1163, 1173, 1174, 1174, 1164, 1163, 1164, 1174, 1175, 1175, 1165, 1164,
        1165, 1175, 1176, 1176, 1166, 1165, 1166, 1176, 1177, 1177, 1167, 1166,
        1167, 1177, 1178, 1178, 1168, 1167, 1168, 1178, 1179, 1179, 1169, 1168,
         779,  789, 1180, 1180, 1170,  779, 1170, 1180, 1181, 1181, 1171, 1170,
        1171, 1181, 1182, 1182, 1172, 1171, 1172, 1182, 1183, 1183, 1173, 1172,
        1173, 1183, 1184, 1184, 1174, 1173, 1174, 1184, 1185, 1185, 1175, 1174,
        1175, 1185, 1186, 1186, 1176, 1175, 1176, 1186, 1187, 1187, 1177, 1176,
        1177, 1187, 1188, 1188, 1178, 1177, 1178, 1188, 1189, 1189, 1179, 1178,
         789,  799, 1190, 1190, 1180,  789, 1180, 1190, 1191, 1191, 1181, 1180,
        1181, 1191, 1192, 1192, 1182, 1181, 1182, 1192, 1193, 1193, 1183, 1182,
        1183, 1193, 1194, 1194, 1184, 1183, 1184, 1194, 1195, 1195, 1185, 1184,
        1185, 1195, 1196, 1196, 1186, 1185, 1186, 1196, 1197, 1197, 1187, 1186,
        1187, 1197, 1198, 1198, 1188, 1187, 1188, 1198, 1199, 1199, 1189, 1188,
         799,  809, 1200, 1200, 1190,  799, 1190, 1200, 1201, 1201, 1191, 1190,
        1191, 1201, 1202, 1202, 1192, 1191, 1192, 1202, 1203, 1203, 1193, 1192,
        1193, 1203, 1204, 1204, 1194, 1193, 1194, 1204, 1205, 1205, 1195, 1194,
        1195, 1205, 1206, 1206, 1196, 1195, 1196, 1206, 1207, 1207, 1197, 1196,
        1197, 1207, 1208, 1208, 1198, 1197, 1198, 1208, 1209, 1209, 1199, 1198,
         809,  819, 1210, 1210, 1200,  809, 1200, 1210, 1211, 1211, 1201, 1200,
        1201, 1211, 1212, 1212, 1202, 1201, 1202, 1212, 1213, 1213, 1203, 1202,
        1203, 1213, 1214, 1214, 1204, 1203, 1204, 1214, 1215, 1215, 1205, 1204,
        1205, 1215, 1216, 1216, 1206, 1205, 1206, 1216, 1217, 1217, 1207, 1206,
        1207, 1217, 1218, 1218, 1208, 1207, 1208, 1218, 1219, 1219, 1209, 1208,
         819,  829, 1220, 1220, 1210,  819, 1210, 1220, 1221, 1221, 1211, 1210,
        1211, 1221, 1222, 1222, 1212, 1211, 1212, 1222, 1223, 1223, 1213, 1212,
        1213, 1223, 1224, 1224, 1214, 1213, 1214, 1224, 1225, 1225, 1215, 1214,
        1215, 1225, 1226, 1226, 1216, 1215, 1216, 1226, 1227, 1227, 1217, 1216,
        1217, 1227, 1228, 1228, 1218, 1217, 1218, 1228, 1229, 1229, 1219, 1218,
         829,  839, 1230, 1230, 1220,  829, 1220, 1230, 1231, 1231, 1221, 1220,
        1221, 1231, 1232, 1232, 1222, 1221, 1222, 1232, 1233, 1233, 1223, 1222,
        1223, 1233, 1234, 1234, 1224, 1223, 1224, 1234, 1235, 1235, 1225, 1224,
        1225, 1235, 1236, 1236, 1226, 1225, 1226, 1236, 1237, 1237, 1227, 1226,
        1227, 1237, 1238, 1238, 1228, 1227, 1228, 1238, 1239, 1239, 1229, 1228,
         839,  459,  841,  841, 1230,  839, 1230,  841,  843,  843, 1231, 1230,
        1231,  843,  845,  845, 1232, 1231, 1232,  845,  847,  847, 1233, 1232,
        1233,  847,  849,  849, 1234, 1233, 1234,  849,  851,  851, 1235, 1234,
        1235,  851,  853,  853, 1236, 1235, 1236,  853,  855,  855, 1237, 1236,
        1237,  855,  857,  857, 1238, 1237, 1238,  857,  859,  859, 1239, 1238,
         859,  858, 1240, 1240, 1241,  859, 1241, 1240, 1242, 1242, 1243, 1241,
        1243, 1242, 1244, 1244, 1245, 1243, 1245, 1244, 1246, 1246, 1247, 1245,
        1247, 1246, 1248, 1248, 1249, 1247, 1249, 1248, 1250, 1250, 1251, 1249,
        1251, 1250, 1252, 1252, 1253, 1251, 1253, 1252, 1254, 1254, 1255, 1253,
        1255, 1254, 1256, 1256, 1257, 1255, 1257, 1256, 1258,  858,  869, 1259,
        1259, 1240,  858, 1240, 1259, 1260, 1260, 1242, 1240, 1242, 1260, 1261,
        1261, 1244, 1242, 1244, 1261, 1262, 1262, 1246, 1244, 1246, 1262, 1263,
        1263, 1248, 1246, 1248, 1263, 1264, 1264, 1250, 1248, 1250, 1264, 1265,
        1265, 1252, 1250, 1252, 1265, 1266, 1266, 1254, 1252, 1254, 1266, 1267,
        1267, 1256, 1254, 1256, 1267, 1258,  869,  879, 1268, 1268, 1259,  869,
        1259, 1268, 1269, 1269, 1260, 1259, 1260, 1269, 1270, 1270, 1261, 1260,
        1261, 1270, 1271, 1271, 1262, 1261, 1262, 1271, 1272, 1272, 1263, 1262,
        1263, 1272, 1273, 1273, 1264, 1263, 1264, 1273, 1274, 1274, 1265, 1264,
        1265, 1274, 1275, 1275, 1266, 1265, 1266, 1275, 1276, 1276, 1267, 1266,
        1267, 1276, 1258,  879,  889, 1277, 1277, 1268,  879, 1268, 1277, 1278,
        1278, 1269, 1268, 1269, 1278, 1279, 1279, 1270, 1269, 1270, 1279, 1280,
        1280, 1271, 1270, 1271, 1280, 1281, 1281, 1272, 1271, 1272, 1281, 1282,
        1282, 1273, 1272, 1273, 1282, 1283, 1283, 1274, 1273, 1274, 1283, 1284,
        1284, 1275, 1274, 1275, 1284, 1285, 1285, 1276, 1275, 1276, 1285, 1258,
         889,  899, 1286, 1286, 1277,  889, 1277, 1286, 1287, 1287, 1278, 1277,
        1278, 1287, 1288, 1288, 1279, 1278, 1279, 1288, 1289, 1289, 1280, 1279,
        1280, 1289, 1290, 1290, 1281, 1280, 1281, 1290, 1291, 1291, 1282, 1281,
        1282, 1291, 1292, 1292, 1283, 1282, 1283, 1292, 1293, 1293, 1284, 1283,
        1284, 1293, 1294, 1294, 1285, 1284, 1285, 1294, 1258,  899,  909, 1295,
        1295, 1286,  899, 1286, 1295, 1296, 1296, 1287, 1286, 1287, 1296, 1297,
        1297, 1288, 1287, 1288, 1297, 1298, 1298, 1289, 1288, 1289, 1298, 1299,
        1299, 1290, 1289, 1290, 1299, 1300, 1300, 1291, 1290, 1291, 1300, 1301,
        1301, 1292, 1291, 1292, 1301, 1302, 1302, 1293, 1292, 1293, 1302, 1303,
        1303, 1294, 1293, 1294, 1303, 1258,  909,  919, 1304, 1304, 1295,  909,
        1295, 1304, 1305, 1305, 1296, 1295, 1296, 1305, 1306, 1306, 1297, 1296,
        1297, 1306, 1307, 1307, 1298, 1297, 1298, 1307, 1308, 1308, 1299, 1298,
        1299, 1308, 1309, 1309, 1300, 1299, 1300, 1309, 1310, 1310, 1301, 1300,
        1301, 1310, 1311, 1311, 1302, 1301, 1302, 1311, 1312, 1312, 1303, 1302,
        1303, 1312, 1258,  919,  929, 1313, 1313, 1304,  919, 1304, 1313, 1314,
        1314, 1305, 1304, 1305, 1314, 1315, 1315, 1306, 1305, 1306, 1315, 1316,
        1316, 1307, 1306, 1307, 1316, 1317, 1317, 1308, 1307, 1308, 1317, 1318,
        1318, 1309, 1308, 1309, 1318, 1319, 1319, 1310, 1309, 1310, 1319, 1320,
        1320, 1311, 1310, 1311, 1320, 1321, 1321, 1312, 1311, 1312, 1321, 1258,
         929,  939, 1322, 1322, 1313,  929, 1313, 1322, 1323, 1323, 1314, 1313,
        1314, 1323, 1324, 1324, 1315, 1314, 1315, 1324, 1325, 1325, 1316, 1315,
        1316, 1325, 1326, 1326, 1317, 1316, 1317, 1326, 1327, 1327, 1318, 1317,
        1318, 1327, 1328, 1328, 1319, 1318, 1319, 1328, 1329, 1329, 1320, 1319,
        1320, 1329, 1330, 1330, 1321, 1320, 1321, 1330, 1258,  939,  949, 1331,
        1331, 1322,  939, 1322, 1331, 1332, 1332, 1323, 1322, 1323, 1332, 1333,
        1333, 1324, 1323, 1324, 1333, 1334, 1334, 1325, 1324, 1325, 1334, 1335,
        1335, 1326, 1325, 1326, 1335, 1336, 1336, 1327, 1326, 1327, 1336, 1337,
        1337, 1328, 1327, 1328, 1337, 1338, 1338, 1329, 1328, 1329, 1338, 1339,
        1339, 1330, 1329, 1330, 1339, 1258,  949,  959, 1340, 1340, 1331,  949,
        1331, 1340, 1341, 1341, 1332, 1331, 1332, 1341, 1342, 1342, 1333, 1332,
        1333, 1342, 1343, 1343, 1334, 1333, 1334, 1343, 1344, 1344, 1335, 1334,
        1335, 1344, 1345, 1345, 1336, 1335, 1336, 1345, 1346, 1346, 1337, 1336,
        1337, 1346, 1347, 1347, 1338, 1337, 1338, 1347, 1348, 1348, 1339, 1338,
        1339, 1348, 1258,  959,  969, 1349, 1349, 1340,  959, 1340, 1349, 1350,
        1350, 1341, 1340, 1341, 1350, 1351, 1351, 1342, 1341, 1342, 1351, 1352,
        1352, 1343, 1342, 1343, 1352, 1353, 1353, 1344, 1343, 1344, 1353, 1354,
        1354, 1345, 1344, 1345, 1354, 1355, 1355, 1346, 1345, 1346, 1355, 1356,
        1356, 1347, 1346, 1347, 1356, 1357, 1357, 1348, 1347, 1348, 1357, 1258,
         969,  979, 1358, 1358, 1349,  969, 1349, 1358, 1359, 1359, 1350, 1349,
        1350, 1359, 1360, 1360, 1351, 1350, 1351, 1360, 1361, 1361, 1352, 1351,
        1352, 1361, 1362, 1362, 1353, 1352, 1353, 1362, 1363, 1363, 1354, 1353,
        1354, 1363, 1364, 1364, 1355, 1354, 1355, 1364, 1365, 1365, 1356, 1355,
        1356, 1365, 1366, 1366, 1357, 1356, 1357, 1366, 1258,  979,  989, 1367,
        1367, 1358,  979, 1358, 1367, 1368, 1368, 1359, 1358, 1359, 1368, 1369,
        1369, 1360, 1359, 1360, 1369, 1370, 1370, 1361, 1360, 1361, 1370, 1371,
        1371, 1362, 1361, 1362, 1371, 1372, 1372, 1363, 1362, 1363, 1372, 1373,
        1373, 1364, 1363, 1364, 1373, 1374, 1374, 1365, 1364, 1365, 1374, 1375,
        1375, 1366, 1365, 1366, 1375, 1258,  989,  999, 1376, 1376, 1367,  989,
        1367, 1376, 1377, 1377, 1368, 1367, 1368, 1377, 1378, 1378, 1369, 1368,
        1369, 1378, 1379, 1379, 1370, 1369, 1370, 1379, 1380, 1380, 1371, 1370,
        1371, 1380, 1381, 1381, 1372, 1371, 1372, 1381, 1382, 1382, 1373, 1372,
        1373, 1382, 1383, 1383, 1374, 1373, 1374, 1383, 1384, 1384, 1375, 1374,
        1375, 1384, 1258,  999, 1009, 1385, 1385, 1376,  999, 1376, 1385, 1386,
        1386, 1377, 1376, 1377, 1386, 1387, 1387, 1378, 1377, 1378, 1387, 1388,
        1388, 1379, 1378, 1379, 1388, 1389, 1389, 1380, 1379, 1380, 1389, 1390,
        1390, 1381, 1380, 1381, 1390, 1391, 1391, 1382, 1381, 1382, 1391, 1392,
        1392, 1383, 1382, 1383, 1392, 1393, 1393, 1384, 1383, 1384, 1393, 1258,
        1009, 1019, 1394, 1394, 1385, 1009, 1385, 1394, 1395, 1395, 1386, 1385,
        1386, 1395, 1396, 1396, 1387, 1386, 1387, 1396, 1397, 1397, 1388, 1387,
        1388, 1397, 1398, 1398, 1389, 1388, 1389, 1398, 1399, 1399, 1390, 1389,
        1390, 1399, 1400, 1400, 1391, 1390, 1391, 1400, 1401, 1401, 1392, 1391,
        1392, 1401, 1402, 1402, 1393, 1392, 1393, 1402, 1258, 1019, 1029, 1403,
        1403, 1394, 1019, 1394, 1403, 1404, 1404, 1395, 1394, 1395, 1404, 1405,
        1405, 1396, 1395, 1396, 1405, 1406, 1406, 1397, 1396, 1397, 1406, 1407,
        1407, 1398, 1397, 1398, 1407, 1408, 1408, 1399, 1398, 1399, 1408, 1409,
        1409, 1400, 1399, 1400, 1409, 1410, 1410, 1401, 1400, 1401, 1410, 1411,
        1411, 1402, 1401, 1402, 1411, 1258, 1029, 1039, 1412, 1412, 1403, 1029,
        1403, 1412, 1413, 1413, 1404, 1403, 1404, 1413, 1414, 1414, 1405, 1404,
        1405, 1414, 1415, 1415, 1406, 1405, 1406, 1415, 1416, 1416, 1407, 1406,
        1407, 1416, 1417, 1417, 1408, 1407, 1408, 1417, 1418, 1418, 1409, 1408,
        1409, 1418, 1419, 1419, 1410, 1409, 1410, 1419, 1420, 1420, 1411, 1410,
        1411, 1420, 1258, 1039, 1049, 1421, 1421, 1412, 1039, 1412, 1421, 1422,
        1422, 1413, 1412, 1413, 1422, 1423, 1423, 1414, 1413, 1414, 1423, 1424,
        1424, 1415, 1414, 1415, 1424, 1425, 1425, 1416, 1415, 1416, 1425, 1426,
        1426, 1417, 1416, 1417, 1426, 1427, 1427, 1418, 1417, 1418, 1427, 1428,
        1428, 1419, 1418, 1419, 1428, 1429, 1429, 1420, 1419, 1420, 1429, 1258,
        1049, 1059, 1430, 1430, 1421, 1049, 1421, 1430, 1431, 1431, 1422, 1421,
        1422, 1431, 1432, 1432, 1423, 1422, 1423, 1432, 1433, 1433, 1424, 1423,
        1424, 1433, 1434, 1434, 1425, 1424, 1425, 1434, 1435, 1435, 1426, 1425,
        1426, 1435, 1436, 1436, 1427, 1426, 1427, 1436, 1437, 1437, 1428, 1427,
        1428, 1437, 1438, 1438, 1429, 1428, 1429, 1438, 1258, 1059, 1069, 1439,
        1439, 1430, 1059, 1430, 1439, 1440, 1440, 1431, 1430, 1431, 1440, 1441,
        1441, 1432, 1431, 1432, 1441, 1442, 1442, 1433, 1432, 1433, 1442, 1443,
        1443, 1434, 1433, 1434, 1443, 1444, 1444, 1435, 1434, 1435, 1444, 1445,
        1445, 1436, 1435, 1436, 1445, 1446, 1446, 1437, 1436, 1437, 1446, 1447,
        1447, 1438, 1437, 1438, 1447, 1258, 1069, 1079, 1448, 1448, 1439, 1069,
        1439, 1448, 1449, 1449, 1440, 1439, 1440, 1449, 1450, 1450, 1441, 1440,
        1441, 1450, 1451, 1451, 1442, 1441, 1442, 1451, 1452, 1452, 1443, 1442,
        1443, 1452, 1453, 1453, 1444, 1443, 1444, 1453, 1454, 1454, 1445, 1444,
        1445, 1454, 1455, 1455, 1446, 1445, 1446, 1455, 1456, 1456, 1447, 1446,
        1447, 1456, 1258, 1079, 1089, 1457, 1457, 1448, 1079, 1448, 1457, 1458,
        1458, 1449, 1448, 1449, 1458, 1459, 1459, 1450, 1449, 1450, 1459, 1460,
        1460, 1451, 1450, 1451, 1460, 1461, 1461, 1452, 1451, 1452, 1461, 1462,
        1462, 1453, 1452, 1453, 1462, 1463, 1463, 1454, 1453, 1454, 1463, 1464,
        1464, 1455, 1454, 1455, 1464, 1465, 1465, 1456, 1455, 1456, 1465, 1258,
        1089, 1099, 1466, 1466, 1457, 1089, 1457, 1466, 1467, 1467, 1458, 1457,
        1458, 1467, 1468, 1468, 1459, 1458, 1459, 1468, 1469, 1469, 1460, 1459,
        1460, 1469, 1470, 1470, 1461, 1460, 1461, 1470, 1471, 1471, 1462, 1461,
        1462, 1471, 1472, 1472, 1463, 1462, 1463, 1472, 1473, 1473, 1464, 1463,
        1464, 1473, 1474, 1474, 1465, 1464, 1465, 1474, 1258, 1099, 1109, 1475,
        1475, 1466, 1099, 1466, 1475, 1476, 1476, 1467, 1466, 1467, 1476, 1477,
        1477, 1468, 1467, 1468, 1477, 1478, 1478, 1469, 1468, 1469, 1478, 1479,
        1479, 1470, 1469, 1470, 1479, 1480, 1480, 1471, 1470, 1471, 1480, 1481,
        1481, 1472, 1471, 1472, 1481, 1482, 1482, 1473, 1472, 1473, 1482, 1483,
        1483, 1474, 1473, 1474, 1483, 1258, 1109, 1119, 1484, 1484, 1475, 1109,
        1475, 1484, 1485, 1485, 1476, 1475, 1476, 1485, 1486, 1486, 1477, 1476,
        1477, 1486, 1487, 1487, 1478, 1477, 1478, 1487, 1488, 1488, 1479, 1478,
        1479, 1488, 1489, 1489, 1480, 1479, 1480, 1489, 1490, 1490, 1481, 1480,
        1481, 1490, 1491, 1491, 1482, 1481, 1482, 1491, 1492, 1492, 1483, 1482,
        1483, 1492, 1258, 1119, 1129, 1493, 1493, 1484, 1119, 1484, 1493, 1494,
        1494, 1485, 1484, 1485, 1494, 1495, 1495, 1486, 1485, 1486, 1495, 1496,
        1496, 1487, 1486, 1487, 1496, 1497, 1497, 1488, 1487, 1488, 1497, 1498,
        1498, 1489, 1488, 1489, 1498, 1499, 1499, 1490, 1489, 1490, 1499, 1500,
        1500, 1491, 1490, 1491, 1500, 1501, 1501, 1492, 1491, 1492, 1501, 1258,
        1129, 1139, 1502, 1502, 1493, 1129, 1493, 1502, 1503, 1503, 1494, 1493,
        1494, 1503, 1504, 1504, 1495, 1494, 1495, 1504, 1505, 1505, 1496, 1495,
        1496, 1505, 1506, 1506, 1497, 1496, 1497, 1506, 1507, 1507, 1498, 1497,
        1498, 1507, 1508, 1508, 1499, 1498, 1499, 1508, 1509, 1509, 1500, 1499,
        1500, 1509, 1510, 1510, 1501, 1500, 1501, 1510, 1258, 1139, 1149, 1511,
        1511, 1502, 1139, 1502, 1511, 1512, 1512, 1503, 1502, 1503, 1512, 1513,
        1513, 1504, 1503, 1504, 1513, 1514, 1514, 1505, 1504, 1505, 1514, 1515,
        1515, 1506, 1505, 1506, 1515, 1516, 1516, 1507, 1506, 1507, 1516, 1517,
        1517, 1508, 1507, 1508, 1517, 1518, 1518, 1509, 1508, 1509, 1518, 1519,
        1519, 1510, 1509, 1510, 1519, 1258, 1149, 1159, 1520, 1520, 1511, 1149,
        1511, 1520, 1521, 1521, 1512, 1511, 1512, 1521, 1522, 1522, 1513, 1512,
        1513, 1522, 1523, 1523, 1514, 1513, 1514, 1523, 1524, 1524, 1515, 1514,
        1515, 1524, 1525, 1525, 1516, 1515, 1516, 1525, 1526, 1526, 1517, 1516,
        1517, 1526, 1527, 1527, 1518, 1517, 1518, 1527, 1528, 1528, 1519, 1518,
        1519, 1528, 1258, 1159, 1169, 1529, 1529, 1520, 1159, 1520, 1529, 1530,
        1530, 1521, 1520, 1521, 1530, 1531, 1531, 1522, 1521, 1522, 1531, 1532,
        1532, 1523, 1522, 1523, 1532, 1533, 1533, 1524, 1523, 1524, 1533, 1534,
        1534, 1525, 1524, 1525, 1534, 1535, 1535, 1526, 1525, 1526, 1535, 1536,
        1536, 1527, 1526, 1527, 1536, 1537, 1537, 1528, 1527, 1528, 1537, 1258,
        1169, 1179, 1538, 1538, 1529, 1169, 1529, 1538, 1539, 1539, 1530, 1529,
        1530, 1539, 1540, 1540, 1531, 1530, 1531, 1540, 1541, 1541, 1532, 1531,
        1532, 1541, 1542, 1542, 1533, 1532, 1533, 1542, 1543, 1543, 1534, 1533,
        1534, 1543, 1544, 1544, 1535, 1534, 1535, 1544, 1545, 1545, 1536, 1535,
        1536, 1545, 1546, 1546, 1537, 1536, 1537, 1546, 1258, 1179, 1189, 1547,
        1547, 1538, 1179, 1538, 1547, 1548, 1548, 1539, 1538, 1539, 1548, 1549,
        1549, 1540, 1539, 1540, 1549, 1550, 1550, 1541, 1540, 1541, 1550, 1551,
        1551, 1542, 1541, 1542, 1551, 1552, 1552, 1543, 1542, 1543, 1552, 1553,
        1553, 1544, 1543, 1544, 1553, 1554, 1554, 1545, 1544, 1545, 1554, 1555,
        1555, 1546, 1545, 1546, 1555, 1258, 1189, 1199, 1556, 1556, 1547, 1189,
        1547, 1556, 1557, 1557, 1548, 1547, 1548, 1557, 1558, 1558, 1549, 1548,
        1549, 1558, 1559, 1559, 1550, 1549, 1550, 1559, 1560, 1560, 1551, 1550,
        1551, 1560, 1561, 1561, 1552, 1551, 1552, 1561, 1562, 1562, 1553, 1552,
        1553, 1562, 1563, 1563, 1554, 1553, 1554, 1563, 1564, 1564, 1555, 1554,
        1555, 1564, 1258, 1199, 1209, 1565, 1565, 1556, 1199, 1556, 1565, 1566,
        1566, 1557, 1556, 1557, 1566, 1567, 1567, 1558, 1557, 1558, 1567, 1568,
        1568, 1559, 1558, 1559, 1568, 1569, 1569, 1560, 1559, 1560, 1569, 1570,
        1570, 1561, 1560, 1561, 1570, 1571, 1571, 1562, 1561, 1562, 1571, 1572,
        1572, 1563, 1562, 1563, 1572, 1573, 1573, 1564, 1563, 1564, 1573, 1258,
        1209, 1219, 1574, 1574, 1565, 1209, 1565, 1574, 1575, 1575, 1566, 1565,
        1566, 1575, 1576, 1576, 1567, 1566, 1567, 1576, 1577, 1577, 1568, 1567,
        1568, 1577, 1578, 1578, 1569, 1568, 1569, 1578, 1579, 1579, 1570, 1569,
        1570, 1579, 1580, 1580, 1571, 1570, 1571, 1580, 1581, 1581, 1572, 1571,
        1572, 1581, 1582, 1582, 1573, 1572, 1573, 1582, 1258, 1219, 1229, 1583,
        1583, 1574, 1219, 1574, 1583, 1584, 1584, 1575, 1574, 1575, 1584, 1585,
        1585, 1576, 1575, 1576, 1585, 1586, 1586, 1577, 1576, 1577, 1586, 1587,
        1587, 1578, 1577, 1578, 1587, 1588, 1588, 1579, 1578, 1579, 1588, 1589,
        1589, 1580, 1579, 1580, 1589, 1590, 1590, 1581, 1580, 1581, 1590, 1591,
        1591, 1582, 1581, 1582, 1591, 1258, 1229, 1239, 1592, 1592, 1583, 1229,
        1583, 1592, 1593, 1593, 1584, 1583, 1584, 1593, 1594, 1594, 1585, 1584,
        1585, 1594, 1595, 1595, 1586, 1585, 1586, 1595, 1596, 1596, 1587, 1586,
        1587, 1596, 1597, 1597, 1588, 1587, 1588, 1597, 1598, 1598, 1589, 1588,
        1589, 1598, 1599, 1599, 1590, 1589, 1590, 1599, 1600, 1600, 1591, 1590,
        1591, 1600, 1258, 1239,  859, 1241, 1241, 1592, 1239, 1592, 1241, 1243,
        1243, 1593, 1592, 1593, 1243, 1245, 1245, 1594, 1593, 1594, 1245, 1247,
        1247, 1595, 1594, 1595, 1247, 1249, 1249, 1596, 1595, 1596, 1249, 1251,
        1251, 1597, 1596, 1597, 1251, 1253, 1253, 1598, 1597, 1598, 1253, 1255,
        1255, 1599, 1598, 1599, 1255, 1257, 1257, 1600, 1599, 1600, 1257, 1258,
        1601, 1602, 1603, 1603, 1604, 1601, 1604, 1603, 1605, 1605, 1606, 1604,
        1606, 1605, 1607, 1607, 1608, 1606, 1608, 1607, 1609, 1609, 1610, 1608,
        1610, 1609, 1611, 1611, 1612, 1610, 1612, 1611, 1613, 1613, 1614, 1612,
        1614, 1613, 1615, 1615, 1616, 1614, 1616, 1615, 1617, 1617, 1618, 1616,
        1618, 1617, 1619, 1619, 1620, 1618, 1620, 1619, 1621, 1621, 1622, 1620,
        1602, 1623, 1624, 1624, 1603, 1602, 1603, 1624, 1625, 1625, 1605, 1603,
        1605, 1625, 1626, 1626, 1607, 1605, 1607, 1626, 1627, 1627, 1609, 1607,
        1609, 1627, 1628, 1628, 1611, 1609, 1611, 1628, 1629, 1629, 1613, 1611,
        1613, 1629, 1630, 1630, 1615, 1613, 1615, 1630, 1631, 1631, 1617, 1615,
        1617, 1631, 1632, 1632, 1619, 1617, 1619, 1632, 1633, 1633, 1621, 1619,
        1623, 1634, 1635, 1635, 1624, 1623, 1624, 1635, 1636, 1636, 1625, 1624,
        1625, 1636, 1637, 1637, 1626, 1625, 1626, 1637, 1638, 1638, 1627, 1626,
        1627, 1638, 1639, 1639, 1628, 1627, 1628, 1639, 1640, 1640, 1629, 1628,
        1629, 1640, 1641, 1641, 1630, 1629, 1630, 1641, 1642, 1642, 1631, 1630,
        1631, 1642, 1643, 1643, 1632, 1631, 1632, 1643, 1644, 1644, 1633, 1632,
        1634, 1645, 1646, 1646, 1635, 1634, 1635, 1646, 1647, 1647, 1636, 1635,
        1636, 1647, 1648, 1648, 1637, 1636, 1637, 1648, 1649, 1649, 1638, 1637,
        1638, 1649, 1650, 1650, 1639, 1638, 1639, 1650, 1651, 1651, 1640, 1639,
        1640, 1651, 1652, 1652, 1641, 1640, 1641, 1652, 1653, 1653, 1642, 1641,
        1642, 1653, 1654, 1654, 1643, 1642, 1643, 1654, 1655, 1655, 1644, 1643,
        1645, 1656, 1657, 1657, 1646, 1645, 1646, 1657, 1658, 1658, 1647, 1646,
        1647, 1658, 1659, 1659, 1648, 1647, 1648, 1659, 1660, 1660, 1649, 1648,
        1649, 1660, 1661, 1661, 1650, 1649, 1650, 1661, 1662, 1662, 1651, 1650,
        1651, 1662, 1663, 1663, 1652, 1651, 1652, 1663, 1664, 1664, 1653, 1652,
        1653, 1664, 1665, 1665, 1654, 1653, 1654, 1665, 1666, 1666, 1655, 1654,
        1656, 1667, 1668, 1668, 1657, 1656, 1657, 1668, 1669, 1669, 1658, 1657,
        1658, 1669, 1670, 1670, 1659, 1658, 1659, 1670, 1671, 1671, 1660, 1659,
        1660, 1671, 1672, 1672, 1661, 1660, 1661, 1672, 1673, 1673, 1662, 1661,
        1662, 1673, 1674, 1674, 1663, 1662, 1663, 1674, 1675, 1675, 1664, 1663,
        1664, 1675, 1676, 1676, 1665, 1664, 1665, 1676, 1677, 1677, 1666, 1665,
        1667, 1678, 1679, 1679, 1668, 1667, 1668, 1679, 1680, 1680, 1669, 1668,
        1669, 1680, 1681, 1681, 1670, 1669, 1670, 1681, 1682, 1682, 1671, 1670,
        1671, 1682, 1683, 1683, 1672, 1671, 1672, 1683, 1684, 1684, 1673, 1672,
        1673, 1684, 1685, 1685, 1674, 1673, 1674, 1685, 1686, 1686, 1675, 1674,
        1675, 1686, 1687, 1687, 1676, 1675, 1676, 1687, 1688, 1688, 1677, 1676,
        1678, 1689, 1690, 1690, 1679, 1678, 1679, 1690, 1691, 1691, 1680, 1679,
        1680, 1691, 1692, 1692, 1681, 1680, 1681, 1692, 1693, 1693, 1682, 1681,
        1682, 1693, 1694, 1694, 1683, 1682, 1683, 1694, 1695, 1695, 1684, 1683,
        1684, 1695, 1696, 1696, 1685, 1684, 1685, 1696, 1697, 1697, 1686, 1685,
        1686, 1697, 1698, 1698, 1687, 1686, 1687, 1698, 1699, 1699, 1688, 1687,
        1689, 1700, 1701, 1701, 1690, 1689, 1690, 1701, 1702, 1702, 1691, 1690,
        1691, 1702, 1703, 1703, 1692, 1691, 1692, 1703, 1704, 1704, 1693, 1692,
        1693, 1704, 1705, 1705, 1694, 1693, 1694, 1705, 1706, 1706, 1695, 1694,
        1695, 1706, 1707, 1707, 1696, 1695, 1696, 1707, 1708, 1708, 1697, 1696,
        1697, 1708, 1709, 1709, 1698, 1697, 1698, 1709, 1710, 1710, 1699, 1698,
        1700, 1711, 1712, 1712, 1701, 1700, 1701, 1712, 1713, 1713, 1702, 1701,
        1702, 1713, 1714, 1714, 1703, 1702, 1703, 1714, 1715, 1715, 1704, 1703,
        1704, 1715, 1716, 1716, 1705, 1704, 1705, 1716, 1717, 1717, 1706, 1705,
        1706, 1717, 1718, 1718, 1707, 1706, 1707, 1718, 1719, 1719, 1708, 1707,
        1708, 1719, 1720, 1720, 1709, 1708, 1709, 1720, 1721, 1721, 1710, 1709,
        1711, 1722, 1723, 1723, 1712, 1711, 1712, 1723, 1724, 1724, 1713, 1712,
        1713, 1724, 1725, 1725, 1714, 1713, 1714, 1725, 1726, 1726, 1715, 1714,
        1715, 1726, 1727, 1727, 1716, 1715, 1716, 1727, 1728, 1728, 1717, 1716,
        1717, 1728, 1729, 1729, 1718, 1717, 1718, 1729, 1730, 1730, 1719, 1718,
        1719, 1730, 1731, 1731, 1720, 1719, 1720, 1731, 1732, 1732, 1721, 1720,
        1722, 1733, 1734, 1734, 1723, 1722, 1723, 1734, 1735, 1735, 1724, 1723,
        1724, 1735, 1736, 1736, 1725, 1724, 1725, 1736, 1737, 1737, 1726, 1725,
        1726, 1737, 1738, 1738, 1727, 1726, 1727, 1738, 1739, 1739, 1728, 1727,
        1728, 1739, 1740, 1740, 1729, 1728, 1729, 1740, 1741, 1741, 1730, 1729,
        1730, 1741, 1742, 1742, 1731, 1730, 1731, 1742, 1743, 1743, 1732, 1731,
        1733, 1744, 1745, 1745, 1734, 1733, 1734, 1745, 1746, 1746, 1735, 1734,
        1735, 1746, 1747, 1747, 1736, 1735, 1736, 1747, 1748, 1748, 1737, 1736,
        1737, 1748, 1749, 1749, 1738, 1737, 1738, 1749, 1750, 1750, 1739, 1738,
        1739, 1750, 1751, 1751, 1740, 1739, 1740, 1751, 1752, 1752, 1741, 1740,
        1741, 1752, 1753, 1753, 1742, 1741, 1742, 1753, 1754, 1754, 1743, 1742,
        1744, 1755, 1756, 1756, 1745, 1744, 1745, 1756, 1757, 1757, 1746, 1745,
        1746, 1757, 1758, 1758, 1747, 1746, 1747, 1758, 1759, 1759, 1748, 1747,
        1748, 1759, 1760, 1760, 1749, 1748, 1749, 1760, 1761, 1761, 1750, 1749,
        1750, 1761, 1762, 1762, 1751, 1750, 1751, 1762, 1763, 1763, 1752, 1751,
        1752, 1763, 1764, 1764, 1753, 1752, 1753, 1764, 1765, 1765, 1754, 1753,
        1755, 1766, 1767, 1767, 1756, 1755, 1756, 1767, 1768, 1768, 1757, 1756,
        1757, 1768, 1769, 1769, 1758, 1757, 1758, 1769, 1770, 1770, 1759, 1758,
        1759, 1770, 1771, 1771, 1760, 1759, 1760, 1771, 1772, 1772, 1761, 1760,
        1761, 1772, 1773, 1773, 1762, 1761, 1762, 1773, 1774, 1774, 1763, 1762,
        1763, 1774, 1775, 1775, 1764, 1763, 1764, 1775, 1776, 1776, 1765, 1764,
        1766, 1777, 1778, 1778, 1767, 1766, 1767, 1778, 1779, 1779, 1768, 1767,
        1768, 1779, 1780, 1780, 1769, 1768, 1769, 1780, 1781, 1781, 1770, 1769,
        1770, 1781, 1782, 1782, 1771, 1770, 1771, 1782, 1783, 1783, 1772, 1771,
        1772, 1783, 1784, 1784, 1773, 1772, 1773, 1784, 1785, 1785, 1774, 1773,
        1774, 1785, 1786, 1786, 1775, 1774, 1775, 1786, 1787, 1787, 1776, 1775,
        1777, 1788, 1789, 1789, 1778, 1777, 1778, 1789, 1790, 1790, 1779, 1778,
        1779, 1790, 1791, 1791, 1780, 1779, 1780, 1791, 1792, 1792, 1781, 1780,
        1781, 1792, 1793, 1793, 1782, 1781, 1782, 1793, 1794, 1794, 1783, 1782,
        1783, 1794, 1795, 1795, 1784, 1783, 1784, 1795, 1796, 1796, 1785, 1784,
        1785, 1796, 1797, 1797, 1786, 1785, 1786, 1797, 1798, 1798, 1787, 1786,
        1788, 1799, 1800, 1800, 1789, 1788, 1789, 1800, 1801, 1801, 1790, 1789,
        1790, 1801, 1802, 1802, 1791, 1790, 1791, 1802, 1803, 1803, 1792, 1791,
        1792, 1803, 1804, 1804, 1793, 1792, 1793, 1804, 1805, 1805, 1794, 1793,
        1794, 1805, 1806, 1806, 1795, 1794, 1795, 1806, 1807, 1807, 1796, 1795,
        1796, 1807, 1808, 1808, 1797, 1796, 1797, 1808, 1809, 1809, 1798, 1797,
        1799, 1810, 1811, 1811, 1800, 1799, 1800, 1811, 1812, 1812, 1801, 1800,
        1801, 1812, 1813, 1813, 1802, 1801, 1802, 1813, 1814, 1814, 1803, 1802,
        1803, 1814, 1815, 1815, 1804, 1803, 1804, 1815, 1816, 1816, 1805, 1804,
        1805, 1816, 1817, 1817, 1806, 1805, 1806, 1817, 1818, 1818, 1807, 1806,
        1807, 1818, 1819, 1819, 1808, 1807, 1808, 1819, 1820, 1820, 1809, 1808,
        1810, 1601, 1604, 1604, 1811, 1810, 1811, 1604, 1606, 1606, 1812, 1811,
        1812, 1606, 1608, 1608, 1813, 1812, 1813, 1608, 1610, 1610, 1814, 1813,
        1814, 1610, 1612, 1612, 1815, 1814, 1815, 1612, 1614, 1614, 1816, 1815,
        1816, 1614, 1616, 1616, 1817, 1816, 1817, 1616, 1618, 1618, 1818, 1817,
        1818, 1618, 1620, 1620, 1819, 1818, 1819, 1620, 1622, 1622, 1820, 1819,
        1622, 1621, 1821, 1821, 1822, 1622, 1822, 1821, 1823, 1823, 1824, 1822,
        1824, 1823, 1825, 1825, 1826, 1824, 1826, 1825, 1827, 1827, 1828, 1826,
        1828, 1827, 1829, 1829, 1830, 1828, 1830, 1829, 1831, 1831, 1832, 1830,
        1832, 1831, 1833, 1833, 1834, 1832, 1834, 1833, 1835, 1835, 1836, 1834,
        1836, 1835, 1837, 1837, 1838, 1836, 1838, 1837, 1839, 1839, 1840, 1838,
        1621, 1633, 1841, 1841, 1821, 1621, 1821, 1841, 1842, 1842, 1823, 1821,
        1823, 1842, 1843, 1843, 1825, 1823, 1825, 1843, 1844, 1844, 1827, 1825,
        1827, 1844, 1845, 1845, 1829, 1827, 1829, 1845, 1846, 1846, 1831, 1829,
        1831, 1846, 1847, 1847, 1833, 1831, 1833, 1847, 1848, 1848, 1835, 1833,
        1835, 1848, 1849, 1849, 1837, 1835, 1837, 1849, 1850, 1850, 1839, 1837,
        1633, 1644, 1851, 1851, 1841, 1633, 1841, 1851, 1852, 1852, 1842, 1841,
        1842, 1852, 1853, 1853, 1843, 1842, 1843, 1853, 1854, 1854, 1844, 1843,
        1844, 1854, 1855, 1855, 1845, 1844, 1845, 1855, 1856, 1856, 1846, 1845,
        1846, 1856, 1857, 1857, 1847, 1846, 1847, 1857, 1858, 1858, 1848, 1847,
        1848, 1858, 1859, 1859, 1849, 1848, 1849, 1859, 1860, 1860, 1850, 1849,
        1644, 1655, 1861, 1861, 1851, 1644, 1851, 1861, 1862, 1862, 1852, 1851,
        1852, 1862, 1863, 1863, 1853, 1852, 1853, 1863, 1864, 1864, 1854, 1853,
        1854, 1864, 1865, 1865, 1855, 1854, 1855, 1865, 1866, 1866, 1856, 1855,
        1856, 1866, 1867, 1867, 1857, 1856, 1857, 1867, 1868, 1868, 1858, 1857,
        1858, 1868, 1869, 1869, 1859, 1858, 1859, 1869, 1870, 1870, 1860, 1859,
        1655, 1666, 1871, 1871, 1861, 1655, 1861, 1871, 1872, 1872, 1862, 1861,
        1862, 1872, 1873, 1873, 1863, 1862, 1863, 1873, 1874, 1874, 1864, 1863,
        1864, 1874, 1875, 1875, 1865, 1864, 1865, 1875, 1876, 1876, 1866, 1865,
        1866, 1876, 1877, 1877, 1867, 1866, 1867, 1877, 1878, 1878, 1868, 1867,
        1868, 1878, 1879, 1879, 1869, 1868, 1869, 1879, 1880, 1880, 1870, 1869,
        1666, 1677, 1881, 1881, 1871, 1666, 1871, 1881, 1882, 1882, 1872, 1871,
        1872, 1882, 1883, 1883, 1873, 1872, 1873, 1883, 1884, 1884, 1874, 1873,
        1874, 1884, 1885, 1885, 1875, 1874, 1875, 1885, 1886, 1886, 1876, 1875,
        1876, 1886, 1887, 1887, 1877, 1876, 1877, 1887, 1888, 1888, 1878, 1877,
        1878, 1888, 1889, 1889, 1879, 1878, 1879, 1889, 1890, 1890, 1880, 1879,
        1677, 1688, 1891, 1891, 1881, 1677, 1881, 1891, 1892, 1892, 1882, 1881,
        1882, 1892, 1893, 1893, 1883, 1882, 1883, 1893, 1894, 1894, 1884, 1883,
        1884, 1894, 1895, 1895, 1885, 1884, 1885, 1895, 1896, 1896, 1886, 1885,
        1886, 1896, 1897, 1897, 1887, 1886, 1887, 1897, 1898, 1898, 1888, 1887,
        1888, 1898, 1899, 1899, 1889, 1888, 1889, 1899, 1900, 1900, 1890, 1889,
        1688, 1699, 1901, 1901, 1891, 1688, 1891, 1901, 1902, 1902, 1892, 1891,
        1892, 1902, 1903, 1903, 1893, 1892, 1893, 1903, 1904, 1904, 1894, 1893,
        1894, 1904, 1905, 1905, 1895, 1894, 1895, 1905, 1906, 1906, 1896, 1895,
        1896, 1906, 1907, 1907, 1897, 1896, 1897, 1907, 1908, 1908, 1898, 1897,
        1898, 1908, 1909, 1909, 1899, 1898, 1899, 1909, 1910, 1910, 1900, 1899,
        1699, 1710, 1911, 1911, 1901, 1699, 1901, 1911, 1912, 1912, 1902, 1901,
        1902, 1912, 1913, 1913, 1903, 1902, 1903, 1913, 1914, 1914, 1904, 1903,
        1904, 1914, 1915, 1915, 1905, 1904, 1905, 1915, 1916, 1916, 1906, 1905,
        1906, 1916, 1917, 1917, 1907, 1906, 1907, 1917, 1918, 1918, 1908, 1907,
        1908, 1918, 1919, 1919, 1909, 1908, 1909, 1919, 1920, 1920, 1910, 1909,
        1710, 1721, 1921, 1921, 1911, 1710, 1911, 1921, 1922, 1922, 1912, 1911,
        1912, 1922, 1923, 1923, 1913, 1912, 1913, 1923, 1924, 1924, 1914, 1913,
        1914, 1924, 1925, 1925, 1915, 1914, 1915, 1925, 1926, 1926, 1916, 1915,
        1916, 1926, 1927, 1927, 1917, 1916, 1917, 1927, 1928, 1928, 1918, 1917,
        1918, 1928, 1929, 1929, 1919, 1918, 1919, 1929, 1930, 1930, 1920, 1919,
        1721, 1732, 1931, 1931, 1921, 1721, 1921, 1931, 1932, 1932, 1922, 1921,
        1922, 1932, 1933, 1933, 1923, 1922, 1923, 1933, 1934, 1934, 1924, 1923,
        1924, 1934, 1935, 1935, 1925, 1924, 1925, 1935, 1936, 1936, 1926, 1925,
        1926, 1936, 1937, 1937, 1927, 1926, 1927, 1937, 1938, 1938, 1928, 1927,
        1928, 1938, 1939, 1939, 1929, 1928, 1929, 1939, 1940, 1940, 1930, 1929,
        1732, 1743, 1941, 1941, 1931, 1732, 1931, 1941, 1942, 1942, 1932, 1931,
        1932, 1942, 1943, 1943, 1933, 1932, 1933, 1943, 1944, 1944, 1934, 1933,
        1934, 1944, 1945, 1945, 1935, 1934, 1935, 1945, 1946, 1946, 1936, 1935,
        1936, 1946, 1947, 1947, 1937, 1936, 1937, 1947, 1948, 1948, 1938, 1937,
        1938, 1948, 1949, 1949, 1939, 1938, 1939, 1949, 1950, 1950, 1940, 1939,
        1743, 1754, 1951, 1951, 1941, 1743, 1941, 1951, 1952, 1952, 1942, 1941,
        1942, 1952, 1953, 1953, 1943, 1942, 1943, 1953, 1954, 1954, 1944, 1943,
        1944, 1954, 1955, 1955, 1945, 1944, 1945, 1955, 1956, 1956, 1946, 1945,
        1946, 1956, 1957, 1957, 1947, 1946, 1947, 1957, 1958, 1958, 1948, 1947,
        1948, 1958, 1959, 1959, 1949, 1948, 1949, 1959, 1960, 1960, 1950, 1949,
        1754, 1765, 1961, 1961, 1951, 1754, 1951, 1961, 1962, 1962, 1952, 1951,
        1952, 1962, 1963, 1963, 1953, 1952, 1953, 1963, 1964, 1964, 1954, 1953,
        1954, 1964, 1965, 1965, 1955, 1954, 1955, 1965, 1966, 1966, 1956, 1955,
        1956, 1966, 1967, 1967, 1957, 1956, 1957, 1967, 1968, 1968, 1958, 1957,
        1958, 1968, 1969, 1969, 1959, 1958, 1959, 1969, 1970, 1970, 1960, 1959,
        1765, 1776, 1971, 1971, 1961, 1765, 1961, 1971, 1972, 1972, 1962, 1961,
        1962, 1972, 1973, 1973, 1963, 1962, 1963, 1973, 1974, 1974, 1964, 1963,
        1964, 1974, 1975, 1975, 1965, 1964, 1965, 1975, 1976, 1976, 1966, 1965,
        1966, 1976, 1977, 1977, 1967, 1966, 1967, 1977, 1978, 1978, 1968, 1967,
        1968, 1978, 1979, 1979, 1969, 1968, 1969, 1979, 1980, 1980, 1970, 1969,
        1776, 1787, 1981, 1981, 1971, 1776, 1971, 1981, 1982, 1982, 1972, 1971,
        1972, 1982, 1983, 1983, 1973, 1972, 1973, 1983, 1984, 1984, 1974, 1973,
        1974, 1984, 1985, 1985, 1975, 1974, 1975, 1985, 1986, 1986, 1976, 1975,
        1976, 1986, 1987, 1987, 1977, 1976, 1977, 1987, 1988, 1988, 1978, 1977,
        1978, 1988, 1989, 1989, 1979, 1978, 1979, 1989, 1990, 1990, 1980, 1979,
        1787, 1798, 1991, 1991, 1981, 1787, 1981, 1991, 1992, 1992, 1982, 1981,
        1982, 1992, 1993, 1993, 1983, 1982, 1983, 1993, 1994, 1994, 1984, 1983,
        1984, 1994, 1995, 1995, 1985, 1984, 1985, 1995, 1996, 1996, 1986, 1985,
        1986, 1996, 1997, 1997, 1987, 1986, 1987, 1997, 1998, 1998, 1988, 1987,
        1988, 1998, 1999, 1999, 1989, 1988, 1989, 1999, 2000, 2000, 1990, 1989,
        1798, 1809, 2001, 2001, 1991, 1798, 1991, 2001, 2002, 2002, 1992, 1991,
        1992, 2002, 2003, 2003, 1993, 1992, 1993, 2003, 2004, 2004, 1994, 1993,
        1994, 2004, 2005, 2005, 1995, 1994, 1995, 2005, 2006, 2006, 1996, 1995,
        1996, 2006, 2007, 2007, 1997, 1996, 1997, 2007, 2008, 2008, 1998, 1997,
        1998, 2008, 2009, 2009, 1999, 1998, 1999, 2009, 2010, 2010, 2000, 1999,
        1809, 1820, 2011, 2011, 2001, 1809, 2001, 2011, 2012, 2012, 2002, 2001,
        2002, 2012, 2013, 2013, 2003, 2002, 2003, 2013, 2014, 2014, 2004, 2003,
        2004, 2014, 2015, 2015, 2005, 2004, 2005, 2015, 2016, 2016, 2006, 2005,
        2006, 2016, 2017, 2017, 2007, 2006, 2007, 2017, 2018, 2018, 2008, 2007,
        2008, 2018, 2019, 2019, 2009, 2008, 2009, 2019, 2020, 2020, 2010, 2009,
        1820, 1622, 1822, 1822, 2011, 1820, 2011, 1822, 1824, 1824, 2012, 2011,
        2012, 1824, 1826, 1826, 2013, 2012, 2013, 1826, 1828, 1828, 2014, 2013,
        2014, 1828, 1830, 1830, 2015, 2014, 2015, 1830, 1832, 1832, 2016, 2015,
        2016, 1832, 1834, 1834, 2017, 2016, 2017, 1834, 1836, 1836, 2018, 2017,
        2018, 1836, 1838, 1838, 2019, 2018, 2019, 1838, 1840, 1840, 2020, 2019,
        2021, 2022, 2023, 2023, 2024, 2021, 2024, 2023, 2025, 2025, 2026, 2024,
        2026, 2025, 2027, 2027, 2028, 2026, 2028, 2027, 2029, 2029, 2030, 2028,
        2030, 2029, 2031, 2031, 2032, 2030, 2032, 2031, 2033, 2033, 2034, 2032,
        2034, 2033, 2035, 2035, 2036, 2034, 2036, 2035, 2037, 2037, 2038, 2036,
        2038, 2037, 2039, 2039, 2040, 2038, 2040, 2039, 2041, 2041, 2042, 2040,
        2022, 2043, 2044, 2044, 2023, 2022, 2023, 2044, 2045, 2045, 2025, 2023,
        2025, 2045, 2046, 2046, 2027, 2025, 2027, 2046, 2047, 2047, 2029, 2027,
        2029, 2047, 2048, 2048, 2031, 2029, 2031, 2048, 2049, 2049, 2033, 2031,
        2033, 2049, 2050, 2050, 2035, 2033, 2035, 2050, 2051, 2051, 2037, 2035,
        2037, 2051, 2052, 2052, 2039, 2037, 2039, 2052, 2053, 2053, 2041, 2039,
        2043, 2054, 2055, 2055, 2044, 2043, 2044, 2055, 2056, 2056, 2045, 2044,
        2045, 2056, 2057, 2057, 2046, 2045, 2046, 2057, 2058, 2058, 2047, 2046,
        2047, 2058, 2059, 2059, 2048, 2047, 2048, 2059, 2060, 2060, 2049, 2048,
        2049, 2060, 2061, 2061, 2050, 2049, 2050, 2061, 2062, 2062, 2051, 2050,
        2051, 2062, 2063, 2063, 2052, 2051, 2052, 2063, 2064, 2064, 2053, 2052,
        2054, 2065, 2066, 2066, 2055, 2054, 2055, 2066, 2067, 2067, 2056, 2055,
        2056, 2067, 2068, 2068, 2057, 2056, 2057, 2068, 2069, 2069, 2058, 2057,
        2058, 2069, 2070, 2070, 2059, 2058, 2059, 2070, 2071, 2071, 2060, 2059,
        2060, 2071, 2072, 2072, 2061, 2060, 2061, 2072, 2073, 2073, 2062, 2061,
        2062, 2073, 2074, 2074, 2063, 2062, 2063, 2074, 2075, 2075, 2064, 2063,
        2065, 2076, 2077, 2077, 2066, 2065, 2066, 2077, 2078, 2078, 2067, 2066,
        2067, 2078, 2079, 2079, 2068, 2067, 2068, 2079, 2080, 2080, 2069, 2068,
        2069, 2080, 2081, 2081, 2070, 2069, 2070, 2081, 2082, 2082, 2071, 2070,
        2071, 2082, 2083, 2083, 2072, 2071, 2072, 2083, 2084, 2084, 2073, 2072,
        2073, 2084, 2085, 2085, 2074, 2073, 2074, 2085, 2086, 2086, 2075, 2074,
        2076, 2087, 2088, 2088, 2077, 2076, 2077, 2088, 2089, 2089, 2078, 2077,
        2078, 2089, 2090, 2090, 2079, 2078, 2079, 2090, 2091, 2091, 2080, 2079,
        2080, 2091, 2092, 2092, 2081, 2080, 2081, 2092, 2093, 2093, 2082, 2081,
        2082, 2093, 2094, 2094, 2083, 2082, 2083, 2094, 2095, 2095, 2084, 2083,
        2084, 2095, 2096, 2096, 2085, 2084, 2085, 2096, 2097, 2097, 2086, 2085,
        2087, 2098, 2099, 2099, 2088, 2087, 2088, 2099, 2100, 2100, 2089, 2088,
        2089, 2100, 2101, 2101, 2090, 2089, 2090, 2101, 2102, 2102, 2091, 2090,
        2091, 2102, 2103, 2103, 2092, 2091, 2092, 2103, 2104, 2104, 2093, 2092,
        2093, 2104, 2105, 2105, 2094, 2093, 2094, 2105, 2106, 2106, 2095, 2094,
        2095, 2106, 2107, 2107, 2096, 2095, 2096, 2107, 2108, 2108, 2097, 2096,
        2098, 2109, 2110, 2110, 2099, 2098, 2099, 2110, 2111, 2111, 2100, 2099,
        2100, 2111, 2112, 2112, 2101, 2100, 2101, 2112, 2113, 2113, 2102, 2101,
        2102, 2113, 2114, 2114, 2103, 2102, 2103, 2114, 2115, 2115, 2104, 2103,
        2104, 2115, 2116, 2116, 2105, 2104, 2105, 2116, 2117, 2117, 2106, 2105,
        2106, 2117, 2118, 2118, 2107, 2106, 2107, 2118, 2119, 2119, 2108, 2107,
        2109, 2120, 2121, 2121, 2110, 2109, 2110, 2121, 2122, 2122, 2111, 2110,
        2111, 2122, 2123, 2123, 2112, 2111, 2112, 2123, 2124, 2124, 2113, 2112,
        2113, 2124, 2125, 2125, 2114, 2113, 2114, 2125, 2126, 2126, 2115, 2114,
        2115, 2126, 2127, 2127, 2116, 2115, 2116, 2127, 2128, 2128, 2117, 2116,
        2117, 2128, 2129, 2129, 2118, 2117, 2118, 2129, 2130, 2130, 2119, 2118,
        2120, 2131, 2132, 2132, 2121, 2120, 2121, 2132, 2133, 2133, 2122, 2121,
        2122, 2133, 2134, 2134, 2123, 2122, 2123, 2134, 2135, 2135, 2124, 2123,
        2124, 2135, 2136, 2136, 2125, 2124, 2125, 2136, 2137, 2137, 2126, 2125,
        2126, 2137, 2138, 2138, 2127, 2126, 2127, 2138, 2139, 2139, 2128, 2127,
        2128, 2139, 2140, 2140, 2129, 2128, 2129, 2140, 2141, 2141, 2130, 2129,
        2131, 2142, 2143, 2143, 2132, 2131, 2132, 2143, 2144, 2144, 2133, 2132,
        2133, 2144, 2145, 2145, 2134, 2133, 2134, 2145, 2146, 2146, 2135, 2134,
        2135, 2146, 2147, 2147, 2136, 2135, 2136, 2147, 2148, 2148, 2137, 2136,
        2137, 2148, 2149, 2149, 2138, 2137, 2138, 2149, 2150, 2150, 2139, 2138,
        2139, 2150, 2151, 2151, 2140, 2139, 2140, 2151, 2152, 2152, 2141, 2140,
        2142, 2153, 2154, 2154, 2143, 2142, 2143, 2154, 2155, 2155, 2144, 2143,
        2144, 2155, 2156, 2156, 2145, 2144, 2145, 2156, 2157, 2157, 2146, 2145,
        2146, 2157, 2158, 2158, 2147, 2146, 2147, 2158, 2159, 2159, 2148, 2147,
        2148, 2159, 2160, 2160, 2149, 2148, 2149, 2160, 2161, 2161, 2150, 2149,
        2150, 2161, 2162, 2162, 2151, 2150, 2151, 2162, 2163, 2163, 2152, 2151,
        2153, 2164, 2165, 2165, 2154, 2153, 2154, 2165, 2166, 2166, 2155, 2154,
        2155, 2166, 2167, 2167, 2156, 2155, 2156, 2167, 2168, 2168, 2157, 2156,
        2157, 2168, 2169, 2169, 2158, 2157, 2158, 2169, 2170, 2170, 2159, 2158,
        2159, 2170, 2171, 2171, 2160, 2159, 2160, 2171, 2172, 2172, 2161, 2160,
        2161, 2172, 2173, 2173, 2162, 2161, 2162, 2173, 2174, 2174, 2163, 2162,
        2164, 2175, 2176, 2176, 2165, 2164, 2165, 2176, 2177, 2177, 2166, 2165,
        2166, 2177, 2178, 2178, 2167, 2166, 2167, 2178, 2179, 2179, 2168, 2167,
        2168, 2179, 2180, 2180, 2169, 2168, 2169, 2180, 2181, 2181, 2170, 2169,
        2170, 2181, 2182, 2182, 2171, 2170, 2171, 2182, 2183, 2183, 2172, 2171,
        2172, 2183, 2184, 2184, 2173, 2172, 2173, 2184, 2185, 2185, 2174, 2173,
        2175, 2186, 2187, 2187, 2176, 2175, 2176, 2187, 2188, 2188, 2177, 2176,
        2177, 2188, 2189, 2189, 2178, 2177, 2178, 2189, 2190, 2190, 2179, 2178,
        2179, 2190, 2191, 2191, 2180, 2179, 2180, 2191, 2192, 2192, 2181, 2180,
        2181, 2192, 2193, 2193, 2182, 2181, 2182, 2193, 2194, 2194, 2183, 2182,
        2183, 2194, 2195, 2195, 2184, 2183, 2184, 2195, 2196, 2196, 2185, 2184,
        2186, 2197, 2198, 2198, 2187, 2186, 2187, 2198, 2199, 2199, 2188, 2187,
        2188, 2199, 2200, 2200, 2189, 2188, 2189, 2200, 2201, 2201, 2190, 2189,
        2190, 2201, 2202, 2202, 2191, 2190, 2191, 2202, 2203, 2203, 2192, 2191,
        2192, 2203, 2204, 2204, 2193, 2192, 2193, 2204, 2205, 2205, 2194, 2193,
        2194, 2205, 2206, 2206, 2195, 2194, 2195, 2206, 2207, 2207, 2196, 2195,
        2197, 2208, 2209, 2209, 2198, 2197, 2198, 2209, 2210, 2210, 2199, 2198,
        2199, 2210, 2211, 2211, 2200, 2199, 2200, 2211, 2212, 2212, 2201, 2200,
        2201, 2212, 2213, 2213, 2202, 2201, 2202, 2213, 2214, 2214, 2203, 2202,
        2203, 2214, 2215, 2215, 2204, 2203, 2204, 2215, 2216, 2216, 2205, 2204,
        2205, 2216, 2217, 2217, 2206, 2205, 2206, 2217, 2218, 2218, 2207, 2206,
        2208, 2219, 2220, 2220, 2209, 2208, 2209, 2220, 2221, 2221, 2210, 2209,
        2210, 2221, 2222, 2222, 2211, 2210, 2211, 2222, 2223, 2223, 2212, 2211,
        2212, 2223, 2224, 2224, 2213, 2212, 2213, 2224, 2225, 2225, 2214, 2213,
        2214, 2225, 2226, 2226, 2215, 2214, 2215, 2226, 2227, 2227, 2216, 2215,
        2216, 2227, 2228, 2228, 2217, 2216, 2217, 2228, 2229, 2229, 2218, 2217,
        2219, 2230, 2231, 2231, 2220, 2219, 2220, 2231, 2232, 2232, 2221, 2220,
        2221, 2232, 2233, 2233, 2222, 2221, 2222, 2233, 2234, 2234, 2223, 2222,
        2223, 2234, 2235, 2235, 2224, 2223, 2224, 2235, 2236, 2236, 2225, 2224,
        2225, 2236, 2237, 2237, 2226, 2225, 2226, 2237, 2238, 2238, 2227, 2226,
        2227, 2238, 2239, 2239, 2228, 2227, 2228, 2239, 2240, 2240, 2229, 2228,
        2230, 2021, 2024, 2024, 2231, 2230, 2231, 2024, 2026, 2026, 2232, 2231,
        2232, 2026, 2028, 2028, 2233, 2232, 2233, 2028, 2030, 2030, 2234, 2233,
        2234, 2030, 2032, 2032, 2235, 2234, 2235, 2032, 2034, 2034, 2236, 2235,
        2236, 2034, 2036, 2036, 2237, 2236, 2237, 2036, 2038, 2038, 2238, 2237,
        2238, 2038, 2040, 2040, 2239, 2238, 2239, 2040, 2042, 2042, 2240, 2239,
        2042, 2041, 2241, 2241, 2242, 2042, 2242, 2241, 2243, 2243, 2244, 2242,
        2244, 2243, 2245, 2245, 2246, 2244, 2246, 2245, 2247, 2247, 2248, 2246,
        2248, 2247, 2249, 2249, 2250, 2248, 2250, 2249, 2251, 2251, 2252, 2250,
        2252, 2251, 2253, 2253, 2254, 2252, 2254, 2253, 2255, 2255, 2256, 2254,
        2256, 2255, 2257, 2257, 2258, 2256, 2258, 2257, 2259, 2259, 2260, 2258,
        2041, 2053, 2261, 2261, 2241, 2041, 2241, 2261, 2262, 2262, 2243, 2241,
        2243, 2262, 2263, 2263, 2245, 2243, 2245, 2263, 2264, 2264, 2247, 2245,
        2247, 2264, 2265, 2265, 2249, 2247, 2249, 2265, 2266, 2266, 2251, 2249,
        2251, 2266, 2267, 2267, 2253, 2251, 2253, 2267, 2268, 2268, 2255, 2253,
        2255, 2268, 2269, 2269, 2257, 2255, 2257, 2269, 2270, 2270, 2259, 2257,
        2053, 2064, 2271, 2271, 2261, 2053, 2261, 2271, 2272, 2272, 2262, 2261,
        2262, 2272, 2273, 2273, 2263, 2262, 2263, 2273, 2274, 2274, 2264, 2263,
        2264, 2274, 2275, 2275, 2265, 2264, 2265, 2275, 2276, 2276, 2266, 2265,
        2266, 2276, 2277, 2277, 2267, 2266, 2267, 2277, 2278, 2278, 2268, 2267,
        2268, 2278, 2279, 2279, 2269, 2268, 2269, 2279, 2280, 2280, 2270, 2269,
        2064, 2075, 2281, 2281, 2271, 2064, 2271, 2281, 2282, 2282, 2272, 2271,
        2272, 2282, 2283, 2283, 2273, 2272, 2273, 2283, 2284, 2284, 2274, 2273,
        2274, 2284, 2285, 2285, 2275, 2274, 2275, 2285, 2286, 2286, 2276, 2275,
        2276, 2286, 2287, 2287, 2277, 2276, 2277, 2287, 2288, 2288, 2278, 2277,
        2278, 2288, 2289, 2289, 2279, 2278, 2279, 2289, 2290, 2290, 2280, 2279,
        2075, 2086, 2291, 2291, 2281, 2075, 2281, 2291, 2292, 2292, 2282, 2281,
        2282, 2292, 2293, 2293, 2283, 2282, 2283, 2293, 2294, 2294, 2284, 2283,
        2284, 2294, 2295, 2295, 2285, 2284, 2285, 2295, 2296, 2296, 2286, 2285,
        2286, 2296, 2297, 2297, 2287, 2286, 2287, 2297, 2298, 2298, 2288, 2287,
        2288, 2298, 2299, 2299, 2289, 2288, 2289, 2299, 2300, 2300, 2290, 2289,
        2086, 2097, 2301, 2301, 2291, 2086, 2291, 2301, 2302, 2302, 2292, 2291,
        2292, 2302, 2303, 2303, 2293, 2292, 2293, 2303, 2304, 2304, 2294, 2293,
        2294, 2304, 2305, 2305, 2295, 2294, 2295, 2305, 2306, 2306, 2296, 2295,
        2296, 2306, 2307, 2307, 2297, 2296, 2297, 2307, 2308, 2308, 2298, 2297,
        2298, 2308, 2309, 2309, 2299, 2298, 2299, 2309, 2310, 2310, 2300, 2299,
        2097, 2108, 2311, 2311, 2301, 2097, 2301, 2311, 2312, 2312, 2302, 2301,
        2302, 2312, 2313, 2313, 2303, 2302, 2303, 2313, 2314, 2314, 2304, 2303,
        2304, 2314, 2315, 2315, 2305, 2304, 2305, 2315, 2316, 2316, 2306, 2305,
        2306, 2316, 2317, 2317, 2307, 2306, 2307, 2317, 2318, 2318, 2308, 2307,
        2308, 2318, 2319, 2319, 2309, 2308, 2309, 2319, 2320, 2320, 2310, 2309,
        2108, 2119, 2321, 2321, 2311, 2108, 2311, 2321, 2322, 2322, 2312, 2311,
        2312, 2322, 2323, 2323, 2313, 2312, 2313, 2323, 2324, 2324, 2314, 2313,
        2314, 2324, 2325, 2325, 2315, 2314, 2315, 2325, 2326, 2326, 2316, 2315,
        2316, 2326, 2327, 2327, 2317, 2316, 2317, 2327, 2328, 2328, 2318, 2317,
        2318, 2328, 2329, 2329, 2319, 2318, 2319, 2329, 2330, 2330, 2320, 2319,
        2119, 2130, 2331, 2331, 2321, 2119, 2321, 2331, 2332, 2332, 2322, 2321,
        2322, 2332, 2333, 2333, 2323, 2322, 2323, 2333, 2334, 2334, 2324, 2323,
        2324, 2334, 2335, 2335, 2325, 2324, 2325, 2335, 2336, 2336, 2326, 2325,
        2326, 2336, 2337, 2337, 2327, 2326, 2327, 2337, 2338, 2338, 2328, 2327,
        2328, 2338, 2339, 2339, 2329, 2328, 2329, 2339, 2340, 2340, 2330, 2329,
        2130, 2141, 2341, 2341, 2331, 2130, 2331, 2341, 2342, 2342, 2332, 2331,
        2332, 2342, 2343, 2343, 2333, 2332, 2333, 2343, 2344, 2344, 2334, 2333,
        2334, 2344, 2345, 2345, 2335, 2334, 2335, 2345, 2346, 2346, 2336, 2335,
        2336, 2346, 2347, 2347, 2337, 2336, 2337, 2347, 2348, 2348, 2338, 2337,
        2338, 2348, 2349, 2349, 2339, 2338, 2339, 2349, 2350, 2350, 2340, 2339,
        2141, 2152, 2351, 2351, 2341, 2141, 2341, 2351, 2352, 2352, 2342, 2341,
        2342, 2352, 2353, 2353, 2343, 2342, 2343, 2353, 2354, 2354, 2344, 2343,
        2344, 2354, 2355, 2355, 2345, 2344, 2345, 2355, 2356, 2356, 2346, 2345,
        2346, 2356, 2357, 2357, 2347, 2346, 2347, 2357, 2358, 2358, 2348, 2347,
        2348, 2358, 2359, 2359, 2349, 2348, 2349, 2359, 2360, 2360, 2350, 2349,
        2152, 2163, 2361, 2361, 2351, 2152, 2351, 2361, 2362, 2362, 2352, 2351,
        2352, 2362, 2363, 2363, 2353, 2352, 2353, 2363, 2364, 2364, 2354, 2353,
        2354, 2364, 2365, 2365, 2355, 2354, 2355, 2365, 2366, 2366, 2356, 2355,
        2356, 2366, 2367, 2367, 2357, 2356, 2357, 2367, 2368, 2368, 2358, 2357,
        2358, 2368, 2369, 2369, 2359, 2358, 2359, 2369, 2370, 2370, 2360, 2359,
        2163, 2174, 2371, 2371, 2361, 2163, 2361, 2371, 2372, 2372, 2362, 2361,
        2362, 2372, 2373, 2373, 2363, 2362, 2363, 2373, 2374, 2374, 2364, 2363,
        2364, 2374, 2375, 2375, 2365, 2364, 2365, 2375, 2376, 2376, 2366, 2365,
        2366, 2376, 2377, 2377, 2367, 2366, 2367, 2377, 2378, 2378, 2368, 2367,
        2368, 2378, 2379, 2379, 2369, 2368, 2369, 2379, 2380, 2380, 2370, 2369,
        2174, 2185, 2381, 2381, 2371, 2174, 2371, 2381, 2382, 2382, 2372, 2371,
        2372, 2382, 2383, 2383, 2373, 2372, 2373, 2383, 2384, 2384, 2374, 2373,
        2374, 2384, 2385, 2385, 2375, 2374, 2375, 2385, 2386, 2386, 2376, 2375,
        2376, 2386, 2387, 2387, 2377, 2376, 2377, 2387, 2388, 2388, 2378, 2377,
        2378, 2388, 2389, 2389, 2379, 2378, 2379, 2389, 2390, 2390, 2380, 2379,
        2185, 2196, 2391, 2391, 2381, 2185, 2381, 2391, 2392, 2392, 2382, 2381,
        2382, 2392, 2393, 2393, 2383, 2382, 2383, 2393, 2394, 2394, 2384, 2383,
        2384, 2394, 2395, 2395, 2385, 2384, 2385, 2395, 2396, 2396, 2386, 2385,
        2386, 2396, 2397, 2397, 2387, 2386, 2387, 2397, 2398, 2398, 2388, 2387,
        2388, 2398, 2399, 2399, 2389, 2388, 2389, 2399, 2400, 2400, 2390, 2389,
        2196, 2207, 2401, 2401, 2391, 2196, 2391, 2401, 2402, 2402, 2392, 2391,
        2392, 2402, 2403, 2403, 2393, 2392, 2393, 2403, 2404, 2404, 2394, 2393,
        2394, 2404, 2405, 2405, 2395, 2394, 2395, 2405, 2406, 2406, 2396, 2395,
        2396, 2406, 2407, 2407, 2397, 2396, 2397, 2407, 2408, 2408, 2398, 2397,
        2398, 2408, 2409, 2409, 2399, 2398, 2399, 2409, 2410, 2410, 2400, 2399,
        2207, 2218, 2411, 2411, 2401, 2207, 2401, 2411, 2412, 2412, 2402, 2401,
        2402, 2412, 2413, 2413, 2403, 2402, 2403, 2413, 2414, 2414, 2404, 2403,
        2404, 2414, 2415, 2415, 2405, 2404, 2405, 2415, 2416, 2416, 2406, 2405,
        2406, 2416, 2417, 2417, 2407, 2406, 2407, 2417, 2418, 2418, 2408, 2407,
        2408, 2418, 2419, 2419, 2409, 2408, 2409, 2419, 2420, 2420, 2410, 2409,
        2218, 2229, 2421, 2421, 2411, 2218, 2411, 2421, 2422, 2422, 2412, 2411,
        2412, 2422, 2423, 2423, 2413, 2412, 2413, 2423, 2424, 2424, 2414, 2413,
        2414, 2424, 2425, 2425, 2415, 2414, 2415, 2425, 2426, 2426, 2416, 2415,
        2416, 2426, 2427, 2427, 2417, 2416, 2417, 2427, 2428, 2428, 2418, 2417,
        2418, 2428, 2429, 2429, 2419, 2418, 2419, 2429, 2430, 2430, 2420, 2419,
        2229, 2240, 2431, 2431, 2421, 2229, 2421, 2431, 2432, 2432, 2422, 2421,
        2422, 2432, 2433, 2433, 2423, 2422, 2423, 2433, 2434, 2434, 2424, 2423,
        2424, 2434, 2435, 2435, 2425, 2424, 2425, 2435, 2436, 2436, 2426, 2425,
        2426, 2436, 2437, 2437, 2427, 2426, 2427, 2437, 2438, 2438, 2428, 2427,
        2428, 2438, 2439, 2439, 2429, 2428, 2429, 2439, 2440, 2440, 2430, 2429,
        2240, 2042, 2242, 2242, 2431, 2240, 2431, 2242, 2244, 2244, 2432, 2431,
        2432, 2244, 2246, 2246, 2433, 2432, 2433, 2246, 2248, 2248, 2434, 2433,
        2434, 2248, 2250, 2250, 2435, 2434, 2435, 2250, 2252, 2252, 2436, 2435,
        2436, 2252, 2254, 2254, 2437, 2436, 2437, 2254, 2256, 2256, 2438, 2437,
        2438, 2256, 2258, 2258, 2439, 2438, 2439, 2258, 2260, 2260, 2440, 2439,
        2441, 2442, 2443, 2442, 2441, 2444, 2444, 2445, 2442, 2445, 2444, 2446,
        2446, 2447, 2445, 2447, 2446, 2448, 2448, 2449, 2447, 2449, 2448, 2450,
        2450, 2451, 2449, 2451, 2450, 2452, 2452, 2453, 2451, 2453, 2452, 2454,
        2454, 2455, 2453, 2455, 2454, 2456, 2456, 2457, 2455, 2457, 2456, 2458,
        2458, 2459, 2457, 2459, 2458, 2460, 2460, 2461, 2459, 2462, 2441, 2443,
        2441, 2462, 2463, 2463, 2444, 2441, 2444, 2463, 2464, 2464, 2446, 2444,
        2446, 2464, 2465, 2465, 2448, 2446, 2448, 2465, 2466, 2466, 2450, 2448,
        2450, 2466, 2467, 2467, 2452, 2450, 2452, 2467, 2468, 2468, 2454, 2452,
        2454, 2468, 2469, 2469, 2456, 2454, 2456, 2469, 2470, 2470, 2458, 2456,
        2458, 2470, 2471, 2471, 2460, 2458, 2472, 2462, 2443, 2462, 2472, 2473,
        2473, 2463, 2462, 2463, 2473, 2474, 2474, 2464, 2463, 2464, 2474, 2475,
        2475, 2465, 2464, 2465, 2475, 2476, 2476, 2466, 2465, 2466, 2476, 2477,
        2477, 2467, 2466, 2467, 2477, 2478, 2478, 2468, 2467, 2468, 2478, 2479,
        2479, 2469, 2468, 2469, 2479, 2480, 2480, 2470, 2469, 2470, 2480, 2481,
        2481, 2471, 2470, 2482, 2472, 2443, 2472, 2482, 2483, 2483, 2473, 2472,
        2473, 2483, 2484, 2484, 2474, 2473, 2474, 2484, 2485, 2485, 2475, 2474,
        2475, 2485, 2486, 2486, 2476, 2475, 2476, 2486, 2487, 2487, 2477, 2476,
        2477, 2487, 2488, 2488, 2478, 2477, 2478, 2488, 2489, 2489, 2479, 2478,
        2479, 2489, 2490, 2490, 2480, 2479, 2480, 2490, 2491, 2491, 2481, 2480,
        2492, 2482, 2443, 2482, 2492, 2493, 2493, 2483, 2482, 2483, 2493, 2494,
        2494, 2484, 2483, 2484, 2494, 2495, 2495, 2485, 2484, 2485, 2495, 2496,
        2496, 2486, 2485, 2486, 2496, 2497, 2497, 2487, 2486, 2487, 2497, 2498,
        2498, 2488, 2487, 2488, 2498, 2499, 2499, 2489, 2488, 2489, 2499, 2500,
        2500, 2490, 2489, 2490, 2500, 2501, 2501, 2491, 2490, 2502, 2492, 2443,
        2492, 2502, 2503, 2503, 2493, 2492, 2493, 2503, 2504, 2504, 2494, 2493,
        2494, 2504, 2505, 2505, 2495, 2494, 2495, 2505, 2506, 2506, 2496, 2495,
        2496, 2506, 2507, 2507, 2497, 2496, 2497, 2507, 2508, 2508, 2498, 2497,
        2498, 2508, 2509, 2509, 2499, 2498, 2499, 2509, 2510, 2510, 2500, 2499,
        2500, 2510, 2511, 2511, 2501, 2500, 2512, 2502, 2443, 2502, 2512, 2513,
        2513, 2503, 2502, 2503, 2513, 2514, 2514, 2504, 2503, 2504, 2514, 2515,
        2515, 2505, 2504, 2505, 2515, 2516, 2516, 2506, 2505, 2506, 2516, 2517,
        2517, 2507, 2506, 2507, 2517, 2518, 2518, 2508, 2507, 2508, 2518, 2519,
        2519, 2509, 2508, 2509, 2519, 2520, 2520, 2510, 2509, 2510, 2520, 2521,
        2521, 2511, 2510, 2522, 2512, 2443, 2512, 2522, 2523, 2523, 2513, 2512,
        2513, 2523, 2524, 2524, 2514, 2513, 2514, 2524, 2525, 2525, 2515, 2514,
        2515, 2525, 2526, 2526, 2516, 2515, 2516, 2526, 2527, 2527, 2517, 2516,
        2517, 2527, 2528, 2528, 2518, 2517, 2518, 2528, 2529, 2529, 2519, 2518,
        2519, 2529, 2530, 2530, 2520, 2519, 2520, 2530, 2531, 2531, 2521, 2520,
        2532, 2522, 2443, 2522, 2532, 2533, 2533, 2523, 2522, 2523, 2533, 2534,
        2534, 2524, 2523, 2524, 2534, 2535, 2535, 2525, 2524, 2525, 2535, 2536,
        2536, 2526, 2525, 2526, 2536, 2537, 2537, 2527, 2526, 2527, 2537, 2538,
        2538, 2528, 2527, 2528, 2538, 2539, 2539, 2529, 2528, 2529, 2539, 2540,
        2540, 2530, 2529, 2530, 2540, 2541, 2541, 2531, 2530, 2542, 2532, 2443,
        2532, 2542, 2543, 2543, 2533, 2532, 2533, 2543, 2544, 2544, 2534, 2533,
        2534, 2544, 2545, 2545, 2535, 2534, 2535, 2545, 2546, 2546, 2536, 2535,
        2536, 2546, 2547, 2547, 2537, 2536, 2537, 2547, 2548, 2548, 2538, 2537,
        2538, 2548, 2549, 2549, 2539, 2538, 2539, 2549, 2550, 2550, 2540, 2539,
        2540, 2550, 2551, 2551, 2541, 2540, 2552, 2542, 2443, 2542, 2552, 2553,
        2553, 2543, 2542, 2543, 2553, 2554, 2554, 2544, 2543, 2544, 2554, 2555,
        2555, 2545, 2544, 2545, 2555, 2556, 2556, 2546, 2545, 2546, 2556, 2557,
        2557, 2547, 2546, 2547, 2557, 2558, 2558, 2548, 2547, 2548, 2558, 2559,
        2559, 2549, 2548, 2549, 2559, 2560, 2560, 2550, 2549, 2550, 2560, 2561,
        2561, 2551, 2550, 2562, 2552, 2443, 2552, 2562, 2563, 2563, 2553, 2552,
        2553, 2563, 2564, 2564, 2554, 2553, 2554, 2564, 2565, 2565, 2555, 2554,
        2555, 2565, 2566, 2566, 2556, 2555, 2556, 2566, 2567, 2567, 2557, 2556,
        2557, 2567, 2568, 2568, 2558, 2557, 2558, 2568, 2569, 2569, 2559, 2558,
        2559, 2569, 2570, 2570, 2560, 2559, 2560, 2570, 2571, 2571, 2561, 2560,
        2572, 2562, 2443, 2562, 2572, 2573, 2573, 2563, 2562, 2563, 2573, 2574,
        2574, 2564, 2563, 2564, 2574, 2575, 2575, 2565, 2564, 2565, 2575, 2576,
        2576, 2566, 2565, 2566, 2576, 2577, 2577, 2567, 2566, 2567, 2577, 2578,
        2578, 2568, 2567, 2568, 2578, 2579, 2579, 2569, 2568, 2569, 2579, 2580,
        2580, 2570, 2569, 2570, 2580, 2581, 2581, 2571, 2570, 2582, 2572, 2443,
        2572, 2582, 2583, 2583, 2573, 2572, 2573, 2583, 2584, 2584, 2574, 2573,
        2574, 2584, 2585, 2585, 2575, 2574, 2575, 2585, 2586, 2586, 2576, 2575,
        2576, 2586, 2587, 2587, 2577, 2576, 2577, 2587, 2588, 2588, 2578, 2577,
        2578, 2588, 2589, 2589, 2579, 2578, 2579, 2589, 2590, 2590, 2580, 2579,
        2580, 2590, 2591, 2591, 2581, 2580, 2592, 2582, 2443, 2582, 2592, 2593,
        2593, 2583, 2582, 2583, 2593, 2594, 2594, 2584, 2583, 2584, 2594, 2595,
        2595, 2585, 2584, 2585, 2595, 2596, 2596, 2586, 2585, 2586, 2596, 2597,
        2597, 2587, 2586, 2587, 2597, 2598, 2598, 2588, 2587, 2588, 2598, 2599,
        2599, 2589, 2588, 2589, 2599, 2600, 2600, 2590, 2589, 2590, 2600, 2601,
        2601, 2591, 2590, 2602, 2592, 2443, 2592, 2602, 2603, 2603, 2593, 2592,
        2593, 2603, 2604, 2604, 2594, 2593, 2594, 2604, 2605, 2605, 2595, 2594,
        2595, 2605, 2606, 2606, 2596, 2595, 2596, 2606, 2607, 2607, 2597, 2596,
        2597, 2607, 2608, 2608, 2598, 2597, 2598, 2608, 2609, 2609, 2599, 2598,
        2599, 2609, 2610, 2610, 2600, 2599, 2600, 2610, 2611, 2611, 2601, 2600,
        2612, 2602, 2443, 2602, 2612, 2613, 2613, 2603, 2602, 2603, 2613, 2614,
        2614, 2604, 2603, 2604, 2614, 2615, 2615, 2605, 2604, 2605, 2615, 2616,
        2616, 2606, 2605, 2606, 2616, 2617, 2617, 2607, 2606, 2607, 2617, 2618,
        2618, 2608, 2607, 2608, 2618, 2619, 2619, 2609, 2608, 2609, 2619, 2620,
        2620, 2610, 2609, 2610, 2620, 2621, 2621, 2611, 2610, 2622, 2612, 2443,
        2612, 2622, 2623, 2623, 2613, 2612, 2613, 2623, 2624, 2624, 2614, 2613,
        2614, 2624, 2625, 2625, 2615, 2614, 2615, 2625, 2626, 2626, 2616, 2615,
        2616, 2626, 2627, 2627, 2617, 2616, 2617, 2627, 2628, 2628, 2618, 2617,
        2618, 2628, 2629, 2629, 2619, 2618, 2619, 2629, 2630, 2630, 2620, 2619,
        2620, 2630, 2631, 2631, 2621, 2620, 2632, 2622, 2443, 2622, 2632, 2633,
        2633, 2623, 2622, 2623, 2633, 2634, 2634, 2624, 2623, 2624, 2634, 2635,
        2635, 2625, 2624, 2625, 2635, 2636, 2636, 2626, 2625, 2626, 2636, 2637,
        2637, 2627, 2626, 2627, 2637, 2638, 2638, 2628, 2627, 2628, 2638, 2639,
        2639, 2629, 2628, 2629, 2639, 2640, 2640, 2630, 2629, 2630, 2640, 2641,
        2641, 2631, 2630, 2642, 2632, 2443, 2632, 2642, 2643, 2643, 2633, 2632,
        2633, 2643, 2644, 2644, 2634, 2633, 2634, 2644, 2645, 2645, 2635, 2634,
        2635, 2645, 2646, 2646, 2636, 2635, 2636, 2646, 2647, 2647, 2637, 2636,
        2637, 2647, 2648, 2648, 2638, 2637, 2638, 2648, 2649, 2649, 2639, 2638,
        2639, 2649, 2650, 2650, 2640, 2639, 2640, 2650, 2651, 2651, 2641, 2640,
        2652, 2642, 2443, 2642, 2652, 2653, 2653, 2643, 2642, 2643, 2653, 2654,
        2654, 2644, 2643, 2644, 2654, 2655, 2655, 2645, 2644, 2645, 2655, 2656,
        2656, 2646, 2645, 2646, 2656, 2657, 2657, 2647, 2646, 2647, 2657, 2658,
        2658, 2648, 2647, 2648, 2658, 2659, 2659, 2649, 2648, 2649, 2659, 2660,
        2660, 2650, 2649, 2650, 2660, 2661, 2661, 2651, 2650, 2662, 2652, 2443,
        2652, 2662, 2663, 2663, 2653, 2652, 2653, 2663, 2664, 2664, 2654, 2653,
        2654, 2664, 2665, 2665, 2655, 2654, 2655, 2665, 2666, 2666, 2656, 2655,
        2656, 2666, 2667, 2667, 2657, 2656, 2657, 2667, 2668, 2668, 2658, 2657,
        2658, 2668, 2669, 2669, 2659, 2658, 2659, 2669, 2670, 2670, 2660, 2659,
        2660, 2670, 2671, 2671, 2661, 2660, 2672, 2662, 2443, 2662, 2672, 2673,
        2673, 2663, 2662, 2663, 2673, 2674, 2674, 2664, 2663, 2664, 2674, 2675,
        2675, 2665, 2664, 2665, 2675, 2676, 2676, 2666, 2665, 2666, 2676, 2677,
        2677, 2667, 2666, 2667, 2677, 2678, 2678, 2668, 2667, 2668, 2678, 2679,
        2679, 2669, 2668, 2669, 2679, 2680, 2680, 2670, 2669, 2670, 2680, 2681,
        2681, 2671, 2670, 2682, 2672, 2443, 2672, 2682, 2683, 2683, 2673, 2672,
        2673, 2683, 2684, 2684, 2674, 2673, 2674, 2684, 2685, 2685, 2675, 2674,
        2675, 2685, 2686, 2686, 2676, 2675, 2676, 2686, 2687, 2687, 2677, 2676,
        2677, 2687, 2688, 2688, 2678, 2677, 2678, 2688, 2689, 2689, 2679, 2678,
        2679, 2689, 2690, 2690, 2680, 2679, 2680, 2690, 2691, 2691, 2681, 2680,
        2692, 2682, 2443, 2682, 2692, 2693, 2693, 2683, 2682, 2683, 2693, 2694,
        2694, 2684, 2683, 2684, 2694, 2695, 2695, 2685, 2684, 2685, 2695, 2696,
        2696, 2686, 2685, 2686, 2696, 2697, 2697, 2687, 2686, 2687, 2697, 2698,
        2698, 2688, 2687, 2688, 2698, 2699, 2699, 2689, 2688, 2689, 2699, 2700,
        2700, 2690, 2689, 2690, 2700, 2701, 2701, 2691, 2690, 2702, 2692, 2443,
        2692, 2702, 2703, 2703, 2693, 2692, 2693, 2703, 2704, 2704, 2694, 2693,
        2694, 2704, 2705, 2705, 2695, 2694, 2695, 2705, 2706, 2706, 2696, 2695,
        2696, 2706, 2707, 2707, 2697, 2696, 2697, 2707, 2708, 2708, 2698, 2697,
        2698, 2708, 2709, 2709, 2699, 2698, 2699, 2709, 2710, 2710, 2700, 2699,
        2700, 2710, 2711, 2711, 2701, 2700, 2712, 2702, 2443, 2702, 2712, 2713,
        2713, 2703, 2702, 2703, 2713, 2714, 2714, 2704, 2703, 2704, 2714, 2715,
        2715, 2705, 2704, 2705, 2715, 2716, 2716, 2706, 2705, 2706, 2716, 2717,
        2717, 2707, 2706, 2707, 2717, 2718, 2718, 2708, 2707, 2708, 2718, 2719,
        2719, 2709, 2708, 2709, 2719, 2720, 2720, 2710, 2709, 2710, 2720, 2721,
        2721, 2711, 2710, 2722, 2712, 2443, 2712, 2722, 2723, 2723, 2713, 2712,
        2713, 2723, 2724, 2724, 2714, 2713, 2714, 2724, 2725, 2725, 2715, 2714,
        2715, 2725, 2726, 2726, 2716, 2715, 2716, 2726, 2727, 2727, 2717, 2716,
        2717, 2727, 2728, 2728, 2718, 2717, 2718, 2728, 2729, 2729, 2719, 2718,
        2719, 2729, 2730, 2730, 2720, 2719, 2720, 2730, 2731, 2731, 2721, 2720,
        2732, 2722, 2443, 2722, 2732, 2733, 2733, 2723, 2722, 2723, 2733, 2734,
        2734, 2724, 2723, 2724, 2734, 2735, 2735, 2725, 2724, 2725, 2735, 2736,
        2736, 2726, 2725, 2726, 2736, 2737, 2737, 2727, 2726, 2727, 2737, 2738,
        2738, 2728, 2727, 2728, 2738, 2739, 2739, 2729, 2728, 2729, 2739, 2740,
        2740, 2730, 2729, 2730, 2740, 2741, 2741, 2731, 2730, 2742, 2732, 2443,
        2732, 2742, 2743, 2743, 2733, 2732, 2733, 2743, 2744, 2744, 2734, 2733,
        2734, 2744, 2745, 2745, 2735, 2734, 2735, 2745, 2746, 2746, 2736, 2735,
        2736, 2746, 2747, 2747, 2737, 2736, 2737, 2747, 2748, 2748, 2738, 2737,
        2738, 2748, 2749, 2749, 2739, 2738, 2739, 2749, 2750, 2750, 2740, 2739,
        2740, 2750, 2751, 2751, 2741, 2740, 2752, 2742, 2443, 2742, 2752, 2753,
        2753, 2743, 2742, 2743, 2753, 2754, 2754, 2744, 2743, 2744, 2754, 2755,
        2755, 2745, 2744, 2745, 2755, 2756, 2756, 2746, 2745, 2746, 2756, 2757,
        2757, 2747, 2746, 2747, 2757, 2758, 2758, 2748, 2747, 2748, 2758, 2759,
        2759, 2749, 2748, 2749, 2759, 2760, 2760, 2750, 2749, 2750, 2760, 2761,
        2761, 2751, 2750, 2762, 2752, 2443, 2752, 2762, 2763, 2763, 2753, 2752,
        2753, 2763, 2764, 2764, 2754, 2753, 2754, 2764, 2765, 2765, 2755, 2754,
        2755, 2765, 2766, 2766, 2756, 2755, 2756, 2766, 2767, 2767, 2757, 2756,
        2757, 2767, 2768, 2768, 2758, 2757, 2758, 2768, 2769, 2769, 2759, 2758,
        2759, 2769, 2770, 2770, 2760, 2759, 2760, 2770, 2771, 2771, 2761, 2760,
        2772, 2762, 2443, 2762, 2772, 2773, 2773, 2763, 2762, 2763, 2773, 2774,
        2774, 2764, 2763, 2764, 2774, 2775, 2775, 2765, 2764, 2765, 2775, 2776,
        2776, 2766, 2765, 2766, 2776, 2777, 2777, 2767, 2766, 2767, 2777, 2778,
        2778, 2768, 2767, 2768, 2778, 2779, 2779, 2769, 2768, 2769, 2779, 2780,
        2780, 2770, 2769, 2770, 2780, 2781, 2781, 2771, 2770, 2782, 2772, 2443,
        2772, 2782, 2783, 2783, 2773, 2772, 2773, 2783, 2784, 2784, 2774, 2773,
        2774, 2784, 2785, 2785, 2775, 2774, 2775, 2785, 2786, 2786, 2776, 2775,
        2776, 2786, 2787, 2787, 2777, 2776, 2777, 2787, 2788, 2788, 2778, 2777,
        2778, 2788, 2789, 2789, 2779, 2778, 2779, 2789, 2790, 2790, 2780, 2779,
        2780, 2790, 2791, 2791, 2781, 2780, 2792, 2782, 2443, 2782, 2792, 2793,
        2793, 2783, 2782, 2783, 2793, 2794, 2794, 2784, 2783, 2784, 2794, 2795,
        2795, 2785, 2784, 2785, 2795, 2796, 2796, 2786, 2785, 2786, 2796, 2797,
        2797, 2787, 2786, 2787, 2797, 2798, 2798, 2788, 2787, 2788, 2798, 2799,
        2799, 2789, 2788, 2789, 2799, 2800, 2800, 2790, 2789, 2790, 2800, 2801,
        2801, 2791, 2790, 2802, 2792, 2443, 2792, 2802, 2803, 2803, 2793, 2792,
        2793, 2803, 2804, 2804, 2794, 2793, 2794, 2804, 2805, 2805, 2795, 2794,
        2795, 2805, 2806, 2806, 2796, 2795, 2796, 2806, 2807, 2807, 2797, 2796,
        2797, 2807, 2808, 2808, 2798, 2797, 2798, 2808, 2809, 2809, 2799, 2798,
        2799, 2809, 2810, 2810, 2800, 2799, 2800, 2810, 2811, 2811, 2801, 2800,
        2812, 2802, 2443, 2802, 2812, 2813, 2813, 2803, 2802, 2803, 2813, 2814,
        2814, 2804, 2803, 2804, 2814, 2815, 2815, 2805, 2804, 2805, 2815, 2816,
        2816, 2806, 2805, 2806, 2816, 2817, 2817, 2807, 2806, 2807, 2817, 2818,
        2818, 2808, 2807, 2808, 2818, 2819, 2819, 2809, 2808, 2809, 2819, 2820,
        2820, 2810, 2809, 2810, 2820, 2821, 2821, 2811, 2810, 2822, 2812, 2443,
        2812, 2822, 2823, 2823, 2813, 2812, 2813, 2823, 2824, 2824, 2814, 2813,
        2814, 2824, 2825, 2825, 2815, 2814, 2815, 2825, 2826, 2826, 2816, 2815,
        2816, 2826, 2827, 2827, 2817, 2816, 2817, 2827, 2828, 2828, 2818, 2817,
        2818, 2828, 2829, 2829, 2819, 2818, 2819, 2829, 2830, 2830, 2820, 2819,
        2820, 2830, 2831, 2831, 2821, 2820, 2832, 2822, 2443, 2822, 2832, 2833,
        2833, 2823, 2822, 2823, 2833, 2834, 2834, 2824, 2823, 2824, 2834, 2835,
        2835, 2825, 2824, 2825, 2835, 2836, 2836, 2826, 2825, 2826, 2836, 2837,
        2837, 2827, 2826, 2827, 2837, 2838, 2838, 2828, 2827, 2828, 2838, 2839,
        2839, 2829, 2828, 2829, 2839, 2840, 2840, 2830, 2829, 2830, 2840, 2841,
        2841, 2831, 2830, 2442, 2832, 2443, 2832, 2442, 2445, 2445, 2833, 2832,
        2833, 2445, 2447, 2447, 2834, 2833, 2834, 2447, 2449, 2449, 2835, 2834,
        2835, 2449, 2451, 2451, 2836, 2835, 2836, 2451, 2453, 2453, 2837, 2836,
        2837, 2453, 2455, 2455, 2838, 2837, 2838, 2455, 2457, 2457, 2839, 2838,
        2839, 2457, 2459, 2459, 2840, 2839, 2840, 2459, 2461, 2461, 2841, 2840,
        2461, 2460, 2842, 2842, 2843, 2461, 2843, 2842, 2844, 2844, 2845, 2843,
        2845, 2844, 2846, 2846, 2847, 2845, 2847, 2846, 2848, 2848, 2849, 2847,
        2849, 2848, 2850, 2850, 2851, 2849, 2851, 2850, 2852, 2852, 2853, 2851,
        2853, 2852, 2854, 2854, 2855, 2853, 2855, 2854, 2856, 2856, 2857, 2855,
        2857, 2856, 2858, 2858, 2859, 2857, 2859, 2858, 2860, 2860, 2861, 2859,
        2460, 2471, 2862, 2862, 2842, 2460, 2842, 2862, 2863, 2863, 2844, 2842,
        2844, 2863, 2864, 2864, 2846, 2844, 2846, 2864, 2865, 2865, 2848, 2846,
        2848, 2865, 2866, 2866, 2850, 2848, 2850, 2866, 2867, 2867, 2852, 2850,
        2852, 2867, 2868, 2868, 2854, 2852, 2854, 2868, 2869, 2869, 2856, 2854,
        2856, 2869, 2870, 2870, 2858, 2856, 2858, 2870, 2871, 2871, 2860, 2858,
        2471, 2481, 2872, 2872, 2862, 2471, 2862, 2872, 2873, 2873, 2863, 2862,
        2863, 2873, 2874, 2874, 2864, 2863, 2864, 2874, 2875, 2875, 2865, 2864,
        2865, 2875, 2876, 2876, 2866, 2865, 2866, 2876, 2877, 2877, 2867, 2866,
        2867, 2877, 2878, 2878, 2868, 2867, 2868, 2878, 2879, 2879, 2869, 2868,
        2869, 2879, 2880, 2880, 2870, 2869, 2870, 2880, 2881, 2881, 2871, 2870,
        2481, 2491, 2882, 2882, 2872, 2481, 2872, 2882, 2883, 2883, 2873, 2872,
        2873, 2883, 2884, 2884, 2874, 2873, 2874, 2884, 2885, 2885, 2875, 2874,
        2875, 2885, 2886, 2886, 2876, 2875, 2876, 2886, 2887, 2887, 2877, 2876,
        2877, 2887, 2888, 2888, 2878, 2877, 2878, 2888, 2889, 2889, 2879, 2878,
        2879, 2889, 2890, 2890, 2880, 2879, 2880, 2890, 2891, 2891, 2881, 2880,
        2491, 2501, 2892, 2892, 2882, 2491, 2882, 2892, 2893, 2893, 2883, 2882,
        2883, 2893, 2894, 2894, 2884, 2883, 2884, 2894, 2895, 2895, 2885, 2884,
        2885, 2895, 2896, 2896, 2886, 2885, 2886, 2896, 2897, 2897, 2887, 2886,
        2887, 2897, 2898, 2898, 2888, 2887, 2888, 2898, 2899, 2899, 2889, 2888,
        2889, 2899, 2900, 2900, 2890, 2889, 2890, 2900, 2901, 2901, 2891, 2890,
        2501, 2511, 2902, 2902, 2892, 2501, 2892, 2902, 2903, 2903, 2893, 2892,
        2893, 2903, 2904, 2904, 2894, 2893, 2894, 2904, 2905, 2905, 2895, 2894,
        2895, 2905, 2906, 2906, 2896, 2895, 2896, 2906, 2907, 2907, 2897, 2896,
        2897, 2907, 2908, 2908, 2898, 2897, 2898, 2908, 2909, 2909, 2899, 2898,
        2899, 2909, 2910, 2910, 2900, 2899, 2900, 2910, 2911, 2911, 2901, 2900,
        2511, 2521, 2912, 2912, 2902, 2511, 2902, 2912, 2913, 2913, 2903, 2902,
        2903, 2913, 2914, 2914, 2904, 2903, 2904, 2914, 2915, 2915, 2905, 2904,
        2905, 2915, 2916, 2916, 2906, 2905, 2906, 2916, 2917, 2917, 2907, 2906,
        2907, 2917, 2918, 2918, 2908, 2907, 2908, 2918, 2919, 2919, 2909, 2908,
        2909, 2919, 2920, 2920, 2910, 2909, 2910, 2920, 2921, 2921, 2911, 2910,
        2521, 2531, 2922, 2922, 2912, 2521, 2912, 2922, 2923, 2923, 2913, 2912,
        2913, 2923, 2924, 2924, 2914, 2913, 2914, 2924, 2925, 2925, 2915, 2914,
        2915, 2925, 2926, 2926, 2916, 2915, 2916, 2926, 2927, 2927, 2917, 2916,
        2917, 2927, 2928, 2928, 2918, 2917, 2918, 2928, 2929, 2929, 2919, 2918,
        2919, 2929, 2930, 2930, 2920, 2919, 2920, 2930, 2931, 2931, 2921, 2920,
        2531, 2541, 2932, 2932, 2922, 2531, 2922, 2932, 2933, 2933, 2923, 2922,
        2923, 2933, 2934, 2934, 2924, 2923, 2924, 2934, 2935, 2935, 2925, 2924,
        2925, 2935, 2936, 2936, 2926, 2925, 2926, 2936, 2937, 2937, 2927, 2926,
        2927, 2937, 2938, 2938, 2928, 2927, 2928, 2938, 2939, 2939, 2929, 2928,
        2929, 2939, 2940, 2940, 2930, 2929, 2930, 2940, 2941, 2941, 2931, 2930,
        2541, 2551, 2942, 2942, 2932, 2541, 2932, 2942, 2943, 2943, 2933, 2932,
        2933, 2943, 2944, 2944, 2934, 2933, 2934, 2944, 2945, 2945, 2935, 2934,
        2935, 2945, 2946, 2946, 2936, 2935, 2936, 2946, 2947, 2947, 2937, 2936,
        2937, 2947, 2948, 2948, 2938, 2937, 2938, 2948, 2949, 2949, 2939, 2938,
        2939, 2949, 2950, 2950, 2940, 2939, 2940, 2950, 2951, 2951, 2941, 2940,
        2551, 2561, 2952, 2952, 2942, 2551, 2942, 2952, 2953, 2953, 2943, 2942,
        2943, 2953, 2954, 2954, 2944, 2943, 2944, 2954, 2955, 2955, 2945, 2944,
        2945, 2955, 2956, 2956, 2946, 2945, 2946, 2956, 2957, 2957, 2947, 2946,
        2947, 2957, 2958, 2958, 2948, 2947, 2948, 2958, 2959, 2959, 2949, 2948,
        2949, 2959, 2960, 2960, 2950, 2949, 2950, 2960, 2961, 2961, 2951, 2950,
        2561, 2571, 2962, 2962, 2952, 2561, 2952, 2962, 2963, 2963, 2953, 2952,
        2953, 2963, 2964, 2964, 2954, 2953, 2954, 2964, 2965, 2965, 2955, 2954,
        2955, 2965, 2966, 2966, 2956, 2955, 2956, 2966, 2967, 2967, 2957, 2956,
        2957, 2967, 2968, 2968, 2958, 2957, 2958, 2968, 2969, 2969, 2959, 2958,
        2959, 2969, 2970, 2970, 2960, 2959, 2960, 2970, 2971, 2971, 2961, 2960,
        2571, 2581, 2972, 2972, 2962, 2571, 2962, 2972, 2973, 2973, 2963, 2962,
        2963, 2973, 2974, 2974, 2964, 2963, 2964, 2974, 2975, 2975, 2965, 2964,
        2965, 2975, 2976, 2976, 2966, 2965, 2966, 2976, 2977, 2977, 2967, 2966,
        2967, 2977, 2978, 2978, 2968, 2967, 2968, 2978, 2979, 2979, 2969, 2968,
        2969, 2979, 2980, 2980, 2970, 2969, 2970, 2980, 2981, 2981, 2971, 2970,
        2581, 2591, 2982, 2982, 2972, 2581, 2972, 2982, 2983, 2983, 2973, 2972,
        2973, 2983, 2984, 2984, 2974, 2973, 2974, 2984, 2985, 2985, 2975, 2974,
        2975, 2985, 2986, 2986, 2976, 2975, 2976, 2986, 2987, 2987, 2977, 2976,
        2977, 2987, 2988, 2988, 2978, 2977, 2978, 2988, 2989, 2989, 2979, 2978,
        2979, 2989, 2990, 2990, 2980, 2979, 2980, 2990, 2991, 2991, 2981, 2980,
        2591, 2601, 2992, 2992, 2982, 2591, 2982, 2992, 2993, 2993, 2983, 2982,
        2983, 2993, 2994, 2994, 2984, 2983, 2984, 2994, 2995, 2995, 2985, 2984,
        2985, 2995, 2996, 2996, 2986, 2985, 2986, 2996, 2997, 2997, 2987, 2986,
        2987, 2997, 2998, 2998, 2988, 2987, 2988, 2998, 2999, 2999, 2989, 2988,
        2989, 2999, 3000, 3000, 2990, 2989, 2990, 3000, 3001, 3001, 2991, 2990,
        2601, 2611, 3002, 3002, 2992, 2601, 2992, 3002, 3003, 3003, 2993, 2992,
        2993, 3003, 3004, 3004, 2994, 2993, 2994, 3004, 3005, 3005, 2995, 2994,
        2995, 3005, 3006, 3006, 2996, 2995, 2996, 3006, 3007, 3007, 2997, 2996,
        2997, 3007, 3008, 3008, 2998, 2997, 2998, 3008, 3009, 3009, 2999, 2998,
        2999, 3009, 3010, 3010, 3000, 2999, 3000, 3010, 3011, 3011, 3001, 3000,
        2611, 2621, 3012, 3012, 3002, 2611, 3002, 3012, 3013, 3013, 3003, 3002,
        3003, 3013, 3014, 3014, 3004, 3003, 3004, 3014, 3015, 3015, 3005, 3004,
        3005, 3015, 3016, 3016, 3006, 3005, 3006, 3016, 3017, 3017, 3007, 3006,
        3007, 3017, 3018, 3018, 3008, 3007, 3008, 3018, 3019, 3019, 3009, 3008,
        3009, 3019, 3020, 3020, 3010, 3009, 3010, 3020, 3021, 3021, 3011, 3010,
        2621, 2631, 3022, 3022, 3012, 2621, 3012, 3022, 3023, 3023, 3013, 3012,
        3013, 3023, 3024, 3024, 3014, 3013, 3014, 3024, 3025, 3025, 3015, 3014,
        3015, 3025, 3026, 3026, 3016, 3015, 3016, 3026, 3027, 3027, 3017, 3016,
        3017, 3027, 3028, 3028, 3018, 3017, 3018, 3028, 3029, 3029, 3019, 3018,
        3019, 3029, 3030, 3030, 3020, 3019, 3020, 3030, 3031, 3031, 3021, 3020,
        2631, 2641, 3032, 3032, 3022, 2631, 3022, 3032, 3033, 3033, 3023, 3022,
        3023, 3033, 3034, 3034, 3024, 3023, 3024, 3034, 3035, 3035, 3025, 3024,
        3025, 3035, 3036, 3036, 3026, 3025, 3026, 3036, 3037, 3037, 3027, 3026,
        3027, 3037, 3038, 3038, 3028, 3027, 3028, 3038, 3039, 3039, 3029, 3028,
        3029, 3039, 3040, 3040, 3030, 3029, 3030, 3040, 3041, 3041, 3031, 3030,
        2641, 2651, 3042, 3042, 3032, 2641, 3032, 3042, 3043, 3043, 3033, 3032,
        3033, 3043, 3044, 3044, 3034, 3033, 3034, 3044, 3045, 3045, 3035, 3034,
        3035, 3045, 3046, 3046, 3036, 3035, 3036, 3046, 3047, 3047, 3037, 3036,
        3037, 3047, 3048, 3048, 3038, 3037, 3038, 3048, 3049, 3049, 3039, 3038,
        3039, 3049, 3050, 3050, 3040, 3039, 3040, 3050, 3051, 3051, 3041, 3040,
        2651, 2661, 3052, 3052, 3042, 2651, 3042, 3052, 3053, 3053, 3043, 3042,
        3043, 3053, 3054, 3054, 3044, 3043, 3044, 3054, 3055, 3055, 3045, 3044,
        3045, 3055, 3056, 3056, 3046, 3045, 3046, 3056, 3057, 3057, 3047, 3046,
        3047, 3057, 3058, 3058, 3048, 3047, 3048, 3058, 3059, 3059, 3049, 3048,
        3049, 3059, 3060, 3060, 3050, 3049, 3050, 3060, 3061, 3061, 3051, 3050,
        2661, 2671, 3062, 3062, 3052, 2661, 3052, 3062, 3063, 3063, 3053, 3052,
        3053, 3063, 3064, 3064, 3054, 3053, 3054, 3064, 3065, 3065, 3055, 3054,
        3055, 3065, 3066, 3066, 3056, 3055, 3056, 3066, 3067, 3067, 3057, 3056,
        3057, 3067, 3068, 3068, 3058, 3057, 3058, 3068, 3069, 3069, 3059, 3058,
        3059, 3069, 3070, 3070, 3060, 3059, 3060, 3070, 3071, 3071, 3061, 3060,
        2671, 2681, 3072, 3072, 3062, 2671, 3062, 3072, 3073, 3073, 3063, 3062,
        3063, 3073, 3074, 3074, 3064, 3063, 3064, 3074, 3075, 3075, 3065, 3064,
        3065, 3075, 3076, 3076, 3066, 3065, 3066, 3076, 3077, 3077, 3067, 3066,
        3067, 3077, 3078, 3078, 3068, 3067, 3068, 3078, 3079, 3079, 3069, 3068,
        3069, 3079, 3080, 3080, 3070, 3069, 3070, 3080, 3081, 3081, 3071, 3070,
        2681, 2691, 3082, 3082, 3072, 2681, 3072, 3082, 3083, 3083, 3073, 3072,
        3073, 3083, 3084, 3084, 3074, 3073, 3074, 3084, 3085, 3085, 3075, 3074,
        3075, 3085, 3086, 3086, 3076, 3075, 3076, 3086, 3087, 3087, 3077, 3076,
        3077, 3087, 3088, 3088, 3078, 3077, 3078, 3088, 3089, 3089, 3079, 3078,
        3079, 3089, 3090, 3090, 3080, 3079, 3080, 3090, 3091, 3091, 3081, 3080,
        2691, 2701, 3092, 3092, 3082, 2691, 3082, 3092, 3093, 3093, 3083, 3082,
        3083, 3093, 3094, 3094, 3084, 3083, 3084, 3094, 3095, 3095, 3085, 3084,
        3085, 3095, 3096, 3096, 3086, 3085, 3086, 3096, 3097, 3097, 3087, 3086,
        3087, 3097, 3098, 3098, 3088, 3087, 3088, 3098, 3099, 3099, 3089, 3088,
        3089, 3099, 3100, 3100, 3090, 3089, 3090, 3100, 3101, 3101, 3091, 3090,
        2701, 2711, 3102, 3102, 3092, 2701, 3092, 3102, 3103, 3103, 3093, 3092,
        3093, 3103, 3104, 3104, 3094, 3093, 3094, 3104, 3105, 3105, 3095, 3094,
        3095, 3105, 3106, 3106, 3096, 3095, 3096, 3106, 3107, 3107, 3097, 3096,
        3097, 3107, 3108, 3108, 3098, 3097, 3098, 3108, 3109, 3109, 3099, 3098,
        3099, 3109, 3110, 3110, 3100, 3099, 3100, 3110, 3111, 3111, 3101, 3100,
        2711, 2721, 3112, 3112, 3102, 2711, 3102, 3112, 3113, 3113, 3103, 3102,
        3103, 3113, 3114, 3114, 3104, 3103, 3104, 3114, 3115, 3115, 3105, 3104,
        3105, 3115, 3116, 3116, 3106, 3105, 3106, 3116, 3117, 3117, 3107, 3106,
        3107, 3117, 3118, 3118, 3108, 3107, 3108, 3118, 3119, 3119, 3109, 3108,
        3109, 3119, 3120, 3120, 3110, 3109, 3110, 3120, 3121, 3121, 3111, 3110,
        2721, 2731, 3122, 3122, 3112, 2721, 3112, 3122, 3123, 3123, 3113, 3112,
        3113, 3123, 3124, 3124, 3114, 3113, 3114, 3124, 3125, 3125, 3115, 3114,
        3115, 3125, 3126, 3126, 3116, 3115, 3116, 3126, 3127, 3127, 3117, 3116,
        3117, 3127, 3128, 3128, 3118, 3117, 3118, 3128, 3129, 3129, 3119, 3118,
        3119, 3129, 3130, 3130, 3120, 3119, 3120, 3130, 3131, 3131, 3121, 3120,
        2731, 2741, 3132, 3132, 3122, 2731, 3122, 3132, 3133, 3133, 3123, 3122,
        3123, 3133, 3134, 3134, 3124, 3123, 3124, 3134, 3135, 3135, 3125, 3124,
        3125, 3135, 3136, 3136, 3126, 3125, 3126, 3136, 3137, 3137, 3127, 3126,
        3127, 3137, 3138, 3138, 3128, 3127, 3128, 3138, 3139, 3139, 3129, 3128,
        3129, 3139, 3140, 3140, 3130, 3129, 3130, 3140, 3141, 3141, 3131, 3130,
        2741, 2751, 3142, 3142, 3132, 2741, 3132, 3142, 3143, 3143, 3133, 3132,
        3133, 3143, 3144, 3144, 3134, 3133, 3134, 3144, 3145, 3145, 3135, 3134,
        3135, 3145, 3146, 3146, 3136, 3135, 3136, 3146, 3147, 3147, 3137, 3136,
        3137, 3147, 3148, 3148, 3138, 3137, 3138, 3148, 3149, 3149, 3139, 3138,
        3139, 3149, 3150, 3150, 3140, 3139, 3140, 3150, 3151, 3151, 3141, 3140,
        2751, 2761, 3152, 3152, 3142, 2751, 3142, 3152, 3153, 3153, 3143, 3142,
        3143, 3153, 3154, 3154, 3144, 3143, 3144, 3154, 3155, 3155, 3145, 3144,
        3145, 3155, 3156, 3156, 3146, 3145, 3146, 3156, 3157, 3157, 3147, 3146,
        3147, 3157, 3158, 3158, 3148, 3147, 3148, 3158, 3159, 3159, 3149, 3148,
        3149, 3159, 3160, 3160, 3150, 3149, 3150, 3160, 3161, 3161, 3151, 3150,
        2761, 2771, 3162, 3162, 3152, 2761, 3152, 3162, 3163, 3163, 3153, 3152,
        3153, 3163, 3164, 3164, 3154, 3153, 3154, 3164, 3165, 3165, 3155, 3154,
        3155, 3165, 3166, 3166, 3156, 3155, 3156, 3166, 3167, 3167, 3157, 3156,
        3157, 3167, 3168, 3168, 3158, 3157, 3158, 3168, 3169, 3169, 3159, 3158,
        3159, 3169, 3170, 3170, 3160, 3159, 3160, 3170, 3171, 3171, 3161, 3160,
        2771, 2781, 3172, 3172, 3162, 2771, 3162, 3172, 3173, 3173, 3163, 3162,
        3163, 3173, 3174, 3174, 3164, 3163, 3164, 3174, 3175, 3175, 3165, 3164,
        3165, 3175, 3176, 3176, 3166, 3165, 3166, 3176, 3177, 3177, 3167, 3166,
        3167, 3177, 3178, 3178, 3168, 3167, 3168, 3178, 3179, 3179, 3169, 3168,
        3169, 3179, 3180, 3180, 3170, 3169, 3170, 3180, 3181, 3181, 3171, 3170,
        2781, 2791, 3182, 3182, 3172, 2781, 3172, 3182, 3183, 3183, 3173, 3172,
        3173, 3183, 3184, 3184, 3174, 3173, 3174, 3184, 3185, 3185, 3175, 3174,
        3175, 3185, 3186, 3186, 3176, 3175, 3176, 3186, 3187, 3187, 3177, 3176,
        3177, 3187, 3188, 3188, 3178, 3177, 3178, 3188, 3189, 3189, 3179, 3178,
        3179, 3189, 3190, 3190, 3180, 3179, 3180, 3190, 3191, 3191, 3181, 3180,
        2791, 2801, 3192, 3192, 3182, 2791, 3182, 3192, 3193, 3193, 3183, 3182,
        3183, 3193, 3194, 3194, 3184, 3183, 3184, 3194, 3195, 3195, 3185, 3184,
        3185, 3195, 3196, 3196, 3186, 3185, 3186, 3196, 3197, 3197, 3187, 3186,
        3187, 3197, 3198, 3198, 3188, 3187, 3188, 3198, 3199, 3199, 3189, 3188,
        3189, 3199, 3200, 3200, 3190, 3189, 3190, 3200, 3201, 3201, 3191, 3190,
        2801, 2811, 3202, 3202, 3192, 2801, 3192, 3202, 3203, 3203, 3193, 3192,
        3193, 3203, 3204, 3204, 3194, 3193, 3194, 3204, 3205, 3205, 3195, 3194,
        3195, 3205, 3206, 3206, 3196, 3195, 3196, 3206, 3207, 3207, 3197, 3196,
        3197, 3207, 3208, 3208, 3198, 3197, 3198, 3208, 3209, 3209, 3199, 3198,
        3199, 3209, 3210, 3210, 3200, 3199, 3200, 3210, 3211, 3211, 3201, 3200,
        2811, 2821, 3212, 3212, 3202, 2811, 3202, 3212, 3213, 3213, 3203, 3202,
        3203, 3213, 3214, 3214, 3204, 3203, 3204, 3214, 3215, 3215, 3205, 3204,
        3205, 3215, 3216, 3216, 3206, 3205, 3206, 3216, 3217, 3217, 3207, 3206,
        3207, 3217, 3218, 3218, 3208, 3207, 3208, 3218, 3219, 3219, 3209, 3208,
        3209, 3219, 3220, 3220, 3210, 3209, 3210, 3220, 3221, 3221, 3211, 3210,
        2821, 2831, 3222, 3222, 3212, 2821, 3212, 3222, 3223, 3223, 3213, 3212,
        3213, 3223, 3224, 3224, 3214, 3213, 3214, 3224, 3225, 3225, 3215, 3214,
        3215, 3225, 3226, 3226, 3216, 3215, 3216, 3226, 3227, 3227, 3217, 3216,
        3217, 3227, 3228, 3228, 3218, 3217, 3218, 3228, 3229, 3229, 3219, 3218,
        3219, 3229, 3230, 3230, 3220, 3219, 3220, 3230, 3231, 3231, 3221, 3220,
        2831, 2841, 3232, 3232, 3222, 2831, 3222, 3232, 3233, 3233, 3223, 3222,
        3223, 3233, 3234, 3234, 3224, 3223, 3224, 3234, 3235, 3235, 3225, 3224,
        3225, 3235, 3236, 3236, 3226, 3225, 3226, 3236, 3237, 3237, 3227, 3226,
        3227, 3237, 3238, 3238, 3228, 3227, 3228, 3238, 3239, 3239, 3229, 3228,
        3229, 3239, 3240, 3240, 3230, 3229, 3230, 3240, 3241, 3241, 3231, 3230,
        2841, 2461, 2843, 2843, 3232, 2841, 3232, 2843, 2845, 2845, 3233, 3232,
        3233, 2845, 2847, 2847, 3234, 3233, 3234, 2847, 2849, 2849, 3235, 3234,
        3235, 2849, 2851, 2851, 3236, 3235, 3236, 2851, 2853, 2853, 3237, 3236,
        3237, 2853, 2855, 2855, 3238, 3237, 3238, 2855, 2857, 2857, 3239, 3238,
        3239, 2857, 2859, 2859, 3240, 3239, 3240, 2859, 2861, 2861, 3241, 3240,
    };
    static const vec3 *vp = reinterpret_cast<const vec3*>(teapot_positions);
    static const vec3 *vn = reinterpret_cast<const vec3*>(teapot_normals);
    static const vec2 *vt = reinterpret_cast<const vec2*>(teapot_texcoords);
    positions.assign(vp, vp + sizeof(teapot_positions) / sizeof(vec3));
    normals.assign(vn, vn + sizeof(teapot_normals) / sizeof(vec3));
    texcoords.assign(vt, vt + sizeof(teapot_texcoords) / sizeof(vec2));
    indices.assign(teapot_indices, teapot_indices + sizeof(teapot_indices) / sizeof(unsigned int));
}

std::vector<unsigned int> create_adjacency(const std::vector<unsigned int>& indices)
{
    assert(indices.size() % 3 == 0);
    std::vector<unsigned int> indices_with_adjacency(indices.size() / 3 * 6);
    for (size_t t = 0; t < indices.size() / 3; t++) {
        unsigned int v[3] = { indices[3 * t + 0], indices[3 * t + 1], indices[3 * t + 2] };
        unsigned int nv[3] = { v[2], v[0], v[1] }; // neighbor triangle vertices for edges 0, 1, 2
        for (size_t nt = 0; nt < indices.size() / 3; nt++) {
            if (nt == t)
                continue;
            unsigned int test_nv[3] = { indices[3 * nt + 0], indices[3 * nt + 1], indices[3 * nt + 2] };
            // Test edges. Neighbor triangles must have the same orientation as the
            // current triangle, so we only have to check one edge direction.
            // edge 0
            if (nv[0] == v[2]) {
                if (v[0] == test_nv[1] && v[1] == test_nv[0])
                    nv[0] = test_nv[2];
                else if (v[0] == test_nv[2] && v[1] == test_nv[1])
                    nv[0] = test_nv[0];
                else if (v[0] == test_nv[0] && v[1] == test_nv[2])
                    nv[0] = test_nv[1];
            }
            // edge 1
            if (nv[1] == v[0]) {
                if (v[1] == test_nv[1] && v[2] == test_nv[0])
                    nv[1] = test_nv[2];
                else if (v[1] == test_nv[2] && v[2] == test_nv[1])
                    nv[1] = test_nv[0];
                else if (v[1] == test_nv[0] && v[2] == test_nv[2])
                    nv[1] = test_nv[1];
            }
            // edge 2
            if (nv[2] == v[1]) {
                if (v[2] == test_nv[1] && v[0] == test_nv[0])
                    nv[2] = test_nv[2];
                else if (v[2] == test_nv[2] && v[0] == test_nv[1])
                    nv[2] = test_nv[0];
                else if (v[2] == test_nv[0] && v[0] == test_nv[2])
                    nv[2] = test_nv[1];
            }
            if (nv[0] != v[2] && nv[1] != v[0] && nv[2] != v[1])
                break;
        }
        indices_with_adjacency[6 * t + 0] = v[0];
        indices_with_adjacency[6 * t + 1] = nv[0];
        indices_with_adjacency[6 * t + 2] = v[1];
        indices_with_adjacency[6 * t + 3] = nv[1];
        indices_with_adjacency[6 * t + 4] = v[2];
        indices_with_adjacency[6 * t + 5] = nv[2];
    }
    return indices_with_adjacency;
}
