#include <GLWidget.hpp>

#include <glbase/gltool.hpp>

#include <glm/gtc/matrix_transform.hpp>

#include <vs.glsl.h>
#include <fs.glsl.h>

#include <iostream>

using namespace glm;

void GLWidget::initializeGL()
{
    // Initialize OpenGL extensions
    glewExperimental = GL_TRUE; // otherwise some function pointers are NULL...
    glewInit();
    glGetError(); // clear an error flag that glewInit() might leave behind
    VERIFY(CG::checkError());

    // Set up the programmable pipeline
    if (prg == 0) {
        GLuint vs = CG::createCompileShader(GL_VERTEX_SHADER, VS_GLSL_STR); VERIFY(vs);
        GLuint fs = CG::createCompileShader(GL_FRAGMENT_SHADER, FS_GLSL_STR); VERIFY(fs);
        prg = glCreateProgram();
        glAttachShader(prg, vs);
        glAttachShader(prg, fs);
        prg = CG::linkProgram(prg); VERIFY(prg);
        glUseProgram(prg);
    }
}

void GLWidget::initializeBuffers(vec3* geometry, int length)
{
    if (vao == 0) { // if vertex array object not initialized
        glGenVertexArrays(1, &vao);

        // --------------------------------------------------------------------
        // TODO (2.1):
        // * bind vao
        // * generate buffer
        // * bind buffer
        // * store buffer data
        // * define data type (vertex attribute)
        // * enable vertex array
        // --------------------------------------------------------------------
    }
}

void GLWidget::setupStar(float d, vec3** star, int &length)
{
    // -------------------------------------------------------------------
    // TODO (2.3):
    // * create the star!
    static vec3 s[] = {vec3(d, d, -d), vec3(-d, d, -d), vec3(-d, -d, -d)};
    // --------------------------------------------------------------------

    // don't touch
    *star = s;
    length = sizeof(s);
}

void GLWidget::resizeGL(int width, int height)
{
    glViewport(0, 0, width, height);
}

void GLWidget::paintGL()
{
    /**************************************************************/
    // Set up a vertex array object (voa) and assign the vertex arrays for the geometry
    vec3* star;
    int length;
    setupStar(0.2, &star, length);
    initializeBuffers(star, length);

    // bind voa to the shader program (selected by 'glUseProgram(prg)')
    glBindVertexArray(vao);


    /**************************************************************/
    // rotate vertices
    angle += 1.0/20.0;
    glm::mat4 rot;
    rot = glm::rotate (rot, angle*0.5f, glm::vec3(1,0,0));
    rot = glm::rotate (rot, angle, glm::vec3(0,0,1));
    GLint rot_loc = glGetUniformLocation (prg, "rot");
    VERIFY(rot_loc != -1);
    glUniformMatrix4fv (rot_loc, 1, GL_FALSE, &rot[0][0]);
    VERIFY(CG::checkError());


    /**************************************************************/
    // Render: clear framebuffer ressources
    glClearColor(1.0, 1.0, 1.0, 1.0);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glEnable(GL_DEPTH_TEST);

    // get shader location of color uniform
    GLint color_loc = glGetUniformLocation (prg, "color");

    // draw solid body
    glEnable (GL_CULL_FACE);
    glCullFace (GL_BACK);
    glEnable (GL_LINE_SMOOTH);
    glEnable (GL_DEPTH_TEST);
    glUniform3f (color_loc, 0.6,0.66,1.0);
    glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);
    glDrawArrays(GL_TRIANGLES, 0, length / sizeof(vec3));

    VERIFY(CG::checkError());

    // draw background outlines
    glDisable (GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);
    glCullFace (GL_FRONT);
    glUniform3f (color_loc, 0,0,0);
    glPolygonMode (GL_FRONT_AND_BACK, GL_LINE);
    glDrawArrays(GL_TRIANGLES, 0, length  / sizeof(vec3));

    VERIFY(CG::checkError());

    // draw foreground outlines
    glEnable (GL_DEPTH_TEST);
    glCullFace (GL_BACK);
    glUniform3f (color_loc, 0.72,0.94,0.6);
    glPolygonMode (GL_FRONT_AND_BACK, GL_LINE);
    glDrawArrays(GL_TRIANGLES, 0, length  / sizeof(vec3));

    VERIFY(CG::checkError());
}

void GLWidget::animateGL()
{
    updateGL();
}

